package csdk.v2.runner.core;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;

import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.UIManager;

/**
 * Dilogo de erro padro do Runner.
 *
 * @author Tecgraf
 */
public class RunnerErrorDialog extends JDialog {

  /**
   * A cor do fundo dos painis.
   */
  private static final Color BACKGROUND_COLOR = UIManager.getDefaults()
    .getColor("TextField.inactiveBackground");

  /** Nmero de colunas */
  private static final int COLUMNS = 80;

  /** Nmero de linhas do campo de mensagem */
  private static final int MESSAGE_ROWS = 4;

  /** Nmero de linhas do campo de pilha */
  private static final int STACK_ROWS = 14;

  /** Erro ou exceo a ser visualizada na janela */
  protected final Throwable error;

  /** Conjunto de informaes adicionais que sero mostradas na janela de erro. */
  protected final String[] additionalInfo;

  /**
   * Contrutor.
   *
   * @param owner a janela me.
   * @param title ttulo do dilogo.
   * @param error exceo a ser visualizada na janela.
   * @param additionalInfo informaes adicionais que sero mostradas na janela
   *        de erro.
   */
  public RunnerErrorDialog(Window owner, String title, Throwable error,
    String[] additionalInfo) {
    super(owner, title);
    this.error = error;
    this.additionalInfo = additionalInfo;
    this.setDefaultCloseOperation(DISPOSE_ON_CLOSE);
    createComponents();
  }

  /**
   * Mtodo interno de construo da inteface grfica.
   */
  private void createComponents() {
    final JLabel messageLabel = new JLabel("Message:");
    JScrollPane msgScrollPane = createMessagePane();

    final JLabel stackTraceLabel = new JLabel("Execution Stack:");
    JScrollPane stackTraceScroll = createStackPane();
    JLabel label = new JLabel("Execution Info:");
    JScrollPane scrollPane = createInfoPane();

    setLayout(new GridBagLayout());

    GridBagConstraints labelGBC = new GridBagConstraints();
    labelGBC.gridx = 0;
    labelGBC.gridy = GridBagConstraints.RELATIVE;
    labelGBC.anchor = GridBagConstraints.WEST;
    labelGBC.insets = new Insets(5, 6, 0, 10);

    GridBagConstraints fieldGBC = new GridBagConstraints();
    fieldGBC.gridx = 0;
    fieldGBC.gridy = GridBagConstraints.RELATIVE;
    fieldGBC.fill = GridBagConstraints.BOTH;
    fieldGBC.insets = new Insets(3, 10, 5, 10);

    add(label, labelGBC);
    add(scrollPane, fieldGBC);

    add(messageLabel, labelGBC);
    add(msgScrollPane, fieldGBC);

    add(stackTraceLabel, labelGBC);
    add(stackTraceScroll, fieldGBC);

    pack();
    setLocationRelativeTo(getOwner());
  }

  /**
   * Cria o painel com as informaes adicionais.
   *
   * @return o painel.
   */
  private JScrollPane createInfoPane() {
    int rows = 3;
    if (additionalInfo != null && additionalInfo.length != 0) {
      rows = additionalInfo.length + 1;
    }

    JTextArea additionalInfoTextArea = new JTextArea(rows, 80);
    additionalInfoTextArea.setEditable(false);
    additionalInfoTextArea.setBackground(BACKGROUND_COLOR);
    additionalInfoTextArea.setText(createAdditionalInfoText());
    return new JScrollPane(additionalInfoTextArea);
  }

  /**
   * Cria o painel com a pilha de execuo.
   *
   * @return o painel.
   */
  private JScrollPane createStackPane() {
    JTextArea stackTraceTextArea = new JTextArea(getStackTraceText());
    stackTraceTextArea.setColumns(COLUMNS);
    stackTraceTextArea.setRows(STACK_ROWS);
    stackTraceTextArea.setEditable(false);
    stackTraceTextArea.setBackground(BACKGROUND_COLOR);
    return new JScrollPane(stackTraceTextArea);
  }

  /**
   * Cria o painel com a mensagem de erro.
   *
   * @return o painel.
   */
  private JScrollPane createMessagePane() {
    // rea de texto de mensagens
    JTextArea messageTextArea = new JTextArea(this.error.getLocalizedMessage());
    messageTextArea.setColumns(COLUMNS);
    messageTextArea.setRows(MESSAGE_ROWS);
    messageTextArea.setEditable(false);
    messageTextArea.setBackground(BACKGROUND_COLOR);
    return new JScrollPane(messageTextArea);
  }

  /**
   * Monta a string que representa as informaes adicionais sobre o contexto do
   * erro.
   *
   * @return texto formatado com as informaes adicionais.
   */
  private String createAdditionalInfoText() {
    StringBuilder info = new StringBuilder();
    if (additionalInfo != null && additionalInfo.length > 0) {
      for (String anAdditionalInfo : additionalInfo) {
        info.append(anAdditionalInfo);
        info.append("\n");
      }
    }
    return info.toString();
  }

  /**
   * Monta a string que representa a pilha de execuo.
   *
   * @return a string com a pilha.
   */
  private String getStackTraceText() {
    Writer writer = new StringWriter();
    error.printStackTrace(new PrintWriter(writer));
    return writer.toString();
  }

}
