package csdk.v1_0.runner.core;

import java.awt.Dimension;
import java.awt.Window;
import java.io.PrintWriter;
import java.io.Serializable;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.Locale;
import java.util.Map;

import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;

import csdk.v1_0.api.application.ApplicationException;
import csdk.v1_0.api.application.IApplicationContext;
import csdk.v1_0.api.application.IMessage;
import csdk.v1_0.api.application.IMessageSender;
import csdk.v1_0.api.core.ICSDKEnvironment;
import csdk.v1_0.api.core.IContext;
import csdk.v1_0.helper.application.MessageSender;
import csdk.v1_0.runner.ApplicationManager;
import csdk.v1_0.runner.Runner;
import csdk.v1_0.runner.application.RunnerApplication;

/**
 * Interface das aplicaes com o ambiente {@link Runner}.
 */
public class RunnerEnvironment implements ICSDKEnvironment, Serializable {

  /**
   * Contextos da aplicao.
   */
  private final Map<Class<? extends IContext>, IContext> contexts;

  /**
   * Construtor.
   * 
   * @param contexts contextos disponveis no ambiente.
   */
  public RunnerEnvironment(Map<Class<? extends IContext>, IContext> contexts) {
    this.contexts = contexts;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void finishApplication() {
    IApplicationContext appContext = getApplicationContext();
    String instanceId = appContext.getInstanceId();
    final ApplicationManager appManager = ApplicationManager.getInstance();
    RunnerApplication app = appManager.getRunningApp(instanceId);
    if (app == null) {
      throw new IllegalStateException(
        "Application "
          + instanceId
          + " cannot be finished, it is not active yet or has already been deactivated");
    }
    if (app.finishApplication()) {
      appManager.deactivate(app);
    }
  }

  /**
   * Obtm o contexto de aplicao do ambiente.
   * 
   * @return o contexto de aplicao.
   */
  private IApplicationContext getApplicationContext() {
    IApplicationContext appContext =
      (IApplicationContext) this.contexts.get(IApplicationContext.class);
    return appContext;
  }

  /**
   * Limpa os contextos.
   */
  public void cleanupContexts() {
    for (IContext context : this.contexts.values()) {
      if (context.isActive()) {
        context.onContextDeactivate();
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String runApplication(String appId) throws ApplicationException {
    final ApplicationManager appManager = ApplicationManager.getInstance();
    RunnerApplication app = appManager.runApplication(appId);
    if (app == null) {
      return null;
    }
    return app.getInstanceId();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean sendMessage(String appId, IMessage message) {
    IApplicationContext appContext = getApplicationContext();
    String instanceId = appContext.getInstanceId();
    String applicationId = appContext.getApplicationId();
    final ApplicationManager appManager = ApplicationManager.getInstance();
    IMessageSender sender = new MessageSender(applicationId, instanceId);
    return appManager.sendMessage(appId, message, sender);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void handleException(Exception error, Window parent) {
    IApplicationContext appContext = getApplicationContext();
    Writer writer = new StringWriter();
    PrintWriter printWriter = new PrintWriter(writer);
    error.printStackTrace(printWriter);
    String errorString = writer.toString();
    String applicationName = appContext.getApplicationName();
    JTextArea errorArea = new JTextArea();
    errorArea.setText(errorString);
    errorArea.setEditable(false);
    errorArea.setCaretPosition(0);
    JScrollPane panel = new JScrollPane(errorArea);
    panel.setPreferredSize(new Dimension(450, 110));
    panel.setAutoscrolls(true);
    JOptionPane.showMessageDialog(parent, panel, applicationName,
      JOptionPane.ERROR_MESSAGE);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <T extends IContext> T getContext(Class<T> contextClass) {
    if (contextClass == null) {
      return null;
    }
    IContext context = this.contexts.get(contextClass);
    if (context != null) {
      try {
        T t = contextClass.cast(context);
        if (!context.isActive()) {
          t.onContextActivate();
        }
        return t;
      }
      catch (Exception e) {
        handleException(e, null);
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Locale getLocale() {
    final ApplicationManager appManager = ApplicationManager.getInstance();
    return appManager.getLocale();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Charset getCharset() {
    return Charset.defaultCharset();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Charset getServerCharset() {
    return null;
  }
}
