package examples;

import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.HashMap;
import java.util.Map;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import csdk.v2.api.application.ApplicationException;
import csdk.v2.api.application.IApplication;
import csdk.v2.api.application.IApplicationContext;
import csdk.v2.api.application.IMessage;
import csdk.v2.api.application.IMessageSender;
import csdk.v2.api.command.CommandException;
import csdk.v2.api.command.IAlgorithmTemplate;
import csdk.v2.api.command.ICommandContext;
import csdk.v2.api.command.ICommandInfo;
import csdk.v2.api.command.ICommandInfo.FinalizationType;
import csdk.v2.api.command.ICommandObserver;
import csdk.v2.api.core.ICSDKEnvironment;
import csdk.v2.helper.command.AlgorithmTemplate;
import csdk.v2.helper.command.parameters.ParameterValueConvertionException;

/**
 * Aplicao de teste de execuo de comandos via CSDK.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class CommandExecutionExample implements IApplication {

  /**
   * Interface padro para o ambiente CSDK.
   */
  private final ICSDKEnvironment csdkInterface;

  /**
   * Contexto da aplicao.
   */
  private final IApplicationContext appContext;

  /**
   * Contexto de comando.
   */
  private final ICommandContext commandContext;

  /**
   * Janela principal da aplicao.
   */
  private final JFrame mainFrame;

  /**
   * Construtor padro.
   * 
   * @param csdkInterface interface padro para o ambiente CSDK.
   */
  public CommandExecutionExample(final ICSDKEnvironment csdkInterface) {
    /* Guarda a referncia para a interface do ambiente CSDK */
    this.csdkInterface = csdkInterface;

    /* Busca os contextos necessrios para a aplicao */
    this.appContext = csdkInterface.getContext(IApplicationContext.class);
    this.commandContext = csdkInterface.getContext(ICommandContext.class);

    /*
     * Cria a janela principal e coloca um observador para finalizar a aplicao
     * quando a janela principal for fechada.
     */
    this.mainFrame = new JFrame();
    this.mainFrame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
    this.mainFrame.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        csdkInterface.finishApplication();
      }
    });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onApplicationStart() throws ApplicationException {
    /* Monta a janela principal */
    initUI();
    this.mainFrame.pack();
    this.mainFrame.setVisible(true);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canEndApplication() {
    /* Mostra um dilogo de confirmao antes de fechar a aplicao */
    String confirmMessage = appContext.getString("confirm.message");
    String applicationName = appContext.getApplicationName();
    int optionPaneType = JOptionPane.YES_NO_OPTION;
    int result =
      JOptionPane.showConfirmDialog(this.mainFrame, confirmMessage,
        applicationName, optionPaneType);
    return result == JOptionPane.YES_OPTION;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onApplicationEnd() throws ApplicationException {
    /* Libera os recursos alocados */
    mainFrame.dispose();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onAsyncMessageReceived(IMessage message, IMessageSender sender) {
    // No faz nada
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object onSyncMessageReceived(IMessage message, IMessageSender sender) {
    return null;
  }


  /**
   * Inicializa a interface da aplicao.
   */
  private void initUI() {
    /* Cria o boto de execuo na janela principal */
    String executeButtonName = appContext.getString("execute.algo.button");
    JButton executeButton = new JButton(executeButtonName);
    executeButton.addActionListener(ev -> {
      try {
        executeAlgorithm();
      }
      catch (Exception e) {
        csdkInterface.handleException(e, mainFrame);
      }
    });
    executeButton.setEnabled(commandContext != null);
    this.mainFrame.add(executeButton);
  }

  /**
   * Executa um algoritmo simples.
   * 
   * @throws CommandException em caso de erro ao submeter o comando.
   * @throws ParameterValueConvertionException em caso de erro na
   *         parametrizao.
   */
  protected void executeAlgorithm() throws CommandException,
    ParameterValueConvertionException {
    /* Gera o mapa de parmetros */
    Map<String, String> parameterValues = new HashMap<>();
    parameterValues.put("ARQUIVO_DE_ENTRADA", "arquivos/entrada.txt");
    parameterValues.put("ARQUIVO_DE_SAIDA", "arquivos/saida.txt");
    String algorithmName = "Otimizador";
    String algorithmVersion = "1.0.0";

    /* Cria o template do algoritmo */
    IAlgorithmTemplate info =
      new AlgorithmTemplate(algorithmName, algorithmVersion, parameterValues);
    String commandDescription = "execuo de teste do otimizador";
    // Deixa o sga ser escolhido automaticamente
    String sga = null;

    /* Executa o comando, cadastrando um observador */
    commandContext.executeAlgorithm(info, commandDescription, sga,
      this.mainFrame, new CommandObserver());
  }

  /**
   * Observador dos comandos executados.
   */
  private final class CommandObserver implements ICommandObserver {
    /**
     * {@inheritDoc}
     */
    @Override
    public void onCommandEnd(final ICommandInfo cmdInfo) {
      SwingUtilities.invokeLater(() -> {
        /* Mostra um dilogo com o estado de finalizao do comando */
        FinalizationType finalizationType = cmdInfo.getFinalizationType();
        String message =
          "Comando finalizado com status " + finalizationType.toString();
        JOptionPane.showMessageDialog(mainFrame, message);
      });
    }
  }

}
