package snakegame.algorithm.manhattan;

import java.awt.Point;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import snakegame.algorithm.IMoveAlgorithm;
import snakegame.algorithm.util.DeltaInt;
import snakegame.algorithm.util.Util;
import snakegame.map.Direction;

/**
 * Algoritmo Guloso 'Distncia em Manhattan'.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ManhattanDistanceAlgorithm implements IMoveAlgorithm {

  /** {@inheritDoc} */
  @Override
  public Direction whereShouldIGo(List<Point> snake, Direction direction,
    Point food, int rows, int columns) {
    Point head = snake.get(0);

    List<Direction> directions = Util.whereIsTheFood(head, food);

    List<DeltaInt> candidates = new ArrayList<>();
    for (Direction d : directions) {
      int distance = distance(head, food, direction);
      candidates.add(new DeltaInt(distance, d));
    }

    Collections.sort(candidates);
    Direction newDirection = candidates.get(0).getDirection();

    if (Util.isOpposite(direction, newDirection)) {
      return turn(snake, direction, rows, columns);
    }
    return newDirection;
  }

  /**
   * Calcula a distncia de 2 pontos considerando uma dada direo.
   * 
   * @param a origem.
   * @param b destino.
   * @param direction direo.
   * @return distncia da cabea para a comida na dada direo.
   */
  private int distance(Point a, Point b, Direction direction) {
    switch (direction) {
      case UP:
      case DOWN:
        return Math.abs(a.y - b.y);
      case LEFT:
      case RIGHT:
        return Math.abs(a.x - b.x);
    }
    return Integer.MAX_VALUE;
  }

  /**
   * Retorna a direo que far a cobra virar.
   * 
   * @param snake cobra.
   * @param direction direo atual.
   * @param rows nmero total de linhas.
   * @param columns nmero total de colunas.
   * @return direo.
   */
  private Direction turn(List<Point> snake, Direction direction, int rows,
    int columns) {
    Point head = snake.get(0);
    switch (direction) {
      case UP:
      case DOWN:
        Point left = new Point(head.x, head.y - 1);
        if (left.y >= 0 && !snake.contains(left)) {
          return Direction.LEFT;
        }
        Point right = new Point(head.x, head.y + 1);
        if (right.y < columns && !snake.contains(right)) {
          return Direction.RIGHT;
        }
      case LEFT:
      case RIGHT:
        Point up = new Point(head.x - 1, head.y);
        if (up.x >= 0 && !snake.contains(up)) {
          return Direction.UP;
        }
        Point down = new Point(head.x + 1, head.y);
        if (down.x < rows && !snake.contains(down)) {
          return Direction.DOWN;
        }
    }
    return direction;
  }
}
