package testefs;

import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.Collection;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

import csdk.v2.api.application.ApplicationException;
import csdk.v2.api.application.IApplication;
import csdk.v2.api.application.IApplicationContext;
import csdk.v2.api.application.IMessage;
import csdk.v2.api.application.IMessageSender;
import csdk.v2.api.application.manager.IApplicationManagerContext;
import csdk.v2.api.core.ICSDKEnvironment;
import csdk.v2.api.filesystem.FileSelectionType;
import csdk.v2.api.filesystem.IFile;
import csdk.v2.api.filesystem.IFileLock;
import csdk.v2.api.filesystem.IFileLock.LockStatus;
import csdk.v2.api.filesystem.local.ILocalFileSystemContext;
import csdk.v2.api.filesystem.local.LocalFileSystemException;
import csdk.v2.helper.application.Message;

/**
 * Aplicao de teste simples.
 *
 * @author Tecgraf/PUC-Rio
 */
public class TesteFS implements IApplication {

  /**
   * Interface padro para o ambiente CSDK.
   */
  private final ICSDKEnvironment csdkInterface;

  /**
   * Contexto de projeto da aplicao.
   */
  private final ILocalFileSystemContext localFileSystemContext;

  /**
   * Contexo de mensagens.
   */
  private final IApplicationManagerContext appManagerContext;

  /**
   * Contexto da aplicao.
   */
  private final IApplicationContext appContext;

  /**
   * Locks obtidos para os arquivos.
   */
  private final Collection<IFileLock> locks;

  /**
   * Janela principal da aplicao.
   */
  private final JFrame mainFrame;

  /**
   * Diretrio corrente.
   */
  private IFile currentDir;

  /**
   * Construtor padro.
   *
   * @param csdkInterface interface padro para o ambiente CSDK.
   */
  public TesteFS(final ICSDKEnvironment csdkInterface) {
    this.csdkInterface = csdkInterface;
    this.appContext = csdkInterface.getContext(IApplicationContext.class);
    this.localFileSystemContext =
      csdkInterface.getContext(ILocalFileSystemContext.class);
    this.appManagerContext =
      csdkInterface.getContext(IApplicationManagerContext.class);
    this.locks = new ArrayList<>();
    this.mainFrame = new JFrame();
    this.mainFrame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
    this.mainFrame.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent event) {
        csdkInterface.finishApplication();
      }
    });
    initUI();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canEndApplication() {
    int result = JOptionPane
      .showConfirmDialog(mainFrame, appContext.getString("confirm.message"),
        appContext.getApplicationName(), JOptionPane.YES_NO_OPTION);
    return result == JOptionPane.YES_OPTION;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onAsyncMessageReceived(IMessage message, IMessageSender sender) {
    // No faz nada
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object onSyncMessageReceived(IMessage message, IMessageSender sender) {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onApplicationEnd() throws ApplicationException {
    mainFrame.dispose();
    for (IFileLock lock : locks) {
      lock.releaseLock(mainFrame);
    }
  }

  /**
   * Inicializa a interface da aplicao.
   */
  private void initUI() {
    String openLabelName = appContext.getString("open.label");
    final JLabel openLabel = new JLabel(openLabelName);
    String openButtonName = appContext.getString("open.button");
    JButton openButton = new JButton(openButtonName);
    openButton.addActionListener(event -> {
      IFile file = openFile(FileSelectionType.FILES_ONLY);
      if (file != null) {
        openLabel.setText(file.getStringPath());
        try {
          IFileLock sharedLock = file.acquireSharedLock(mainFrame);
          locks.add(sharedLock);
          if (sharedLock.getLockStatus() == LockStatus.LOCK_SHARED || sharedLock
            .getLockStatus() == LockStatus.LOCK_EXCLUSIVE) {
            openFileInTextViewer(file);
          }
          else {
            String lockMessage = appContext.getString("lock.message");
            JOptionPane.showMessageDialog(mainFrame, lockMessage);
          }
        }
        catch (Exception e) {
          csdkInterface.handleException(e, mainFrame);
        }
      }
    });
    openButton.setEnabled(this.localFileSystemContext != null);
    String openMultipleLabelName = appContext.getString("open.multiple.label");
    final JLabel openMultipleLabel = new JLabel(openMultipleLabelName);
    String openMultipleButtonName =
      appContext.getString("open.multiple.button");
    JButton openMultipleButton = new JButton(openMultipleButtonName);
    openMultipleButton.addActionListener(event -> {
      IFile[] files = openMultipleFiles(FileSelectionType.FILES_ONLY);
      StringBuilder builder = new StringBuilder();
      if (files != null) {
        for (IFile file : files) {
          builder.append(file.getStringPath());
          builder.append(" ");
          try {
            IFileLock sharedLock = file.acquireSharedLock(mainFrame);
            locks.add(sharedLock);
            if (sharedLock
              .getLockStatus() == LockStatus.LOCK_SHARED || sharedLock
              .getLockStatus() == LockStatus.LOCK_EXCLUSIVE) {
              openFileInTextViewer(file);
            }
            else {
              String lockMessage = appContext.getString("lock.message");
              JOptionPane.showMessageDialog(mainFrame, lockMessage);
            }
          }
          catch (Exception e) {
            csdkInterface.handleException(e, mainFrame);
          }
        }
      }
      openMultipleLabel.setText(builder.toString());
    });
    openMultipleButton.setEnabled(this.localFileSystemContext != null);
    String openDirLabelName = appContext.getString("open.dir.label");
    final JLabel openDirLabel = new JLabel(openDirLabelName);
    String openDirButtonName = appContext.getString("open.dir.button");
    JButton openDirButton = new JButton(openDirButtonName);
    openDirButton.addActionListener(event -> {
      IFile file = openFile(FileSelectionType.DIRECTORIES_ONLY);
      if (file != null) {
        openDirLabel.setText(file.getStringPath());
      }
    });
    openDirButton.setEnabled(this.localFileSystemContext != null);
    String openMultipleDirLabelName =
      appContext.getString("open.multiple.dir.label");
    final JLabel openMultipleDirLabel = new JLabel(openMultipleDirLabelName);
    String openMultipleDirButtonName =
      appContext.getString("open.multiple.dir.button");
    JButton openMultipleDirButton = new JButton(openMultipleDirButtonName);
    openMultipleDirButton.addActionListener(event -> {
      IFile[] files = openMultipleFiles(FileSelectionType.DIRECTORIES_ONLY);
      StringBuilder builder = new StringBuilder();
      if (files != null) {
        for (IFile file : files) {
          builder.append(file.getStringPath());
          builder.append(" ");
        }
      }
      openMultipleDirLabel.setText(builder.toString());
    });
    openMultipleDirButton.setEnabled(this.localFileSystemContext != null);
    String saveLabelName = appContext.getString("save.label");
    final JLabel saveLabel = new JLabel(saveLabelName);
    String saveButtonName = appContext.getString("save.button");
    JButton saveButton = new JButton(saveButtonName);
    saveButton.addActionListener(event -> {
      IFile file = saveFile();
      if (file != null) {
        saveLabel.setText(file.getStringPath());
        try {
          IFileLock exclusiveLock = file.acquireExclusiveLock(mainFrame);
          locks.add(exclusiveLock);
          if (exclusiveLock.getLockStatus() == LockStatus.LOCK_EXCLUSIVE) {
            openFileInTextViewer(file);
          }
          else {
            String lockMessage = appContext.getString("lock.message");
            JOptionPane.showMessageDialog(mainFrame, lockMessage);
          }
        }
        catch (Exception e) {
          csdkInterface.handleException(e, mainFrame);
        }
      }
    });
    saveButton.setEnabled(this.localFileSystemContext != null);
    JPanel panel = new JPanel();
    panel.add(openLabel);
    panel.add(openButton);
    panel.add(openMultipleLabel);
    panel.add(openMultipleButton);
    panel.add(openDirLabel);
    panel.add(openDirButton);
    panel.add(openMultipleDirLabel);
    panel.add(openMultipleDirButton);
    panel.add(saveLabel);
    panel.add(saveButton);
    mainFrame.add(panel);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onApplicationStart() throws ApplicationException {
    mainFrame.pack();
    mainFrame.setVisible(true);
  }

  /**
   * Abre um visualizador de textos com o arquivo selecionado.
   *
   * @param file o arquivo.
   */
  private void openFileInTextViewer(IFile file) {
    try {
      String textViewerId = appManagerContext.runApplication("textviewer");
      Message message = new Message("file", file);
      appManagerContext.sendSyncMessage(textViewerId, message);
    }
    catch (ApplicationException e) {
      csdkInterface.handleException(e, mainFrame);
    }
  }

  /**
   * Apresenta um navegador de arquivos em modo de abertura.
   *
   * @param selectionType tipo de seleo do navegador (somente arquivos ou
   * somente diretrios).
   * @return o arquivo selecionado pelo usurio ou nulo, caso a operao tenha
   * sido cancelada pelo usurio.
   */
  private IFile openFile(FileSelectionType selectionType) {
    try {
      String[] fileTypes = appContext.getApplicationFileTypes();
      IFile selectedFile = localFileSystemContext
        .browseLocalFileInOpenMode(fileTypes, false, currentDir, selectionType,
          mainFrame);
      if (selectedFile != null) {
        currentDir = selectedFile.getParent();
      }
      return selectedFile;
    }
    catch (LocalFileSystemException e) {
      this.csdkInterface.handleException(e, mainFrame);
      return null;
    }
  }

  /**
   * Apresenta um navegador de mltiplos arquivos em modo de abertura.
   *
   * @param selectionType tipo de seleo do navegador (somente arquivos ou
   * somente diretrios).
   * @return o arquivo selecionado pelo usurio ou nulo, caso a operao tenha
   * sido cancelada pelo usurio.
   */
  private IFile[] openMultipleFiles(FileSelectionType selectionType) {
    try {
      String[] fileTypes = appContext.getApplicationFileTypes();
      IFile[] selectedFiles = localFileSystemContext
        .browseMultipleLocalFilesInOpenMode(fileTypes, false, currentDir,
          selectionType, mainFrame);
      if (selectedFiles != null && selectedFiles.length > 0) {
        currentDir = selectedFiles[0].getParent();
      }
      return selectedFiles;
    }
    catch (LocalFileSystemException e) {
      this.csdkInterface.handleException(e, mainFrame);
      return null;
    }
  }

  /**
   * Apresenta um navegador de arquivos em modo de gravao.
   *
   * @return o arquivo selecionado pelo usurio ou nulo, caso a operao tenha
   * sido cancelada pelo usurio.
   */
  private IFile saveFile() {
    try {
      String[] fileTypes = appContext.getApplicationFileTypes();
      IFile selectedFile = localFileSystemContext
        .browseLocalFileInSaveMode(fileTypes, false, currentDir, mainFrame);
      if (selectedFile != null) {
        currentDir = selectedFile.getParent();
      }
      return selectedFile;
    }
    catch (LocalFileSystemException e) {
      this.csdkInterface.handleException(e, mainFrame);
      return null;
    }
  }

}
