package user;

import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;

import csdk.v1_0.api.application.ApplicationException;
import csdk.v1_0.api.core.ICSDKEnvironment;
import csdk.v1_0.api.user.IUser;
import csdk.v1_0.api.user.IUserContext;
import csdk.v1_0.api.user.UserException;
import csdk.v1_0.helper.AbstractCSDKWindowApplication;

/**
 * Exemplo de uso do contexto de usurio.s
 * 
 * @author Tecgraf
 */
public class UserApplication extends AbstractCSDKWindowApplication {

  /**
   * O contexto.
   */
  private IUserContext userContext;

  /**
   * Construtor.
   * 
   * @param env o ambiente.
   */
  public UserApplication(ICSDKEnvironment env) {
    super(env);
    this.userContext = env.getContext(IUserContext.class);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void applicationStarted(JFrame frame) throws ApplicationException {
    if (userContext != null && userContext.isActive()) {
      String currentUserLabel = getString("current.user.button");
      JButton currentUser = new JButton(currentUserLabel);
      currentUser.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          showCurrentUser();
        }
      });
      String userByLoginLabel = getString("user.by.login.button");
      JButton userByLogin = new JButton(userByLoginLabel);
      userByLogin.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          showUserByLogin();
        }
      });
      String allUsersLabel = getString("all.users.button");
      JButton allUsers = new JButton(allUsersLabel);
      allUsers.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          showAllUsers();
        }
      });
      frame.setLayout(new FlowLayout());
      frame.add(currentUser);
      frame.add(userByLogin);
      frame.add(allUsers);
      frame.pack();
    }
    else {
      String message = getString("error.message");
      JOptionPane.showMessageDialog(getApplicationFrame(), message);
    }
  }

  /**
   * Mostra todos os usurios do sistema.
   */
  private void showAllUsers() {
    try {
      IUser[] users = userContext.getAllUsers(getApplicationFrame());
      StringBuffer buffer = new StringBuffer();
      for (IUser user : users) {
        String description = createUserDescription(user);
        buffer.append(description + "\n");
      }
      String message = buffer.toString();
      JOptionPane.showMessageDialog(getApplicationFrame(), message);
    }
    catch (UserException e) {
      handleException(e, getApplicationFrame());
    }
  }

  /**
   * Mostra o usurio especificado.
   */
  private void showUserByLogin() {
    try {
      String login =
        JOptionPane.showInputDialog(getApplicationFrame(), "login:");
      if (login != null && !login.isEmpty()) {
        IUser user = userContext.getUserByLogin(login, getApplicationFrame());
        String message = createUserDescription(user);
        JOptionPane.showMessageDialog(getApplicationFrame(), message);
      }
    }
    catch (UserException e) {
      handleException(e, getApplicationFrame());
    }
  }

  /**
   * Mostra o usurio corrente.
   */
  private void showCurrentUser() {
    try {
      IUser user = userContext.getCurrentUser();
      String message = createUserDescription(user);
      JOptionPane.showMessageDialog(getApplicationFrame(), message);
    }
    catch (UserException e) {
      handleException(e, getApplicationFrame());
    }
  }

  /**
   * Cria a descrio textual do usurio.
   * 
   * @param user o usurio.
   * @return a descrio.
   */
  private String createUserDescription(IUser user) {
    StringBuffer buffer = new StringBuffer();
    String name = user.getName();
    String login = user.getLogin();
    String[] emails = user.getEmails();
    boolean admin = user.isAdmin();
    buffer.append("login: ");
    buffer.append(login);
    if (admin) {
      buffer.append("*");
    }
    buffer.append("\n");
    buffer.append("name: ");
    buffer.append(name);
    buffer.append("\n");
    if (emails.length > 0) {
      buffer.append("emails: ");
      buffer.append("\n");
      for (String email : emails) {
        buffer.append(email);
        buffer.append("\n");
      }
    }
    return buffer.toString();

  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void applicationEnded() throws ApplicationException {
    // No faz nada.
  }

}
