/*
 * Detalhes da ltima alterao:
 * 
 * $Author: costa $ $Date: 2010-03-09 11:36:02 -0300 (Tue, 09 Mar 2010) $
 * $Revision: 102686 $
 */
package tecgraf.javautils.configurationmanager;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

/**
 * Representa a configurao necessria para o funcionamento de uma determinada
 * classe. Uma configurao  um conjunto de propriedades.
 * 
 * @author Tecgraf
 */
public class Configuration {

  /** Sufixo. */
  private static final String PROPERTY_NAME_FORMAT_SUFFIX = ".{0}";

  /**
   * Cache esttico para as propriedades. Trata-se de um mapa que armazena os
   * resultados de solicitaes de propriedades para uma determinada classe (o
   * segundo mapa, associado a cada classe). Os valores armazenados no segundo
   * mapa para cada propriedade podem ser <code>null</code> caso a propriedade
   * no tenha valor definido.
   * 
   * O objetivo deste cache  evitar repetidas consultas aos demais mapas na
   * hierarquia de classes.
   */
  private static final Map<Class<?>, Map<String, String>> cache =
    new HashMap<Class<?>, Map<String, String>>();

  /**
   * Um mapa de propriedades. Esse mapa tem como chave um nome de propriedade (
   * {@link String}) e valores ({@link String}).
   */
  private Map<String, String> propertiesMap;

  /** A classe dona desta configurao. */
  private Class<?> ownerClass;

  /**
   * A configurao pai desta configurao. Caso uma propriedade no seja
   * encontrada na configurao, esta ser procurada na configurao pai.
   */
  private Configuration parent;

  /**
   * Conjunto das configuraes filhas da configurao corrente. Usado para
   * invalidar os caches das mesmas quando o cache corrente for invalidado.
   */
  private Set<Configuration> children;

  /**
   * Construtor.
   * 
   * @param ownerClass classe dona da configurao.
   * @param parent configurao pai da configurao atual. Pode ser nula.
   * 
   * @throws IllegalArgumentException caso a classe recebida esteja nula.
   */
  Configuration(Class<?> ownerClass, Configuration parent) {
    if (ownerClass == null) {
      throw new IllegalArgumentException(
        "A classe dona da configurao no pode ser nula.");
    }
    this.ownerClass = ownerClass;
    this.parent = parent;
    if (parent != null) {
      parent.addChild(this);
    }
    this.propertiesMap = new HashMap<String, String>();
    synchronized (cache) {
      cache.put(ownerClass, new HashMap<String, String>());
    }
    log("Configurao criada");
  }

  /**
   * Obtm todas as chaves das propriedades.
   * 
   * @return Um conjunto com as chaves das propriedades. Se no houver chaves,
   *         retornar um conjunto vazio. Em ambos os casos, o conjunto
   *         retornado no pode ser modificado.
   */
  public Set<String> getPropertyKeys() {
    Set<String> propertyKeys = new HashSet<String>();
    if (this.parent != null) {
      propertyKeys.addAll(this.parent.getPropertyKeys());
    }
    propertyKeys.addAll(this.propertiesMap.keySet());
    return Collections.unmodifiableSet(propertyKeys);
  }

  /**
   * Obtm uma propriedade opcional.
   * 
   * @param name nome da propriedade.
   * 
   * @return valor da propriedade, ou null, caso a propriedade no exista.
   */
  public String getOptionalProperty(String name) {
    if (name == null) {
      throw new IllegalArgumentException(
        "O nome da propriedade no pode ser nulo.");
    }
    log("Tentando obter o valor da propriedade %s.", name);

    synchronized (cache) {
      Map<String, String> propsCache = cache.get(ownerClass);
      if (propsCache.containsKey(name)) {
        //        System.out.println(String.format("CACHE HIT [%s]: %s = %s", ownerClass
        //          .getName(), name, propsCache.get(name)));
        return propsCache.get(name);
      }
    }

    String value = this.propertiesMap.get(name);
    if (value == null && this.parent != null) {
      log("Verificando o valor da propriedade %s na configurao-pai.", name);
      value = this.parent.getOptionalProperty(name);
    }
    /*
     * atualizamos o cache com o valor obtido
     */
    synchronized (cache) {
      cache.get(ownerClass).put(name, value);
    }

    if (value == null) {
      log("O valor da propriedade %s no foi encontrado.", name);
    }
    else {
      log("Valor da propriedade %s encontrado. Valor = %s.", name, value);
    }
    return value;
  }

  /**
   * Obtm uma propriedade opcional.
   * 
   * @param name nome da propriedade.
   * @param defaultValue valor padro da propriedade.
   * 
   * @return valor da propriedade, ou o valor padro, caso a propriedade no
   *         exista.
   */
  public String getOptionalProperty(String name, String defaultValue) {
    if (name == null) {
      throw new IllegalArgumentException(
        "O nome da propriedade no pode ser nulo.");
    }
    if (defaultValue == null) {
      throw new IllegalArgumentException("O valor default no pode ser nulo.");
    }
    log("Tentando obter o valor da propriedade %s (valor-padro %s)", name,
      defaultValue);
    String value = this.getOptionalProperty(name);
    if (value == null) {
      log("O valor da propriedade %s no foi encontrado.\n"
        + "Utilizando valor-padro (%s)", name, defaultValue);
      return defaultValue;
    }
    return value;
  }

  /**
   * Obtm uma propriedade obrigatria.
   * 
   * @param name nome da propriedade.
   * 
   * @return valor da propriedade.
   * 
   * @throws MissingPropertyException caso a propriedade no esteja definida.
   */
  public String getMandatoryProperty(String name)
    throws MissingPropertyException {
    if (name == null) {
      throw new IllegalArgumentException(
        "O nome da propriedade no pode ser nulo.");
    }
    String value = this.getOptionalProperty(name);
    if (value == null) {
      throw new MissingPropertyException(this.ownerClass, name);
    }
    return value;
  }

  /**
   * Obtm uma propriedade opcional do tipo Boolean.
   * 
   * @param name nome da propriedade.
   * 
   * @return booleano definido pela propriedade, ou null, caso a propriedade no
   *         exista.
   */
  public Boolean getOptionalBooleanProperty(String name) {
    if (name == null) {
      throw new IllegalArgumentException(
        "O nome da propriedade no pode ser nulo.");
    }
    String value = this.getOptionalProperty(name);
    if (value == null) {
      return null;
    }
    return Boolean.parseBoolean(value);
  }

  /**
   * Obtm uma propriedade opcional do tipo Boolean.
   * 
   * @param name nome da propriedade.
   * @param defaultValue valor padro da propriedade.
   * 
   * @return booleano definido pela propriedade, ou o valor padro, caso a
   *         propriedade no exista.
   */
  public Boolean getOptionalBooleanProperty(String name, boolean defaultValue) {
    if (name == null) {
      throw new IllegalArgumentException(
        "O nome da propriedade no pode ser nulo.");
    }
    String value = this.getOptionalProperty(name);
    if (value == null) {
      return defaultValue;
    }
    return Boolean.valueOf(value);
  }

  /**
   * Obtm uma propriedade opcional do tipo Boolean.
   * 
   * @param name nome da propriedade.
   * 
   * @return booleano definido pela propriedade.
   * 
   * @throws MissingPropertyException caso a propriedade no exista.
   */
  public Boolean getMandatoryBooleanProperty(String name)
    throws MissingPropertyException {
    if (name == null) {
      throw new IllegalArgumentException(
        "O nome da propriedade no pode ser nulo.");
    }
    String value = this.getOptionalProperty(name);
    if (value == null) {
      throw new MissingPropertyException(this.ownerClass, name);
    }
    return Boolean.valueOf(value);
  }

  /**
   * Obtm uma propriedade opcional do tipo Long.
   * 
   * @param name nome da propriedade.
   * @param defaultValue valor padro da propriedade.
   * 
   * @return Long definido pela propriedade, ou o valor padro, caso a
   *         propriedade no exista.
   */
  public Long getOptionalLongProperty(String name, long defaultValue) {
    if (name == null) {
      throw new IllegalArgumentException(
        "O nome da propriedade no pode ser nulo.");
    }
    String value = this.getOptionalProperty(name);
    if (value == null) {
      return defaultValue;
    }
    return Long.valueOf(value);
  }

  /**
   * Obtm uma propriedade opcional do tipo Integer.
   * 
   * @param name nome da propriedade.
   * @param defaultValue valor padro da propriedade.
   * 
   * @return inteiro definido pela propriedade, ou o valor padro, caso a
   *         propriedade no exista.
   */
  public Integer getOptionalIntegerProperty(String name, int defaultValue) {
    if (name == null) {
      throw new IllegalArgumentException(
        "O nome da propriedade no pode ser nulo.");
    }
    String value = this.getOptionalProperty(name);
    if (value == null) {
      return defaultValue;
    }
    return Integer.valueOf(value);
  }

  /**
   * Obtm uma propriedade opcional que  uma classe.
   * 
   * @param <T> classe que ser obtida.
   * @param name nome da propriedade.
   * 
   * @return classe definida pela propriedade, ou null, caso a propriedade no
   *         exista.
   * 
   * @throws ClassNotFoundException caso o classe definida no exista.
   */
  @SuppressWarnings("unchecked")
  public <T> Class<T> getOptionalClassProperty(String name)
    throws ClassNotFoundException {
    if (name == null) {
      throw new IllegalArgumentException(
        "O nome da propriedade no pode ser nulo.");
    }
    String value = this.getOptionalProperty(name);
    if (value == null) {
      return null;
    }
    return (Class<T>) Class.forName(value);
  }

  /**
   * Obtm uma propriedade opcional que  uma classe.
   * 
   * @param <T> classe que ser obtida.
   * @param name nome da propriedade.
   * @param defaultClass classe padro.
   * 
   * @return classe definida pela propriedade, ou a classe padro, caso a
   *         propriedade no exista.
   * 
   * @throws ClassNotFoundException caso o classe definida no exista.
   */
  public <T> Class<T> getOptionalClassProperty(String name,
    Class<T> defaultClass) throws ClassNotFoundException {
    if (name == null) {
      throw new IllegalArgumentException(
        "O nome da propriedade no pode ser nulo.");
    }
    if (defaultClass == null) {
      throw new IllegalArgumentException("A classe default no pode ser nula.");
    }
    Class<T> clazz = this.getOptionalClassProperty(name);
    if (clazz == null) {
      return defaultClass;
    }
    return clazz;
  }

  /**
   * Obtm uma propriedade obrigatria que  uma classe.
   * 
   * @param <T> classe que ser obtida.
   * @param name nome da propriedade.
   * 
   * @return classe definida pela propriedade.
   * 
   * @throws MissingPropertyException caso a propriedade no exista.
   * @throws ClassNotFoundException caso a classe definida no exista.
   */
  public <T> Class<T> getMandatoryClassProperty(String name)
    throws MissingPropertyException, ClassNotFoundException {
    if (name == null) {
      throw new IllegalArgumentException(
        "O nome da propriedade no pode ser nulo.");
    }
    Class<T> clazz = this.getOptionalClassProperty(name);
    if (clazz == null) {
      throw new MissingPropertyException(this.ownerClass, name);
    }
    return clazz;
  }

  /**
   * Obtm uma lista de valores de propriedades que seguem o padro para
   * propriedades multi-valoradas usado pelo JDK da Sun.
   * <ul>
   * <li>propriedade.1=ABCD</li>
   * <li>propriedade.2=EFGH</li>
   * </ul>
   * <ul>
   * <li>propriedade.nome.1=ABCD</li>
   * <li>propriedade.nome.2=EFGH</li>
   * </ul>
   * 
   * @param name nome da propriedade (nos exemplos acima, seriam "propriedade" e
   *        "propriedade.nome" respectivamente).
   * 
   * @return lista com todas as propriedades (caso no existam propriedades, a
   *         lista estar vazia).
   */
  public List<String> getOptionalListProperty(String name) {
    if (name == null) {
      throw new IllegalArgumentException(
        "O padro do nome da propriedade no pode ser nulo.");
    }
    String pattern = name + PROPERTY_NAME_FORMAT_SUFFIX;
    List<String> list = new LinkedList<String>();
    for (int i = 1; true; i++) {
      String value = this.getOptionalProperty(MessageFormat.format(pattern, i));
      if (value == null) {
        break;
      }
      list.add(value);
    }
    return Collections.unmodifiableList(list);
  }

  /**
   * <p>
   * Obtm uma lista de valores de propriedades que seguem o padro para
   * propriedades multi-valoradas usado pelo JDK da Sun.
   * </p>
   * <ul>
   * <li>propriedade.1=ABCD</li>
   * <li>propriedade.2=EFGH</li>
   * </ul>
   * <ul>
   * <li>propriedade.nome.1=ABCD</li>
   * <li>propriedade.nome.2=EFGH</li>
   * </ul>
   * 
   * @param name nome da propriedade (nos exemplos acima, seriam "propriedade" e
   *        "propriedade.nome" respectivamente).
   * 
   * @return lista com todas as propriedades.
   * 
   * @throws MissingPropertyException caso no seja encontrada nenhuma
   *         propriedade no padro especificado.
   */
  public List<String> getMandatoryListProperty(String name)
    throws MissingPropertyException {
    if (name == null) {
      throw new IllegalArgumentException(
        "O padro do nome da propriedade no pode ser nulo.");
    }
    String pattern = name + PROPERTY_NAME_FORMAT_SUFFIX;
    List<String> list = new LinkedList<String>();
    for (int i = 1; true; i++) {
      String value = this.getOptionalProperty(MessageFormat.format(pattern, i));
      if (value == null) {
        break;
      }
      list.add(value);
    }
    if (list.isEmpty()) {
      throw new MissingPropertyException(this.ownerClass, name);
    }
    return Collections.unmodifiableList(list);
  }

  /**
   * Obtm uma propriedade opcional que  uma locale.
   * 
   * @param name nome da propriedade.
   * 
   * @return locale definida pela propriedade, ou null, caso a propriedade no
   *         exista ou no represente uma locale.
   */
  public Locale getOptionalLocaleProperty(String name) {
    if (name == null) {
      throw new IllegalArgumentException(
        "O nome da propriedade no pode ser nulo.");
    }
    String value = this.getOptionalProperty(name);
    if (value == null) {
      return null;
    }
    return parseLocale(value);
  }

  /**
   * Obtm uma propriedade opcional que  uma locale.
   * 
   * @param name nome da propriedade.
   * @param defaultLocale locale padro.
   * 
   * @return locale definida pela propriedade, ou a locale padro, caso a
   *         propriedade no exista.
   */
  public Locale getOptionalLocaleProperty(String name, Locale defaultLocale) {
    if (name == null) {
      throw new IllegalArgumentException(
        "O nome da propriedade no pode ser nulo.");
    }
    if (defaultLocale == null) {
      throw new IllegalArgumentException("A locale default no pode ser nula.");
    }
    Locale locale = this.getOptionalLocaleProperty(name);
    if (locale == null) {
      return defaultLocale;
    }
    return locale;
  }

  /**
   * Obtm uma propriedade obrigatria que  uma locale.
   * 
   * @param name nome da propriedade.
   * 
   * @return locale definida pela propriedade.
   * 
   * @throws MissingPropertyException caso a propriedade no esteja definida.
   */
  public Locale getMandatoryLocaleProperty(String name)
    throws MissingPropertyException {
    if (name == null) {
      throw new IllegalArgumentException(
        "O nome da propriedade no pode ser nulo.");
    }
    Locale locale = this.getOptionalLocaleProperty(name);
    if (locale == null) {
      throw new MissingPropertyException(this.ownerClass, name);
    }
    return locale;
  }

  /**
   * Obtm uma lista de propriedades que so locales.
   * 
   * @param name nome da propriedade.
   * 
   * @return lista com todos os locales (caso no existam locales, a lista
   *         estar vazia).
   */
  public List<Locale> getOptionalLocaleListProperty(String name) {
    if (name == null) {
      throw new IllegalArgumentException(
        "O padro do nome da propriedade no pode ser nulo.");
    }
    List<String> propertyList = this.getOptionalListProperty(name);
    List<Locale> localeList = new ArrayList<Locale>(propertyList.size());
    for (String property : propertyList) {
      Locale locale = parseLocale(property);
      if (locale != null) {
        localeList.add(locale);
      }
    }
    return Collections.unmodifiableList(localeList);
  }

  /**
   * Obtm uma lista de propriedades que so locales.
   * 
   * @param name nome da propriedade.
   * 
   * @return lista com todos os locales.
   * 
   * @throws MissingPropertyException caso no seja encontrado nenhum locale.
   */
  public List<Locale> getMandatoryLocaleListProperty(String name)
    throws MissingPropertyException {
    if (name == null) {
      throw new IllegalArgumentException(
        "O padro do nome da propriedade no pode ser nulo.");
    }
    List<String> propertyList = this.getMandatoryListProperty(name);
    List<Locale> localeList = new ArrayList<Locale>(propertyList.size());
    for (String property : propertyList) {
      Locale locale = parseLocale(property);
      if (locale != null) {
        localeList.add(locale);
      }
    }
    if (localeList.isEmpty()) {
      throw new MissingPropertyException(this.ownerClass, name);
    }
    return Collections.unmodifiableList(localeList);
  }

  /**
   * Obtm um valor de uma enumerao.
   * 
   * @param <E> tipo da enumerao.
   * 
   * @param name nome da propriedade.
   * @param enumerationClass classe da enumerao.
   * @param defaultValue valor-padro.
   * 
   * @return valor da enumerao.
   * @throws IllegalFormatException o formato da propriedade no for o nome de
   *         algum valor da enumerao.
   */
  public <E extends Enum<E>> E getOptionalEnumerationProperty(String name,
    Class<E> enumerationClass, E defaultValue) throws IllegalFormatException {
    String valueName = getOptionalProperty(name);
    if (valueName == null) {
      return defaultValue;
    }
    for (E value : enumerationClass.getEnumConstants()) {
      if (valueName.equals(value.name())) {
        return value;
      }
    }
    String details = String.format("Valores vlidos:\n");
    for (E value : enumerationClass.getEnumConstants()) {
      details += value.name() + "\n";
    }
    throw new IllegalFormatException(ownerClass, name, valueName, details);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return String.format("configurao : classe %s (pai : %s)",
      ownerClass.getName(), parent);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null || !getClass().isAssignableFrom(obj.getClass())) {
      return false;
    }
    Configuration other = (Configuration) obj;
    return ownerClass.equals(other.ownerClass);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return ownerClass.hashCode();
  }

  /**
   * Faz um merge das propriedades contidas atualmente na configurao com as
   * propriedades recebidas.
   * 
   * @param properties propriedades a serem "mescladas" com as propriedades
   *        atuais.
   * 
   * @throws IllegalArgumentException caso o conjunto de propriedades recebido
   *         seja nulo.
   */
  void merge(Properties properties) {
    if (properties == null) {
      throw new IllegalArgumentException(
        "A coleo de propriedades no pode ser nula.");
    }
    log("Realizando merge");
    log("Propriedades antes do merge: %s", propertiesMap);
    log("Novas propriedades: %s", properties);
    for (Map.Entry<Object, Object> entry : properties.entrySet()) {
      propertiesMap.put((String) entry.getKey(), (String) entry.getValue());
    }
    log("Propriedades depois do merge: %s", propertiesMap);
    clearCacheDownwards();
  }

  /**
   * Registra no {@link Logger} uma mensagem.
   * 
   * @param message mensagem.
   * @param args argumentos da mensagem.
   */
  private void log(String message, Object... args) {
    Logger.getInstance().log("%s : %s", this, String.format(message, args));
  }

  /**
   * Registra no {@link Logger} uma mensagem.
   * 
   * @param message mensagem.
   */
  private void log(String message) {
    Logger.getInstance().log("%s : %s", this, message);
  }

  /**
   * Adiciona uma configurao ao conjunto de filhos da configurao corrente.
   * Se a configurao j consta do conjunto de filhos, no faz nada.
   * 
   * @param child configurao-filho.
   */
  private void addChild(Configuration child) {
    if (children == null) {
      children = new HashSet<Configuration>();
    }
    children.add(child);
  }

  /** Limpa o cache da configurao corrente e o de todos os seus descendentes. */
  private void clearCacheDownwards() {
    synchronized (cache) {
      cache.get(ownerClass).clear();
      if (children != null) {
        Iterator<Configuration> iterator = children.iterator();
        while (iterator.hasNext()) {
          iterator.next().clearCacheDownwards();
        }
      }
    }
  }

  /**
   * Constri um objeto {@link Locale} dado uma {@link String}.
   * 
   * @param localeStr locale.
   * @return locale.
   */
  private Locale parseLocale(String localeStr) {
    if (localeStr == null || localeStr.length() == 0) {
      return null;
    }
    String[] tokens = localeStr.split("_", 3);
    switch (tokens.length) {
      case 0:
        return null;
      case 1: // Ex: es
        return new Locale(tokens[0]);
      case 2: // Ex: pt_BR
        return new Locale(tokens[0], tokens[1]);
      case 3: // Ex: en_US_Tradicional_WIN
        return new Locale(tokens[0], tokens[1], tokens[2]);
    }
    return null;
  }
}
