package tecgraf.javautils.gui.table;

import java.awt.Color;
import java.awt.Component;
import java.awt.Rectangle;

import javax.swing.JTable;
import javax.swing.JTextArea;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellRenderer;

/**
 * Renderizador padro para desenho de clulas com quebra de linha dentro de uma
 * tabela.
 * 
 * @author Leonardo Barros
 */
public class DefaultLineWrapTableCellRenderer extends JTextArea implements
  TableCellRenderer {
  /** Borda vazia para clulas sem foco */
  protected static Border noFocusBorder = new EmptyBorder(1, 1, 1, 1);

  /**
   * Cria o renderizador para clulas com quebra de linha.
   */
  public DefaultLineWrapTableCellRenderer() {
    setLineWrap(true);
  }

  /**
   * {@inheritDoc} <br>
   * Esse componente permite que as clulas da tabela contenham mltiplas
   * linhas.
   */
  public Component getTableCellRendererComponent(JTable table, Object value,
    boolean isSelected, boolean hasFocus, int rowIndex, int viewColumnIndex) {
    // Configura cores da tabela de primeiro plano e fundo, de acordo com o 
    // estado (selecionada/no selecionada) da clula.
    if (table != null) {
      if (isSelected) {
        setForeground(table.getSelectionForeground());
        setBackground(table.getSelectionBackground());
      }
      else {
        setForeground(table.getForeground());
        setBackground(table.getBackground());
      }
    }
    // Configura borda e cores de primeiro plano e fundo, dependendo se a clula
    // sendo desenhada esteja com o foco ou no, e se ela for editvel ou no.
    if (hasFocus) {
      setBorder(UIManager.getBorder("Table.focusCellHighlightBorder"));
    }
    else {
      setBorder(noFocusBorder);
    }
    // Copia a fonte da tabela.
    if (table != null) {
      setFont(table.getFont());
    }
    // Copia o texto e ajusta a altura das linhas do componente.
    String text = (value == null) ? null : value.toString();
    adjustHeight(text);
    setText(text);
    return this;
  }

  /**
   * Este mtodo atribui a menor altura possvel para o componente que garanta a
   * exibio de todas as linhas contidas na clula.
   * 
   * @param text texto a ser exibido.
   */
  protected void adjustHeight(String text) {
    if (text == null) {
      setRows(1);
      return;
    }
    String[] lines = text.split("\n");
    setRows(lines.length);
  }

  /**
   * {@inheritDoc} Sobrecarregado por razes de performance.
   * 
   * @see DefaultTableCellRenderer
   */
  @Override
  public boolean isOpaque() {
    Color back = getBackground();
    Component table = getParent();
    if (table != null) {
      table = table.getParent();
    }
    boolean colorMatch =
      (back != null) && (table != null) && back.equals(table.getBackground())
        && table.isOpaque();
    return !colorMatch && super.isOpaque();
  }

  /**
   * {@inheritDoc} Sobrecarregado por razes de performance.
   * 
   * @see DefaultTableCellRenderer
   */
  @Override
  public void validate() {
  }

  /**
   * {@inheritDoc} Sobrecarregado por razes de performance.
   * 
   * @see DefaultTableCellRenderer
   */
  @Override
  public void revalidate() {
  }

  /**
   * {@inheritDoc} Sobrecarregado por razes de performance.
   * 
   * @see DefaultTableCellRenderer
   */
  @Override
  public void repaint(long tm, int x, int y, int width, int height) {
  }

  /**
   * {@inheritDoc} Sobrecarregado por razes de performance.
   * 
   * @see DefaultTableCellRenderer
   */
  @Override
  public void repaint(Rectangle r) {
  }

  /**
   * {@inheritDoc} Sobrecarregado por razes de performance.
   * 
   * @see DefaultTableCellRenderer
   */
  @Override
  protected void firePropertyChange(String propertyName, Object oldValue,
    Object newValue) {
    if (propertyName.equals("document")) {
      super.firePropertyChange(propertyName, oldValue, newValue);
    }
  }

  /**
   * {@inheritDoc} Sobrecarregado por razes de performance.
   * 
   * @see DefaultTableCellRenderer
   */
  @Override
  public void firePropertyChange(String propertyName, boolean oldValue,
    boolean newValue) {
  }

  /**
   * {@inheritDoc} Sobrecarregado por razes de performance.
   * 
   * @see DefaultTableCellRenderer
   */
  @Override
  public void repaint() {
  }
}
