/*
 * $Id: Colored.java 101174 2010-01-28 18:58:09Z giovani $
 */
package tecgraf.javautils.gui.table;

import java.awt.Color;

import javax.swing.ImageIcon;

/**
 * Componente a ser apresentado nas clulas da tabela de alocao e de lotes.
 * Permite que tenhamos clulas coloridas.
 * 
 * @param <T> classe do componente apresentado.
 */
public class Colored<T extends Comparable<T>> implements Comparable<Colored<T>> {
  /** Cor padro */
  private final static Color STANDARD_COLOR = Color.BLACK;

  /** Alinhamento do cone */
  public static enum IconAlignment {
    /** Alinhamento  esquerda */
    LEFT,
    /** Alinhamento  direita */
    RIGHT
  };

  /** Valor do componente */
  private T value;
  /** Cor do componente */
  private Color color;
  /** cone a ser apresentado */
  private ImageIcon icon;
  /**
   * Determina se o cone fica  esquerda ou  direita do texto. Por default o
   * cone fica  esquerda.
   */
  private IconAlignment iconAlign = IconAlignment.LEFT;
  /** Texto a ser apresentado no tooltip */
  private String tooltipText;
  /** Indica se o texto deve ser apresentado */
  private boolean showValue;

  /**
   * Cria um componente de uma cor, ainda sem sabermos o seu valor.
   * 
   * @param color cor do componente.
   */
  public Colored(Color color) {
    this(null, color);
  }

  /**
   * Cria um componente utilizando a cor padro.
   * 
   * @param value valor do componente.
   */
  public Colored(T value) {
    this(value, STANDARD_COLOR);
  }

  /**
   * Cria um componente que no  nmero, com um valor e uma cor.
   * 
   * @param value valor do componente.
   * @param color cor do componente.
   */
  public Colored(T value, Color color) {
    this(value, color, null);
  }

  /**
   * Cria um componente que no  nmero, com um valor, uma cor e um cone.
   * 
   * @param value valor do componente.
   * @param color cor do componente.
   * @param icon cone do componente.
   */
  public Colored(T value, Color color, ImageIcon icon) {
    this(value, color, icon, true);
  }

  /**
   * Cria um componente que no  nmero, com um valor, uma cor e um cone.
   * 
   * @param value valor do componente.
   * @param color cor do componente.
   * @param icon cone do componente.
   * @param showText indica se o texto deve ser apresentado.
   */
  public Colored(T value, Color color, ImageIcon icon, boolean showText) {
    set(value, color, icon);
    this.showValue = showText;
  }

  /**
   * Modifica o valor, a cor e a indicao se o componente  nmero.
   * 
   * @param value valor do componente.
   * @param color cor do componente.
   */
  public void set(T value, Color color) {
    set(value);
    this.color = color;
  }

  /**
   * Modifica o valor e a indicao se o componente  nmero.
   * 
   * @param value valor do componente.
   */
  public void set(T value) {
    this.value = value;
  }

  /**
   * Modifica o valor, a cor, a indicao se o componente  nmero e o seu
   * cone.
   * 
   * @param value valor do componente.
   * @param color cor do componente.
   * @param icon cone do componente.
   */
  public void set(T value, Color color, ImageIcon icon) {
    set(value, color);
    this.icon = icon;
  }

  /**
   * Altera o alinhamento do cone.
   * 
   * @param side lado do alinhamento.
   */
  public void setIconAlignment(IconAlignment side) {
    iconAlign = side;
  }

  /**
   * Informa o valor do componente.
   * 
   * @return valor do componente.
   */
  public T getValue() {
    return value;
  }

  /**
   * Compara o valor deste objeto Colored ao valor de um objeto "o".
   * 
   * @param o objeto Colored sendo comparado
   * 
   * @return o resultado da comparao dos valores dos objetos
   */
  public int compareTo(Colored<T> o) {
    if (o != null && this.value != null && o.value != null) {
      return this.value.compareTo(o.value);
    }
    if (o == null || o.value == null) {
      return 1;
    }
    return -1;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object o) {
    if ((o == null) || !Colored.class.isInstance(o)) {
      return false;
    }
    Colored<?> other = Colored.class.cast(o);
    if (value == null && other.value == null) {
      return true;
    }
    if ((value == null && other.value != null) || !value.equals(other.value)) {
      return false;
    }
    if ((color == null && other.color != null) || !color.equals(other.color)) {
      return false;
    }
    return true;
  }

  /**
   * Informa a cor do componente.
   * 
   * @return cor do componente.
   */
  public Color getColor() {
    return color;
  }

  /**
   * Informa o cone do componente.
   * 
   * @return cone do componente.
   */
  public ImageIcon getIcon() {
    return icon;
  }

  /**
   * Informa de que lado do texto fica o cone.
   * 
   * @return qual lado do texto fica o cone.
   */
  public IconAlignment getIconAlignment() {
    return iconAlign;
  }

  /**
   * Obtm o texto a ser apresentado no tooltip.
   * 
   * @return texto a ser apresentado no tooltip.
   */
  public String getToolTipText() {
    return tooltipText;
  }

  /**
   * Altera o texto a ser apresentado no tooltip.
   * 
   * @param tooltipText texto a ser apresentado no tooltip.
   */
  public void setTooltipText(String tooltipText) {
    this.tooltipText = tooltipText;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    if (value != null) {
      return value.toString();
    }
    return null;
  }

  /**
   * Indica se o valor deve ser apresentado.
   * 
   * @return verdadeiro se o valor tiver que ser apresentado.
   */
  public boolean showValue() {
    return showValue;
  }
}
