/*
 * $Id: DefaultPrintableTable.java 150399 2014-02-26 19:08:39Z oikawa $
 */
package tecgraf.javautils.gui.print;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Paint;
import java.awt.Shape;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.ImageObserver;
import java.awt.print.PageFormat;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.util.BitSet;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.ImageIcon;
import javax.swing.table.TableModel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.table.Colored;
import tecgraf.javautils.gui.table.FooterCell;

/**
 * Implementao padro para uma tabela a ser impressa.
 * 
 * @see DefaultPrintableTableSample
 */
public class DefaultPrintableTable implements PrintableTable, ImageObserver {
  /** String para boolean true */
  final private static String trueString = LNG.get("javautils.yes");
  /** String para boolean false */
  final private static String falseString = LNG.get("javautils.no");

  /** Modelo da tabela a ser impressa */
  private DefaultPrintableTableModel model;
  /** Fonte padro */
  private Font defaultFont;
  /** Texto e fonte para cada clula da tabela */
  private CellInfo[][] cellInfo;
  /**
   * Altura de cada linha: considera tanto a altura do texto quanto a espessura
   * das linhas divisrias (diferentemente da {@link #columnSize}. O primeiro
   * elemento do array, que representa a linha superior, considera em sua altura
   * a altura do texto, da linha divisria superior e da linha divisria
   * inferior. As demais linhas s consideram a altura do texto mais a linha
   * divisria inferior.
   */
  private float[] rowHeight;
  /**
   * Peso proporcional das colunas, isto , qual o percentual da largura
   * disponvel deve ser atribudo a cada uma (entre 0 e 1). Por exemplo, se
   * tivermos duas colunas, uma com peso 0.2 e outra com peso 0.8, a primeira
   * receber 20% do espao disponvel, e a segunda 80%.
   */
  private float[] columnWeight;
  /**
   * Os agrupamentos das colunas da tabela. Ser <code>null</code> caso no haja
   * agrupamentos para as colunas.
   */
  private List<TableColumnGroup> columnGroups;
  /**
   * Quantidade de colunas da tabela, que varia de acordo com a existncia do
   * header de linhas.
   */
  private int columnCount;
  /**
   * Largura de cada coluna: considera somente o espao reservado para o texto,
   * sem contar com as linhas divisrias (diferentemente da {@link #rowHeight}).
   */
  private float[] columnSize;
  /** Espessura das linhas da tabela */
  private float lineWidth;
  /** Indica se as linhas da tabela devem ser desenhadas */
  private boolean drawLines;
  /** espaamento horizontal entre os textos e as linhas */
  private float horizontalInset;
  /** espaamento vertical entre os textos e as linhas */
  private float verticalInset;
  /** Comentrio que aparece colado na tabela, ao seu final */
  private String posText;
  /** Fonte do comentrio que aparece colado na tabela, ao seu final */
  private Font posTextFont;
  /** Fonte modificada do comentrio que aparece colado na tabela, no final */
  private Font modifiedPosTextFont;
  /** Comentrio que aparece colado na tabela, acima dela em todas as pginas */
  private String header;
  /** Fonte do comentrio que aparece colado na tabela, acima dela */
  private Font headerFont;
  /** Fonte modificada do comentrio que aparece colado na tabela, acima dela */
  private Font modifiedHeaderFont;
  /** Ttulo que aparece apenas uma vez, acima da tabela */
  private String title;
  /** Fonte do ttulo que aparece apenas uma vez, acima da tabela */
  private Font titleFont;
  /** Fonte do ttulo modificada que aparece apenas uma vez, acima da tabela */
  private Font modifiedTitleFont;
  /** Formato de impresso da data */
  private DateFormat dateFormat;
  /** Formato de impresso dos decimais */
  private NumberFormat decimalFormat;
  /** Impresso em andamento? */
  private boolean printing;
  /** Simulao em andamento? */
  private boolean simulating;
  /** Primeira pgina impressa */
  private int firstPage;
  /** ltima pgina impressa */
  private int lastPage;
  /** Pgina do ttulo */
  private int titlePage;
  /** A prxima linha a ser impressa */
  private int nextRow;
  /** A ltima linha impressa */
  private int lastRow;
  /** A prxima coluna a ser impressa */
  private int nextCol;
  /** A ltima coluna impressa */
  private int lastCol;
  /** Quantidade de linhas impressas na ltima pgina */
  private int lastPageNumRows;
  /**
   * Quantidade de paginas verticais impressas (desconsiderando as extrapolaes
   * horizontais).
   */
  private int verticalPagesCount;
  /** A altura do papel utilizada no ltimo print */
  private float height;
  /** A largura do papel utilizada no ltimo print */
  private float width;
  /** Ferramenta para impresso de textos */
  private TextTool textTool;
  /** Indica se deve quebrar a tabela se no couber na pgina */
  private boolean breakTable;
  /** Indica se deve ajustar a tabela  largura da janela */
  private boolean adjustWidth;
  /** Largura da tabela impressa */
  private float tableWidth;
  /** Alinhamento da tabela */
  private int alignment;
  /** Espaamento  esquerda da tabela */
  private float xInset;
  /** O bit setado indica que a borda abaixo da linha  invisvel */
  private BitSet lineBorderInvisible;
  /** O bit setado indica que a borda  direita da coluna  invisvel */
  private BitSet columnBorderInvisible;
  /**
   * Colunas que esto unidas em cada linha. Se o bit da posio "x" tem valor
   * <code>true</code>, significa que a coluna "x" est unida  coluna "x + 1".
   */
  private BitSet[] joinColumns;
  /** Indica se deve alinhar o header da tabela como as colunas */
  private boolean alignTableHeader;
  /** Indica o tipo de quebra para valores da tabela */
  private String lineWrap = TextTool.WRAP_NONE;
  /** Configurao de impresso */
  private PrintConfiguration configuration;
  /** Indica se a fonte da clula deve ser alterada pela configurao */
  private boolean changeCellFont;
  /** Indica se a fonte dos textos ao redor deve ser alterada */
  private boolean changeTextFont;
  /** Indica se a o carregamento da imagem foi completado */
  private boolean imageRenderingDone;
  /** Mapa de alinhamentos para cada coluna da tabela */
  private Map<Integer, Integer> columnAlignments;
  /** Tabela de cores para cada clula da tabela */
  private Color[][] cellBackground;
  /** ltima posio vertical utilizada para desenho da tabela */
  private float lastY;

  /**
   * Armazena a fonte e o texto de cada clula da tabela. Tambm guarda se o
   * texto representa, ou no, um nmero. Permite tambm armazenar imagens.
   */
  private class CellInfo {
    /** Fonte */
    Font font;
    /** Texto */
    String text;
    /** Indica se  um nmero */
    boolean isNumber;
    /** Imagem */
    public ImageIcon image;

    /**
     * Construtor.
     * 
     * @param font fonte da clula.
     */
    CellInfo(Font font) {
      this.font =
        (!changeCellFont) ? font : font.deriveFont(font.getSize2D()
          * configuration.getFontRate());
    }
  };

  /**
   * Construtor. Ajusta os valores padro e inicializa os campos necessrios.
   * 
   * @param model modelo da tabela a ser impressa.
   */
  public DefaultPrintableTable(TableModel model) {
    this(model, new Font("SansSerif", Font.PLAIN, 10));
  }

  /**
   * Construtor. Ajusta os valores padro e inicializa os campos necessrios.
   * 
   * @param model modelo da tabela a ser impressa.
   * @param font fonte a ser utilizada.
   */
  public DefaultPrintableTable(TableModel model, Font font) {
    this(new DefaultPrintableTableModel(model), font, true);
  }

  /**
   * Construtor. Ajusta os valores padro e inicializa os campos necessrios.
   * 
   * @param model modelo da tabela a ser impressa.
   * @param font fonte a ser utilizada.
   * @param changeCellFont indica se a fonte da clula pode ser alterada pela
   *        proporo.
   */
  public DefaultPrintableTable(TableModel model, Font font,
    boolean changeCellFont) {
    this(new DefaultPrintableTableModel(model), font, changeCellFont);
  }

  /**
   * Construtor. Ajusta os valores padro e inicializa os campos necessrios.
   * 
   * @param model modelo da tabela a ser impressa.
   * @param font fonte a ser utilizada.
   * @param changeCellFont indica se a fonte da clula pode ser alterada pela
   *        proporo.
   * @param changeTextFont indica se a fonte dos textos pode ser alterada pela
   *        proporo.
   */
  public DefaultPrintableTable(TableModel model, Font font,
    boolean changeCellFont, boolean changeTextFont) {
    this(new DefaultPrintableTableModel(model), font, changeCellFont,
      changeTextFont, null);
  }

  /**
   * Construtor. Ajusta os valores padro e inicializa os campos necessrios.
   * 
   * @param model modelo da tabela a ser impressa.
   * @param font fonte a ser utilizada.
   * @param changeCellFont indica se a fonte da clula pode ser alterada pela
   *        proporo.
   * @param changeTextFont indica se a fonte dos textos pode ser alterada pela
   *        proporo.
   * @param columnGroups os agrupamentos das colunas da tabela. Se for
   *        <code>null</code>, a tabela ser impressa sem agrupamentos de
   *        colunas.
   */
  public DefaultPrintableTable(TableModel model, Font font,
    boolean changeCellFont, boolean changeTextFont,
    List<TableColumnGroup> columnGroups) {
    this(new DefaultPrintableTableModel(model), font, changeCellFont,
      changeTextFont, columnGroups);
  }

  /**
   * Construtor. Ajuda os valores padro e inicializa os campos necessrios.
   * 
   * @param model modelo da tabela a ser impressa.
   */
  public DefaultPrintableTable(PrintableTableModel model) {
    this(model, new Font("SansSerif", Font.PLAIN, 10));
  }

  /**
   * Construtor. Ajusta os valores padro e inicializa os campos necessrios.
   * 
   * @param model modelo da tabela a ser impressa.
   * @param font fonte a ser utilizada.
   * @param changeCellFont indica se a fonte da clula pode ser alterada pela
   *        proporo.
   * 
   * @throws IllegalArgumentException quando o modelo passado por parmetro 
   *         nulo.
   */
  public DefaultPrintableTable(DefaultPrintableTableModel model, Font font,
    boolean changeCellFont) {
    this(model, font, changeCellFont, true, null);
  }

  /**
   * Construtor. Ajusta os valores padro e inicializa os campos necessrios.
   * 
   * @param model modelo da tabela a ser impressa.
   * @param font fonte a ser utilizada.
   * @param changeCellFont indica se a fonte da clula pode ser alterada pela
   *        proporo.
   * @param changeTextFont indica se a fonte dos textos pode ser alterada pela
   *        proporo.
   * @param columnGroups os agrupamentos das colunas da tabela. Se for
   *        <code>null</code>, a tabela ser impressa sem agrupamentos de
   *        colunas.
   * 
   * @throws IllegalArgumentException quando o modelo passado por parmetro 
   *         nulo.
   */
  public DefaultPrintableTable(DefaultPrintableTableModel model, Font font,
    boolean changeCellFont, boolean changeTextFont,
    List<TableColumnGroup> columnGroups) {
    if (model == null) {
      throw new IllegalArgumentException("O modelo no pode ser null.");
    }
    this.model = model;
    setInsets(1, 1);
    setLineWidth(0.1F);
    defaultFont = font;
    this.changeCellFont = changeCellFont;
    this.changeTextFont = changeTextFont;
    this.columnGroups = columnGroups;
    if (this.columnGroups != null && this.columnGroups.isEmpty()) {
      this.columnGroups = null;
    }
    boolean hasRowNames = model.hasRowNames();
    boolean hasColumnGroups = this.columnGroups != null;
    int rowCount = model.getRowCount() + ((hasColumnGroups) ? 2 : 1);
    columnCount = model.getColumnCount() + (hasRowNames ? 1 : 0);
    cellInfo = new CellInfo[rowCount][columnCount];
    cellBackground = new Color[rowCount][columnCount];
    rowHeight = new float[rowCount];
    columnSize = new float[model.getColumnCount() + ((hasRowNames) ? 1 : 0)];
    dateFormat = DateFormat.getDateInstance();
    decimalFormat = NumberFormat.getInstance(LNG.getLocale());
    textTool = new TextTool();
    printing = false;
    simulating = false;
    alignment = LEFT;
    columnAlignments = new HashMap<Integer, Integer>();
    xInset = 0;
    adjustWidth = true;
    alignTableHeader = false;
    lineBorderInvisible = new BitSet();
    columnBorderInvisible = new BitSet();
    joinColumns = new BitSet[rowCount];
    for (int row = 0; row < joinColumns.length; row++) {
      joinColumns[row] = new BitSet();
    }
    initPrinting(new PrintConfiguration());
  }

  /**
   * Preenche um CellInfo para cada clula da tabela. As clulas cuja fonte
   * tenha sido especificada pelo usurio j tero um CellInfo criado, mas sem o
   * texto correspondente. Para as demais clulas, este mtodo cria o CellInfo,
   * usando a fonte padro. Em qualquer caso, armazena no CellInfo o texto, a
   * imagem (se houver) e o indicativo se  nmero ou no.
   * 
   * @param g2d Graphics2D necessrio para calcular o tamanho dos textos.
   * @param availableWidthForText espao disponvel para o texto das colunas
   *        (descontando-se as linhas).
   */
  private void buildCellInfo(Graphics2D g2d, float availableWidthForText) {
    for (int row = 0; row < cellInfo.length; row++) {
      for (int column = 0; column < cellInfo[row].length; column++) {
        String text = getCellText(row, column);
        ImageIcon image = getCellImage(row, column);
        boolean isNumber = isCellNumber(row, column);
        if (cellInfo[row][column] == null) {
          cellInfo[row][column] = new CellInfo(defaultFont);
        }
        g2d.setFont(cellInfo[row][column].font);
        cellInfo[row][column].image = image;
        cellInfo[row][column].isNumber = isNumber;

        cellInfo[row][column].text = text;
        if (columnWeight != null) {
          float maxWidth =
            (availableWidthForText * columnWeight[column]) - horizontalInset;
          cellInfo[row][column].text =
            textTool.wrapText(g2d, text, (int) maxWidth, getLineWrap());
        }
      }
    }
  }

  /**
   * Distribui o espao extra (se houver) uniformemente entre as colunas da
   * tabela.
   * 
   * @param imageableWidth espao disponvel para impresso da tabela.
   * @param availableWidthForText espao disponvel somente para o texto das
   *        colunas, descontando-se a largura das linhas.
   * @param linesWidth largura ocupada pelas bordas (linhas verticais da
   *        tabela).
   */
  private void distributeSpaceEvenly(float imageableWidth,
    float availableWidthForText, float linesWidth) {
    tableWidth = imageableWidth;
    float totalColumnSize = 0;
    for (int column = 0; column < columnSize.length; column++) {
      totalColumnSize += columnSize[column];
    }
    float blankSpaceByColumn =
      (availableWidthForText - totalColumnSize) / columnSize.length;
    if (blankSpaceByColumn <= 0) {
      return;
    }
    if (!adjustWidth) {
      blankSpaceByColumn /= 8;
    }
    for (int column = 0; column < columnSize.length; column++) {
      columnSize[column] += blankSpaceByColumn;
    }
    if (!adjustWidth) {
      totalColumnSize += (blankSpaceByColumn * columnSize.length);
      tableWidth = totalColumnSize + linesWidth;

      switch (alignment) {
        case LEFT:
          xInset = 0;
          break;
        case RIGHT:
          xInset = imageableWidth - tableWidth;
          break;
        case CENTER:
          xInset = (imageableWidth - tableWidth) / 2;
          break;
      }
    }
  }

  /**
   * Distribui o espao disponvel proporcionalmente entre as colunas, de acordo
   * com uma distribuio previamente definida (ver mtodo
   * {@link #setColumnPart(float[])}).
   * 
   * @param imageableWidth espao disponvel para impresso da tabela.
   * @param availableWidthForText espao disponvel somente para o texto das
   *        colunas, descontando-se a largura das linhas.
   */
  private void distributeSpaceProportionally(float imageableWidth,
    float availableWidthForText) {
    /* Largura das colunas distribuda proporcionalmente */
    tableWidth = imageableWidth;
    for (int col = 0; col < columnSize.length; col++) {
      columnSize[col] = availableWidthForText * columnWeight[col];
    }
  }

  /**
   * Une colunas da tabela com base na informao em {@link #columnGroups}.
   */
  private void joinColumnGroups() {
    for (TableColumnGroup group : columnGroups) {
      int colOffset = (model.hasRowNames()) ? 1 : 0;
      int startCol = group.getStartIndex() + colOffset;
      int endCol = group.getEndIndex() + colOffset;
      /*
       * -2  o ndice da linha dos agrupamentos de colunas no contexto do
       * mtodo joinColumns()
       */
      joinColumns(-2, startCol, endCol);
    }
  }

  /**
   * Calcula a altura e a largura preferida para cada clula.
   * 
   * @param g2d referncia para a ferramenta de desenho.
   */
  private void calculatePreferredCellSize(Graphics2D g2d) {
    Point2D pt = new Point2D.Float(0, 0);
    Rectangle2D bbox = new Rectangle2D.Float();
    for (int row = 0; row < cellInfo.length; row++) {
      for (int column = 0; column < cellInfo[row].length; column++) {

        float preferredCellWidth = 0;
        float preferredCellHeight = 0;
        if (cellInfo[row][column].image == null) {
          // Sem imagem
          textTool.getBBox(g2d, cellInfo[row][column].text, pt,
            TextTool.NORTH_WEST, bbox);
          if (isJoined(row, column)) {
            preferredCellWidth = 0;
          }
          else {
            preferredCellWidth = (float) bbox.getWidth() + 2 * horizontalInset;
          }
          preferredCellHeight = (float) bbox.getHeight() + 2 * verticalInset;
        }
        else {
          // Com imagem
          preferredCellWidth =
            cellInfo[row][column].image.getIconWidth() + 2 * horizontalInset;
          preferredCellHeight =
            cellInfo[row][column].image.getIconHeight() + 2 * verticalInset;
        }
        if (lineWidth > 0) {
          if (row == 0) {
            // primeira linha da tabela fica com a linha desenhada acima:
            preferredCellHeight += lineWidth;
          }
          // cada linha da tabela fica com a linha desenhada abaixo:
          preferredCellHeight += lineWidth;
        }
        rowHeight[row] = Math.max(rowHeight[row], preferredCellHeight);
        columnSize[column] = Math.max(columnSize[column], preferredCellWidth);
      }
    }
  }

  /**
   * Calcula a largura preferida de todas as clulas que representarem grupos.
   * 
   * @param g2d referncia para a ferramenta de desenho.
   */
  private void calculateColumnGroupsPreferredWidth(Graphics2D g2d) {
    /*
     * Estou assumindo que os agrupamentos de colunas s podem ser feitos na
     * primeira linha, e que o contedo da primeira clula do agrupamento ir
     * prevalecer sobre todas as clulas do agrupamento.
     */
    Point2D pt = new Point2D.Float(0, 0);
    Rectangle2D bbox = new Rectangle2D.Float();
    for (TableColumnGroup columnGroup : columnGroups) {
      int startIndex = columnGroup.getStartIndex();
      int endIndex = columnGroup.getEndIndex();
      float currentNeighborColumnSizes = 0;
      for (int i = startIndex; i <= endIndex; i++) {
        currentNeighborColumnSizes += columnSize[i];
      }
      float colGroupSize = 0;
      if (cellInfo[0][startIndex].image == null) {
        textTool.getBBox(g2d, cellInfo[0][startIndex].text, pt,
          TextTool.NORTH_WEST, bbox);
        colGroupSize = (float) bbox.getWidth() + 2 * horizontalInset;
      }
      else {
        colGroupSize =
          cellInfo[0][startIndex].image.getIconWidth() + 2 * horizontalInset;
      }
      float colGroupRemaining = colGroupSize - currentNeighborColumnSizes;
      if (colGroupRemaining > 0) {
        shareRemainingAmongColumnSizes(colGroupRemaining, startIndex, endIndex,
          currentNeighborColumnSizes);
      }
    }
  }

  /**
   * @param row ndice da linha da clula.
   * @param column ndice da coluna da clula.
   * @return texto para a clula.
   */
  private String getCellText(int row, int column) {
    String text = null;
    if (columnGroups != null && row == 0) {
      for (TableColumnGroup group : columnGroups) {
        if (column == group.getStartIndex()) {
          text = group.getName();
          break;
        }
      }
    }
    else {
      if (row == 0 || (columnGroups != null && row == 1)) {
        String cornerName = model.getCornerName();
        if (cornerName != null) {
          if (column == 0) {
            text = cornerName;
          }
          else {
            text = model.getColumnName(column - 1);
          }
        }
        else {
          text = model.getColumnName(column);
        }
      }
      else {
        int adjustedRow = columnGroups == null ? row - 1 : row - 2;
        if (column == 0 && model.hasRowNames()) {
          text = model.getRowName(adjustedRow);
        }
        else {
          Object value;
          if (model.hasRowNames()) {
            value = model.getValueAt(adjustedRow, column - 1);
          }
          else {
            value = model.getValueAt(adjustedRow, column);
          }
          if (value == null) {
            text = "";
          }
          else if (FooterCell.class.isInstance(value)) {
            value = FooterCell.class.cast(value).getValue();
          }
          if (value == null) {
            text = "";
          }
          else if (Colored.class.isInstance(value)) {
            value = Colored.class.cast(value).getValue();
          }
          if (value == null) {
            text = "";
          }
          else {
            if (value instanceof Date) {
              text = dateFormat.format((Date) value);
            }
            else if (value instanceof Number) {
              text = decimalFormat.format(value);
            }
            else if (value instanceof Boolean) {
              text = ((Boolean) value) ? trueString : falseString;
            }
            else {
              text = value.toString();
            }
          }
        }
      }
    }
    if (text == null) {
      text = "";
    }
    return text;
  }

  /**
   * @param row ndice da linha da clula.
   * @param column ndica da coluna da clula.
   * @return true se a clula contiver um nmero, false caso contrrio.
   */
  private boolean isCellNumber(int row, int column) {
    boolean isNumber = false;
    if (row == 0) {
      return false;
    }
    if (row == 1 && columnGroups != null) {
      return false;
    }
    int adjustedRow = columnGroups == null ? row - 1 : row - 2;
    Object value;
    if (model.hasRowNames()) {
      value = model.getValueAt(adjustedRow, column - 1);
    }
    else {
      value = model.getValueAt(adjustedRow, column);
    }

    if (FooterCell.class.isInstance(value)) {
      value = FooterCell.class.cast(value).getValue();
    }
    else if (Colored.class.isInstance(value)) {
      value = Colored.class.cast(value).getValue();
    }

    if (value instanceof Number) {
      isNumber = true;
    }
    return isNumber;
  }

  /**
   * @param row ndice da linha da clula.
   * @param column ndica da coluna da clula.
   * @return imagem da clula especificada ou null se a clula no contiver
   *         nenhuma imagem.
   */
  private ImageIcon getCellImage(int row, int column) {
    ImageIcon image = null;
    Object value;
    if (row == 0) {
      return null;
    }
    if (row == 1 && columnGroups != null) {
      return null;
    }
    int adjustedRow = columnGroups == null ? row - 1 : row - 2;
    if (model.hasRowNames()) {
      value = model.getValueAt(adjustedRow, column - 1);
    }
    else {
      value = model.getValueAt(adjustedRow, column);
    }

    if (FooterCell.class.isInstance(value)) {
      value = FooterCell.class.cast(value).getValue();
    }
    else if (Colored.class.isInstance(value)) {
      value = Colored.class.cast(value).getValue();
    }

    if (ImageIcon.class.isInstance(value)) {
      image = ImageIcon.class.cast(value);
    }
    return image;
  }

  /**
   * <p>
   * Distribui proporcionalmente o espao remanescente do agrupamento de colunas
   * existentes.
   * </p>
   * Este mtodo  chamado no caso de uma ou mais colunas agrupadas necessitarem
   * de mais espao do que a soma dos espaos ocupados pelas mesmas colunas, nas
   * demais linhas. A sobra  ento distribuda proporcionalmente entre as
   * colunas, para que a largura total seja igual ou maior  largura das colunas
   * agrupadas.
   * 
   * @param colGroupRemaining espao remanescente do agrupamento de colunas.
   * @param startIndex ndice inicial do agrupamento.
   * @param endIndex ndice final do agrupamento.
   * @param totalColumnSizes soma dos espaos ocupados pelas colunas das linhas
   *        abaixo do agrupamento de colunas.
   */
  private void shareRemainingAmongColumnSizes(float colGroupRemaining,
    int startIndex, int endIndex, float totalColumnSizes) {
    for (int i = startIndex; i <= endIndex; i++) {
      columnSize[i] += colGroupRemaining * columnSize[i] / totalColumnSizes;
    }
  }

  /**
   * Inicializa as informaes necessrias para a impresso.
   * 
   * @param g2d Graphics2D da impresso.
   * @param page ndice da primeira pgina sendo solicitada.
   * @param imageableWidth largura da rea a ser impressa.
   */
  private void init(Graphics2D g2d, int page, float imageableWidth) {
    if (columnGroups != null) {
      joinColumnGroups();
    }
    // largura ocupada pelas linhas verticais (bordas)
    float linesWidth = 0;
    // largura disponvel para as clulas (descontando as bordas)
    float availableWidthForText = imageableWidth;
    if (drawLines) {
      linesWidth = lineWidth * (columnSize.length + 1);
      availableWidthForText = imageableWidth - linesWidth;
    }
    buildCellInfo(g2d, availableWidthForText);
    calculatePreferredCellSize(g2d);
    if (columnGroups != null) {
      calculateColumnGroupsPreferredWidth(g2d);
    }

    if (columnWeight != null) {
      distributeSpaceProportionally(imageableWidth, availableWidthForText);
    }
    else {
      distributeSpaceEvenly(imageableWidth, availableWidthForText, linesWidth);
    }
    nextRow = 0;
    lastRow = -1;
    nextCol = 0;
    lastCol = -1;
    height = 0;
    width = 0;
    titlePage = page;
    firstPage = page;
    lastPage = page - 1;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setDefaultFont(Font font) {
    defaultFont = font;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setCellFont(int fromRow, int toRow, int fromColumn, int toColumn,
    Font font) {
    // ajuste de coordenadas: do modelo para a matriz de impresso
    fromRow++;
    toRow++;
    if (columnGroups != null) {
      fromRow++;
      toRow++;
    }
    if (model.hasRowNames()) {
      fromColumn++;
      toColumn++;
    }
    for (int row = fromRow; row <= toRow; row++) {
      for (int column = fromColumn; column <= toColumn; column++) {
        cellInfo[row][column] = new CellInfo(font);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setColumnNameFont(int fromColumn, int toColumn, Font font) {
    // ajuste de coordenadas: do modelo para a matriz de impresso
    if (model.hasRowNames()) {
      fromColumn++;
      toColumn++;
    }
    for (int column = fromColumn; column <= toColumn; column++) {
      cellInfo[0][column] = new CellInfo(font);
      if (columnGroups != null) {
        cellInfo[1][column] = new CellInfo(font);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setRowNameFont(int fromRow, int toRow, Font font) {
    if (model.hasRowNames()) {
      // ajuste de coordenadas: do modelo para a matriz de impresso
      fromRow++;
      toRow++;
      if (columnGroups != null) {
        fromRow++;
        toRow++;
      }
      for (int row = fromRow; row <= toRow; row++) {
        cellInfo[row][0] = new CellInfo(font);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setCornerNameFont(Font font) {
    if (model.getCornerName() != null) {
      int row = 0;
      if (columnGroups != null) {
        row = 1;
      }
      cellInfo[row][0] = new CellInfo(font);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setLineWidth(float lineWidth) {
    this.lineWidth = lineWidth;
    drawLines = (lineWidth >= 0);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setInsets(float dx, float dy) {
    this.horizontalInset = dx;
    this.verticalInset = dy;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setPosText(String posText) {
    setPosText(posText, defaultFont);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setPosText(String posText, Font posTextFont) {
    this.posText = posText;
    this.posTextFont = posTextFont;
    modifiedPosTextFont =
      posTextFont.deriveFont(posTextFont.getSize()
        * configuration.getFontRate());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setTableHeader(String header) {
    setTableHeader(header, defaultFont);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setTableHeader(String header, Font headerFont) {
    this.header = header;
    this.headerFont = headerFont;
    modifiedHeaderFont =
      headerFont.deriveFont(headerFont.getSize() * configuration.getFontRate());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setTitle(String title) {
    setTitle(title, defaultFont);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setTitle(String title, Font titleFont) {
    this.title = title;
    this.titleFont = titleFont;
    modifiedTitleFont =
      titleFont.deriveFont(titleFont.getSize() * configuration.getFontRate());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setDateFormat(DateFormat dateFormat) {
    this.dateFormat = dateFormat;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setDecimalFormat(NumberFormat decimalFormat) {
    this.decimalFormat = decimalFormat;
  }

  /**
   * Imprime um texto na posio indicada. O texto s  desenhado se couber na
   * parte livre da pgina, ou se o parmetro <code>force</code> for verdadeiro.
   * A parte livre da pgina  o que sobra entre o getHeight e a altura livre
   * (imageable) do PageFormat.
   * 
   * @param g2d Graphics2D para impresso.
   * @param format especificao da pgina.
   * @param text texto a ser impresso.
   * @param pt ponto de referncia.
   * @param ref relao entre o ponto e a caixa envolvente do texto.
   * @param alignment alinhamento do texto, caso tenha mais de uma linha.
   * @param force se verdadeiro, fora o desenho mesmo que no caiba.
   * @param draw indica se  para desenhar ou apenas para simular.
   * 
   * @return verdadeiro se o texto foi impresso, falso caso contrrio.
   */
  private boolean drawText(Graphics2D g2d, PageFormat format, String text,
    Point2D pt, String ref, String alignment, boolean force, boolean draw) {
    float y = (float) format.getImageableY();
    float h = (float) format.getImageableHeight();
    Rectangle2D bbox = new Rectangle2D.Float();
    textTool.getBBox(g2d, text, pt, ref, bbox);
    if (bbox.getMaxY() > y + h && !force) {
      return false;
    }
    if (draw) {
      textTool.draw(g2d, text, pt, ref, alignment);
    }
    height += bbox.getHeight();
    width += bbox.getWidth();
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean simulatePrint(Graphics2D g2d, PageFormat format, int page) {
    if (!simulating) {
      init(g2d, page, (float) format.getImageableWidth());
      simulating = true;
    }
    if ((page == firstPage) && (!breakTable)) {
      if (!print(g2d, format, page, false)) {
        init(g2d, page, (float) format.getImageableWidth());
        return false;
      }
    }
    return print(g2d, format, page, false);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean print(Graphics2D g2d, PageFormat format, int page) {
    if (!printing) {
      init(g2d, page, (float) format.getImageableWidth());
      printing = true;
    }
    if ((page == firstPage) && (!breakTable)) {
      if (!print(g2d, format, page, false)) {
        init(g2d, page, (float) format.getImageableWidth());
        return false;
      }
    }
    return print(g2d, format, page, true);
  }

  /**
   * Calcula o nmero de colunas da tabela que cabem em uma pgina.
   * 
   * @param x coordenada horizontal inicial de desenho da tabela.
   * @param availableWidth largura disponvel para a tabela.
   * 
   * @return nmero de colunas da tabela que cabem em uma pgina.
   */
  private int calculateColumnsInPage(float x, float availableWidth) {
    int numCols = 0;
    float lastX = x + width + lineWidth;
    for (int col = nextCol; col < columnCount; col++) {
      if (lastX + columnSize[col] + lineWidth > x + availableWidth) {
        break;
      }
      lastX += columnSize[col] + lineWidth;
      numCols++;
    }
    return numCols;
  }

  /**
   * Calcula o nmero de linhas da tabela que cabem em uma pgina.
   * 
   * @param y coordenada vertical inicial de desenho da tabela.
   * @param availableHeight altura disponvel para a tabela.
   * 
   * @return nmero de colunas da tabela que cabem em uma pgina.
   */
  private int calculateRowsInPage(float y, float availableHeight) {
    int numRows = 0;
    lastY = y + height;
    for (int row = nextRow; row < rowHeight.length; row++) {
      if (lastY + rowHeight[row] > y + availableHeight) {
        break;
      }
      lastY += rowHeight[row];
      numRows++;
    }
    return numRows;
  }

  /**
   * Desenha linhas horizontais (bordas da tabela).
   * 
   * @param g2d referncia para a ferramenta de desenho.
   * @param x coordenada horizontal inicial.
   * @param y coordenada vertical inicial.
   * @param numRows nmero de linhas da tabela.
   */
  private void drawHorizontalLines(Graphics2D g2d, float x, float y, int numRows) {
    float halfLineWidth = lineWidth / 2;
    Line2D line = new Line2D.Float();
    g2d.setStroke(new BasicStroke(lineWidth, BasicStroke.CAP_BUTT,
      BasicStroke.JOIN_BEVEL));
    float y1 = y + height;
    line.setLine(x, y1 + halfLineWidth, x + tableWidth, y1 + halfLineWidth);
    g2d.draw(line);
    for (int row = nextRow; row < nextRow + numRows; row++) {
      y1 += rowHeight[row];
      if (lineBorderInvisible.get(row)) {
        continue;
      }
      line.setLine(x, y1 - halfLineWidth, x + tableWidth, y1 - halfLineWidth);
      g2d.draw(line);
    }
  }

  /**
   * Desenha linhas verticais (bordas da tabela).
   * 
   * @param g2d referncia para a ferramenta de desenho.
   * @param x coordenada horizontal inicial.
   * @param y coordenada vertical inicial.
   * @param numCols nmero de colunas da tabela.
   */
  private void drawVerticalLines(Graphics2D g2d, float x, float y, int numCols) {
    float halfLineWidth = lineWidth / 2;
    Line2D line = new Line2D.Float();
    float x1 = x;
    line.setLine(x1 + halfLineWidth, y + height, x1 + halfLineWidth, lastY);
    g2d.draw(line);
    for (int column = nextCol; column < nextCol + numCols; column++) {
      float adjustedY = y;
      if (columnGroups != null && nextRow == 0 && isInColumnGroup(column)) {
        adjustedY += rowHeight[0];
      }
      x1 += lineWidth + columnSize[column];
      if (columnBorderInvisible.get(column)) {
        continue;
      }
      line.setLine(x1 + halfLineWidth, adjustedY + height, x1 + halfLineWidth,
        lastY);
      g2d.draw(line);
    }
  }

  /**
   * Imprime a tabela.
   * 
   * @param g2d contexto grfico.
   * @param format formato da pgina.
   * @param page ndice da pgina.
   * @param draw indica se deve desenhar no contexto.
   * 
   * @return verdadeiro se a tabela foi totalmente impressa.
   */
  private boolean print(Graphics2D g2d, PageFormat format, int page,
    boolean draw) {
    boolean pageEmpty = true;
    float x = (float) format.getImageableX() + xInset;
    float y = (float) format.getImageableY();
    float availableWidth = (float) format.getImageableWidth();
    float availableHeight = (float) format.getImageableHeight();
    height = 0;
    // Ttulo
    if (title != null && page == titlePage) {
      Point2D pt = new Point2D.Float(x + tableWidth / 2, y);
      boolean force = page > firstPage;
      g2d.setFont(modifiedTitleFont);
      if (!drawText(g2d, format, title, pt, TextTool.NORTH,
        TextTool.ALIGN_CENTER, force, draw)) {
        titlePage = page + 1;
        if (titlePage > firstPage + 1) {
          // Se no cabe em uma pgina vazia, desiste da impresso.
          return true;
        }
        lastPage = page;
        return false;
      }
      pageEmpty = false;
    }
    // Header
    if (header != null) {
      Point2D pt = new Point2D.Float(x, y + height);
      g2d.setFont(modifiedHeaderFont);
      if (!drawText(g2d, format, header, pt, TextTool.NORTH_WEST,
        TextTool.ALIGN_LEFT, false, draw)) {
        if (!pageEmpty || page == firstPage) {
          lastPage = page;
          return false;
        }
        // Se no cabe em uma pgina vazia, desiste da impresso.
        return true;
      }
      pageEmpty = false;
    }
    width = 0;
    // Tabela
    if (page > lastPage && lastRow > 0) {
      // Verifica se a ltima coluna impressa corresponde a ltima coluna da tabela 
      if (lastCol == columnCount - 1) {
        nextRow = lastRow + 1;
        nextCol = 0;
        verticalPagesCount++;
      }
      else {
        nextRow = verticalPagesCount * lastPageNumRows;
        nextCol = lastCol + 1;
      }
    }
    if (nextRow < cellInfo.length) {
      int numCols = calculateColumnsInPage(x, availableWidth);
      int numRows = calculateRowsInPage(y, availableHeight);
      lastPageNumRows = numRows;
      if (numRows == 0 && pageEmpty && page != firstPage) {
        // Se no cabe nem uma linha em uma pgina vazia, desiste da impresso.
        return true;
      }
      if (numRows == 0 && !pageEmpty && page != titlePage) {
        // Imprimindo apenas headers infinitamente... Desiste da impresso.
        return true;
      }
      if (numRows == 0 && !pageEmpty) {
        lastPage = page;
        return false;
      }
      if (draw && drawLines) {
        drawHorizontalLines(g2d, x, y, numRows);
        drawVerticalLines(g2d, x, y, numCols);
      }
      Shape oldClip = g2d.getClip();
      Rectangle2D clip = new Rectangle2D.Float();
      Point2D pt = new Point2D.Float();
      for (int row = nextRow; row < nextRow + numRows; row++) {
        float x1 = x;
        float y1 = y + height;
        for (int column = nextCol; column < nextCol + numCols; column++) {
          String ref;
          String alignment;
          float columnWidth = getColumnWidth(row, column);
          /* Colunas unidas */
          if (columnWidth < 0) {
            continue;
          }
          if (row == 0 || (columnGroups != null && row == 1)) {
            float yOffset = (row == 0) ? lineWidth : 0;
            float lineOffset = (row == 0) ? 2 * lineWidth : lineWidth;
            clip.setRect(x1 + lineWidth, y1 + yOffset, columnWidth,
              rowHeight[row] - lineOffset);
            if (!alignTableHeader) {
              //#TODO na reviso de classes, utilizar os mtodos 
              // setRefPointLocation e getCellReference tambm para o cabealho
              pt.setLocation(x1 + lineWidth + columnWidth / 2, y1
                + rowHeight[row] / 2);
              ref = TextTool.CENTER;
              alignment = TextTool.ALIGN_CENTER;
            }
            else {
              if (cellInfo[row + 1][column].isNumber) {
                pt.setLocation(x1 + lineWidth + columnWidth - horizontalInset,
                  y1 + (rowHeight[row] - lineWidth) / 2);
                ref = TextTool.EAST;
              }
              else {
                pt.setLocation(x1 + lineWidth + horizontalInset, y1
                  + (rowHeight[row] - lineWidth) / 2);
                ref = TextTool.WEST;
              }
              alignment = TextTool.ALIGN_LEFT;
            }
          }
          else {
            clip.setRect(x1 + lineWidth, y1, columnWidth, rowHeight[row]
              - lineWidth);
            if (cellInfo[row][column].isNumber) {
              Integer cellAlignment = columnAlignments.get(column);
              if (cellAlignment == null) {
                cellAlignment = PrintableTable.RIGHT;
              }
              setRefPointLocation(pt, lineWidth, row, x1, y1, columnWidth,
                cellAlignment);
              ref = getCellReference(cellAlignment);
            }
            else if (cellInfo[row][column].image != null) {
              int imgWidth = cellInfo[row][column].image.getIconWidth();
              int imgHeight = cellInfo[row][column].image.getIconHeight();
              pt.setLocation((x1 + lineWidth + (columnWidth - imgWidth) / 2),
                (y1 + lineWidth + (rowHeight[row] - imgHeight) / 2));
              ref = TextTool.EAST; // No necessrio para desenhar imagem
            }
            else {
              Integer cellAlignment = columnAlignments.get(column);
              if (cellAlignment == null) {
                cellAlignment = PrintableTable.LEFT;
              }
              setRefPointLocation(pt, lineWidth, row, x1, y1, columnWidth,
                cellAlignment);
              ref = getCellReference(cellAlignment);
            }
            alignment = TextTool.ALIGN_LEFT;
          }
          g2d.setClip(clip);
          g2d.setFont(cellInfo[row][column].font);
          String text = cellInfo[row][column].text;
          ImageIcon image = cellInfo[row][column].image;
          if (draw) {
            fillBackground(g2d, lineWidth, row, column, x1, y1);
            if (image == null) {
              textTool.draw(g2d, text, pt, ref, alignment);
            }
            else {
              drawImage(image, g2d, (int) pt.getX(), (int) pt.getY());
            }
          }
          x1 += columnWidth + lineWidth;
          width += columnSize[column];
        }
        height += rowHeight[row];
      }
      g2d.setClip(oldClip);
      lastRow = nextRow + numRows - 1;
      lastCol = nextCol + numCols - 1;
      // Verificando se todas as linhas e colunas foram impressas
      if (nextRow + numRows < cellInfo.length
        || nextCol + numCols < columnCount) {
        lastPage = page;
        return false;
      }
      pageEmpty = false;
    }
    // Rodap
    if (posText != null) {
      Point2D pt = new Point2D.Float(x, y + height);
      g2d.setFont(modifiedPosTextFont);
      if (!drawText(g2d, format, posText, pt, TextTool.NORTH_WEST,
        TextTool.ALIGN_LEFT, false, draw)) {
        if (!pageEmpty) {
          lastPage = page;
          return false;
        }
        // Se no cabe em uma pgina vazia, desiste da impresso.
        return true;
      }
      pageEmpty = false;
    }
    lastPage = page;
    return true;
  }

  /**
   * Desenha a caixa da clula, isto , as linhas delimitadoras e, se for o
   * caso, preenche o seu fundo com uma cor pr-determinada.
   * 
   * @param g2d superfcie de desenho.
   * @param sW espessura da linha.
   * @param row ndice da linha.
   * @param col ndice da coluna.
   * @param x coordenada horizontal do canto superior esquerdo da clula.
   * @param y coordenada vertical do canto superior esquerdo da clula.
   */
  private void fillBackground(Graphics2D g2d, float sW, int row, int col,
    float x, float y) {
    Color background = cellBackground[row][col];
    if (background != null) {
      Paint paintBackup = g2d.getPaint();
      g2d.setPaint(background);
      Rectangle2D rect;
      if (row == 0) {
        rect =
          new Rectangle2D.Float(x + sW, y + sW, getColumnWidth(row, col),
            rowHeight[row] - 2 * sW);
      }
      else {
        rect =
          new Rectangle2D.Float(x + sW, y, getColumnWidth(row, col),
            rowHeight[row] - sW);
      }
      g2d.fill(rect);
      g2d.setPaint(paintBackup);
    }
  }

  /**
   * Posiciona o ponto de referncia para desenho de uma clula, de acordo com o
   * alinhamento desejado.
   * 
   * @param pt ponto a ser posicionado.
   * @param _lineWidth espessura das linhas da tabela.
   * @param row ndice da linha atual.
   * @param x1 coordenada horizontal do canto esquerdo da clula.
   * @param y1 coordenada vertical do canto superior da clula.
   * @param columnWidth largura da coluna.
   * @param alignment alinhamento desejado.
   */
  private void setRefPointLocation(Point2D pt, float _lineWidth, int row,
    float x1, float y1, float columnWidth, int alignment) {
    switch (alignment) {
      case LEFT:
        pt.setLocation(x1 + _lineWidth + horizontalInset, y1
          + (rowHeight[row] - _lineWidth) / 2);
        break;
      case CENTER:
        pt.setLocation(x1 + (lineWidth + columnWidth - horizontalInset) / 2, y1
          + (rowHeight[row] - _lineWidth) / 2);
        break;
      case RIGHT:
        pt.setLocation(x1 + _lineWidth + columnWidth - horizontalInset, y1
          + (rowHeight[row] - _lineWidth) / 2);
        break;
      default:
        throw new IllegalArgumentException("alignment: " + alignment);
    }
  }

  /**
   * Retorna o posicionamento do texto em relao ao ponto de referncia, de
   * acordo com o alinhamento desejado para a clula.
   * 
   * @param alignment alinhamento da clula.
   * 
   * @return posicionamento do texto em relao ao ponto de referncia.
   */
  private String getCellReference(int alignment) {
    switch (alignment) {
      case LEFT:
        return TextTool.WEST;
      case CENTER:
        return TextTool.CENTER;
      case RIGHT:
        return TextTool.EAST;
      default:
        throw new IllegalArgumentException("alignment: " + alignment);
    }
  }

  /**
   * Desenha uma imagem nas coordenadas passadas.
   * 
   * @param image imagem a ser desenhada.
   * @param g2 contexto grfico.
   * @param x coordenada horizontal.
   * @param y coordenada vertical.
   */
  private void drawImage(ImageIcon image, Graphics2D g2, int x, int y) {
    int height = image.getIconHeight();
    int width = image.getIconWidth();
    Image img = image.getImage();
    imageRenderingDone = false;
    boolean drawn = g2.drawImage(img, x, y, width, height, this);
    if (!drawn) {
      while (true) {
        synchronized (this) {
          if (imageRenderingDone) {
            break;
          }
        }
        try {
          Thread.sleep(100);
        }
        catch (InterruptedException e) {
        }
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized boolean imageUpdate(Image img, int infoflags, int x,
    int y, int width, int height) {
    imageRenderingDone = ((infoflags & (ERROR | FRAMEBITS | ALLBITS)) != 0);
    return !imageRenderingDone;
  }

  /**
   * Determina se, na linha dos agrupamentos de colunas, deve ser desenhada a
   * linha vertical depois da coluna <code>column</code>.
   * 
   * @param column uma coluna da tabela.
   * @return falso, caso a linha vertical aps a coluna <code>column</code> no
   *         deva ser desenhada na linha dos agrupamentos de colunas.
   */
  private boolean isInColumnGroup(int column) {
    for (TableColumnGroup group : columnGroups) {
      if (column >= group.getStartIndex() && column < group.getEndIndex()) {
        return true;
      }
    }
    return false;
  }

  /**
   * Obtm a largura da coluna de determinada linha, verificando se a coluna
   * est unida a outras. Se a coluna  a primeira das unidas, retorna a soma
   * das larguras das colunas, seno retorna -1.
   * 
   * @param row linha a ter a largura da coluna determinada.
   * @param column coluna a ter a largura determinada.
   * @return a largura da coluna de determinada linha;
   */
  private float getColumnWidth(int row, int column) {
    if (isJoined(row, column)) {
      if (isFirstJoinColumn(row, column)) {
        float width = columnSize[column];
        int col = column;
        do {
          width += lineWidth + columnSize[++col];
        } while (!isLastJoinColumn(row, col));
        return width;
      }
      return -1;
    }
    return columnSize[column];
  }

  /**
   * Verifica se  a primeira coluna do grupamento de colunas.
   * 
   * @param row linha do grupamento.
   * @param column coluna do grupamento.
   * 
   * @return verdadeiro se for a primeira coluna do grupamento de colunas.
   */
  private boolean isFirstJoinColumn(int row, int column) {
    if (!isJoined(row, column)) {
      throw new IllegalStateException("Linha e coluna no esto agrupadas");
    }
    if (column == 0) {
      return true;
    }
    return joinColumns[row].get(column) && !joinColumns[row].get(column - 1);
  }

  /**
   * Verifica se  a ltima coluna do grupamento de colunas.
   * 
   * @param row linha do grupamento.
   * @param column coluna do grupamento.
   * 
   * @return verdadeiro se for a ltima coluna do grupamento de colunas.
   */
  private boolean isLastJoinColumn(int row, int column) {
    if (!isJoined(row, column)) {
      throw new IllegalStateException("Linha e coluna no esto agrupadas");
    }
    if (column == 0) {
      return false;
    }
    return !joinColumns[row].get(column) && joinColumns[row].get(column - 1);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public float getHeight() {
    return height;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setNamesDefaultFont(Font font) {
    setRowNameFont(0, model.getRowCount() - 1, font);
    setColumnNameFont(0, model.getColumnCount() - 1, font);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setColumnPart(float[] columnPart) {
    if (columnPart.length != columnSize.length) {
      throw new IllegalArgumentException(
        "Nmero de colunas diferente do nmero de colunas da tabela");
    }
    float total = 0;
    for (float part : columnPart) {
      total += part;
    }
    float factor = 1 / total;
    for (int i = 0; i < columnPart.length; i++) {
      columnPart[i] = columnPart[i] * factor;
    }
    this.columnWeight = columnPart;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setAdjustWidth(boolean adjustWidth) {
    this.adjustWidth = adjustWidth;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setColumnAlignment(int[] columnIndexes, int alignment) {
    if (columnIndexes == null) {
      throw new IllegalArgumentException("columnIndexes == null");
    }
    for (int i = 0; i < columnIndexes.length; i++) {
      columnAlignments.put(columnIndexes[i], alignment);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setAlignment(int alignment) {
    this.alignment = alignment;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void joinColumns(int row, int startCol, int endCol) {
    /* Header da tabela */
    int rowIndex = row + 1;
    if (columnGroups != null) {
      rowIndex++;
    }
    joinColumns[rowIndex].set(startCol, endCol);
  }

  /**
   * <p>
   * Indica se a clula especificada pelos parmetros (linha, coluna) faz parte
   * ou no de uma agregao de colunas.
   * </p>
   * OBS: Este mtodo no valida parmetros negativos.
   * 
   * @param row ndice da linha (0-based).
   * @param column ndice da coluna (0-based).
   * 
   * @return true se a clula fizer parte de uma agregao de colunas.
   */
  private boolean isJoined(int row, int column) {
    if (column == 0) {
      return joinColumns[row].get(column);
    }
    return (joinColumns[row].get(column) || joinColumns[row].get(column - 1));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setColumnBorderVisible(int startCol, int endCol, boolean visible) {
    columnBorderInvisible.set(startCol, endCol, !visible);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setLineBorderVisible(int startRow, int endRow, boolean visible) {
    /* Header da tabela */
    int startRowIndex = startRow + 1;
    int endRowIndex = endRow + 1;
    lineBorderInvisible.set(startRowIndex, endRowIndex, !visible);
  }

  /**
   * Obtm o modelo utilizado na tabela.
   * 
   * @return modelo utilizado na tabela.
   */
  public TableModel getModel() {
    return model.getModel();
  }

  /**
   * Indica se deve alinhar o header da tabela como as colunas.
   * 
   * @param alignTableHeader verdadeiro para alinhar. Se falso, centraliza.
   */
  public void setAlignTableHeader(boolean alignTableHeader) {
    this.alignTableHeader = alignTableHeader;
  }

  /**
   * Obtm o tipo de quebra de linha para os valores da tabela
   * 
   * @return tipo de quebra de linha
   */
  public String getLineWrap() {
    return lineWrap;
  }

  /**
   * Seta o tipo de quebra de linha para os valores da tabela
   * 
   * @param lineWrap tipo de quebra de linha
   */
  public void setLineWrap(String lineWrap) {
    this.lineWrap = lineWrap;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initPrinting(PrintConfiguration configuration) {
    printing = false;
    simulating = false;
    this.configuration = configuration;
    breakTable = configuration.isTableBreak();
    if (!changeCellFont) {
      return;
    }
    for (int row = 0; row < rowHeight.length; row++) {
      rowHeight[row] = 0;
    }
    if (!changeTextFont) {
      return;
    }
    float fontRate = configuration.getFontRate();
    if (posTextFont != null) {
      modifiedPosTextFont =
        posTextFont.deriveFont(posTextFont.getSize() * fontRate);
    }
    if (headerFont != null) {
      modifiedHeaderFont =
        headerFont.deriveFont(headerFont.getSize() * fontRate);
    }
    if (titleFont != null) {
      modifiedTitleFont = titleFont.deriveFont(titleFont.getSize() * fontRate);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public float getWidth() {
    return tableWidth;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setRowColor(int row, int startCol, int endCol, Color color) {
    for (int i = startCol; i <= endCol; i++) {
      cellBackground[row][i] = color;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setColumnColor(int column, int startRow, int endRow, Color color) {
    for (int i = startRow; i <= endRow; i++) {
      cellBackground[i][column] = color;
    }

  }
}
