package tecgraf.javautils.gui.table;

import java.awt.Component;
import java.util.List;

import javax.swing.ListSelectionModel;
import javax.swing.SortOrder;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import tecgraf.javautils.gui.table.SortableTable;

/**
 * <p>
 * Facilita a construo de tabelas de objetos.
 * </p>
 * <p>
 * Voc pode utilizar uma nica instncia desta classe para criar vrias
 * tabelas, fazendo com que todas tenham a mesma configurao.
 * </p>
 * <p>
 * Para criar uma tabela voc precisa de duas coisas:
 * <ul>
 * <li>os modelos das colunas que podem ser criados implementando
 * {@link IColumn} ou estendendo {@link AbstractColumn}.</li>
 * <li>uma lista com os objetos que devero aparecer na tabela.</li>
 * </ul>
 * </p>
 * Exemplo:<br>
 * <code>
 * ...
 * IColumn<Person>[] columns = new IColumns<Person>[]{
 *   new NameColumn(),
 *   new PhoneColumn(),
 *   new AddressColumn()
 * };
 * ObjectTableBuilder builder = new ObjectTableBuilder(columns);
 * ... 
 * List<Person> phonebook = new ArrayList<Person>();
 * ...
 * SortableTable table = builder.build(phonebook);
 * ...
 * </code> <br>
 * Existem tambm alguns mtodos no builder que facilitam a personalizao da
 * tabela, veja os mtodos {@link #setColumnsWidthPolicy(ColumnsWidthPolicy)},
 * {@link #setSelectionMode(SelectionMode)} e
 * {@link #setSortOrder(int, SortOrder)}. </p>
 * <p>
 * Essa classe pode ser estendida para ser o construtor de outro tipo de tabela
 * que herde de {@link SortableTable}. Para isso, basta sobrescrever o mtodo
 * {@link #createTable(TableModel)}.
 * </p>
 * 
 * @param <R> Tipo dos objetos a serem representados na tabela.
 * 
 * @see ObjectTableBuilderSample
 * 
 * @author Tecgraf
 */
public class ObjectTableBuilder<R> {

  /**
   * Modo de seleo das linhas da tabela.
   * 
   * @author Tecgraf
   */
  public enum SelectionMode {
    /** Nenhuma linha  selecionvel. */
    NO_SELECTION,
    /** Permite selecionar apenas uma linha por vez. */
    SINGLE_SELECTION,
    /** Permite selecionar um intervalo de linhas. */
    SINGLE_INTERVAL_SELECTION,
    /** Permite selecionar mltiplos intervalos de linhas. */
    MULTIPLE_INTERVAL_SELECTION
  }

  /**
   * Poltica de ajuste da largura das colunas.
   * 
   * @author Tecgraf
   */
  public enum ColumnsWidthPolicy {
    /**
     * No altera a largura das colunas.
     */
    NONE,
    /**
     * Altera, <b>no momento da criao da tabela</b>, a largura de cada coluna
     * de acordo com o tamanho de seu ttulo.
     */
    ADJUST_BY_TITLE,
    /**
     * Altera, <b>no momento da criao da tabela</b>, a largura de cada coluna
     * de acordo com o tamanho de seu ttulo e suas entradas.
     */
    ADJUST_BY_DATA,
    /**
     * Altera a largura de cada coluna de acordo com o tamanho de seu ttulo e
     * suas entradas. Mudanas nos dados da tabela poderam acarretar em mudanas
     * na largura das colunas.
     */
    ADJUST_BY_DATA_DYNAMIC,
  }

  /** Modelo das colunas da tabela na ordem em que devem aparecer. */
  private IColumn<R>[] columns;

  /**
   * Indica o modo de ajuste da largura das colunas.
   */
  private ColumnsWidthPolicy columnsWidthPolicy;

  /** Indica o modo de seleo das linhas da tabela. */
  private SelectionMode selectionMode;

  /**
   * ndice da coluna pela qual a tabela ser ordenada, se {@code sortOrder} for
   * diferente de {@link SortOrder#UNSORTED}.
   */
  private int sortColumnIndex;
  /**
   * Indica se a tabela deve ser ordenada por alguma coluna e qual o sentido da
   * ordenao, {@link SortOrder#ASCENDING crescente} ou
   * {@link SortOrder#DESCENDING decrescente}.
   */
  private SortOrder sortOrder;

  /**
   * Construtor.
   * 
   * @param columns Modelo das colunas da tabela na ordem em que devem aparecer.
   */
  public ObjectTableBuilder(final IColumn<R>... columns) {
    if (null == columns)
      throw new NullPointerException("columns can't be null");
    if (columns.length == 0)
      throw new IllegalArgumentException("columns.length == 0");

    this.columns = columns;

    this.columnsWidthPolicy = ColumnsWidthPolicy.ADJUST_BY_TITLE;
    this.selectionMode = SelectionMode.NO_SELECTION;
    this.sortColumnIndex = -1;
    this.sortOrder = SortOrder.UNSORTED;
  }

  /**
   * Atribui a poltica de ajuste da largura das colunas da tabela.<br>
   * O padro  manter a largura das colunas como foram criadas.
   * 
   * @param policy Modo de ajuste da largura das colunas. O padro 
   *        {@link ColumnsWidthPolicy#NONE no alterar} a largura das colunas.
   * @see ColumnsWidthPolicy
   */
  public void setColumnsWidthPolicy(ColumnsWidthPolicy policy) {
    if (null == policy)
      throw new NullPointerException("policy can't be null");

    this.columnsWidthPolicy = policy;
  }

  /**
   * Atribui o modo de seleo das linhas da tabela. <br>
   * O padro  as linhas da tabela no serem selecionveis.
   * 
   * @param mode modo de seleo das linhas da tabela. O padro 
   *        {@link SelectionMode#NO_SELECTION sem seleo}.
   * @see SelectionMode
   */
  public void setSelectionMode(SelectionMode mode) {
    if (null == mode)
      throw new NullPointerException("mode can't be null");

    this.selectionMode = mode;
  }

  /**
   * Atribui o modo de ordenao da tabela. Se {@code sortOrder} for igual a
   * {@link SortOrder#UNSORTED} a tabela no ser ordenada.<br>
   * O padro  uma tabela inicialmente desordenada.
   * 
   * @param columnIndex ndice da coluna.
   * @param order Sentido da ordenao, {@link SortOrder#ASCENDING crescente} ou
   *        {@link SortOrder#DESCENDING decrescente}.
   */
  public void setSortOrder(int columnIndex, SortOrder order) {
    if (columnIndex < 0 || columnIndex >= columns.length)
      throw new ArrayIndexOutOfBoundsException(columnIndex);
    if (null == order)
      throw new NullPointerException("order can't be null");

    this.sortColumnIndex = columnIndex;
    this.sortOrder = order;
  }

  /**
   * Constroi a tabela.
   * 
   * @param objs Objetos que sero mostrados na tabela.
   * 
   * @return uma tabela ordenada para objetos do tipo <R>.
   */
  public SortableTable build(List<R> objs) {
    TableModel model = new ColumnsObjectTableModel<R>(objs, columns);
    SortableTable table = createTable(model);
    /*
     * Cria o renderizador e o editor de cada coluna.
     */
    setCellRendererAndEditor(table);
    /*
     * Atribui o tipo de seleo das linhas da tabela.
     */
    setSelectionMode(table);
    /*
     * Ajusta a largura das colunas de acordo com a poltica definida.
     */
    setColumnsWidthPolicy(table);
    /*
     * Ordena os dados da tabela.
     */
    sortTable(table);

    return table;
  }

  /**
   * <p>
   * Cria uma tabela do tipo {@link SortableTable} dado o modelo.
   * </p>
   * <p>
   * Esse mtodo {@code protected} permite que esse {@link ObjectTableBuilder
   * construtor} seja estendido para criar outro tipo de tabela que herde de
   * {@link SortableTable}.
   * </p>
   * 
   * @param model modelo da tabela.
   * 
   * @return uma nova tabela.
   */
  protected SortableTable createTable(TableModel model) {
    return new SortableTable(model);
  }  
  
  /**
   * Ajusta a largura das colunas de uma dada tabela de acordo com a poltica
   * definida em {@link #setColumnsWidthPolicy(ColumnsWidthPolicy)}.
   * 
   * @param table tabela a ser configurada.
   */
  private void setColumnsWidthPolicy(final SortableTable table) {
    switch (columnsWidthPolicy) {
      case ADJUST_BY_TITLE:
        adjustColumnsWidth(table, false);
        break;
      case ADJUST_BY_DATA_DYNAMIC:
        table.getModel().addTableModelListener(new TableModelListener() {
          @Override
          public void tableChanged(TableModelEvent e) {
            adjustColumnsWidth(table, true);
          }
        });
        // no tem break por que ele acumula a lgica do modo PACK_BY_DATA. 
      case ADJUST_BY_DATA:
        adjustColumnsWidth(table, true);
        break;
      default: // FIXED_WIDTH 
        break;
    }
  }

  /**
   * Ajusta a largura das colunas de uma dada tabela.
   * 
   * @param table tabela a ter a menor largura da coluna obtida.
   * @param adjustByData se <tt>false</tt>, ajusta a largura das colunas levando
   *        em conta apenas seus ttulos; se <tt>true</tt> considera tambm seus
   *        dados.
   */
  private void adjustColumnsWidth(SortableTable table, boolean adjustByData) {
    // Obtm o renderizador dos ttulos da tabela.
    TableColumnModel columnModel = table.getColumnModel();
    TableCellRenderer headerRenderer =
      table.getTableHeader().getDefaultRenderer();

    for (int colInx = 0; colInx < table.getColumnCount(); colInx++) {
      String columnName = table.getColumnName(colInx);
      /*
       * Obtm a largura do ttulo de uma determinada coluna como a largura
       * preferencial para aquela coluna.
       */
      Component header =
        headerRenderer.getTableCellRendererComponent(table, columnName, false,
          false, 0, colInx);
      // maxWidth armazena a largura preferencial da coluna.  
      int prefWidth = header.getPreferredSize().width;

      /*
       * Verifica se deve-se considerar o tamanho de todos os dados, no s do
       * ttulo.
       */
      if (adjustByData) {
        // Obtm o renderizador da coluna.
        TableColumn column = columnModel.getColumn(colInx);
        TableCellRenderer columnRenderer = column.getCellRenderer();
        if (null == columnRenderer) {
          columnRenderer = headerRenderer;
        }
        /*
         * Percorre as linhas da tabela obtendo a largura de cada dado em uma
         * determinada coluna e verifica se  maior que 'width', se for
         * substitui o valor de 'width' para esta. 'width' ser a nova largura
         * preferencial daquela coluna.
         */
        TableModel tableModel = table.getModel();
        for (int rowInx = 0; rowInx < tableModel.getRowCount(); rowInx++) {
          Object value = tableModel.getValueAt(rowInx, colInx);
          if (value != null) {
            Component component =
              columnRenderer.getTableCellRendererComponent(table, value, false,
                false, rowInx, colInx);
            int width = component.getPreferredSize().width;
            prefWidth = Math.max(prefWidth, width);
          }
        }
      }

      TableColumn aColumnModel = columnModel.getColumn(colInx);
      /*
       * O +1 adiciona um espao extra no width para evitar os trs pontinhos
       * (...)
       */
      aColumnModel.setPreferredWidth(prefWidth + 1);
    }
  }

  /**
   * Cria o renderizador e o editor de cada coluna.
   * 
   * @param table tabela a ser configurada.
   */
  private void setCellRendererAndEditor(SortableTable table) {
    TableColumnModel columnModel = table.getColumnModel();
    for (int inx = 0; inx < columnModel.getColumnCount(); inx++) {
      TableColumn column = columnModel.getColumn(inx);
      TableCellRenderer renderer = columns[inx].createTableCellRenderer();
      if (null != renderer) {
        column.setCellRenderer(renderer);
      }
      TableCellEditor editor = columns[inx].createTableCellEditor();
      if (null != editor) {
        column.setCellEditor(editor);
      }
    }
  }

  /**
   * Atribui o tipo de seleo das linhas da tabela.
   * 
   * @param table tabela a ser configurada.
   */
  private void setSelectionMode(SortableTable table) {
    TableColumnModel columnModel = table.getColumnModel();
    switch (selectionMode) {
      case SINGLE_SELECTION:
        table.setRowSelectionAllowed(true);
        columnModel.getSelectionModel().setSelectionMode(
          ListSelectionModel.SINGLE_SELECTION);
        break;
      case SINGLE_INTERVAL_SELECTION:
        table.setRowSelectionAllowed(true);
        columnModel.getSelectionModel().setSelectionMode(
          ListSelectionModel.SINGLE_INTERVAL_SELECTION);
        break;
      case MULTIPLE_INTERVAL_SELECTION:
        table.setRowSelectionAllowed(true);
        columnModel.getSelectionModel().setSelectionMode(
          ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
        break;
      default: // NONE
        table.setRowSelectionAllowed(false);
    }
  }

  /**
   * Ordena a tabela.
   * 
   * @param table tabela a ser configurada.
   */
  private void sortTable(SortableTable table) {
    if (sortOrder == SortOrder.UNSORTED) {
      table.sort(0, sortOrder);
    }
    else {
      // Ordena a tabela
      table.sort(sortColumnIndex, sortOrder);
    }
  }
}
