package tecgraf.javautils.gui.panel;

import java.awt.Component;
import java.awt.Container;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSeparator;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIResources;

/**
 * Painel expansvel. Consiste de dois painis: um com o cabealho (ttulo do
 * painel + boto de controle), e outro com o contedo a ser gerenciado.
 * 
 * @author Tecgraf
 */
public class ExpandablePanel extends JPanel {
  /**
   * Id do evento enviado quando o painel  aberto.
   */
  public static final String EXPAND_ACTION_ID = "expand";
  /**
   * Id do evento enviado quando o painel  fechado.
   */
  public static final String COLLAPSE_ACTION_ID = "colLapse";
  /**
   * Identao  esquerda para alinhar ao ttulo do painel.
   * <p>
   * O 12  um "nmero mgico" obtido empiricamente...
   */
  public static final int LEFT_INSET = GUIResources.BUTTON_EXPAND_ICON
    .getIconWidth() + 12;
  /**
   * Indica se o painel est expandido.
   */
  private boolean isExpanded = false;
  /**
   * Boto para controle do painel.
   */
  private JButton controlButton;
  /**
   * Ttulo do painel.
   */
  private JLabel panelTitle;
  /**
   * Listener do painel.
   */
  private ActionListener listener = null;
  /**
   * Painel com o contedo propriamente dito.
   */
  private final JPanel contentPanel;

  /**
   * Constantes para definir o posicionamento do boto de controle.
   */
  public enum Position {
    /**
     * Controle  esquerda do ttulo.
     */
    CONTROL_LEFT,
    /**
     * Controle imediatamente  direita do ttulo.
     */
    CONTROL_RIGHT,
    /**
     * Controle  direita do ttulo, junto ao limite do painel (alinhamento
     * "east").
     */
    CONTROL_RIGHTMOST,
  }

  /**
   * Torna um boto "transparente": sem borda, sem "fundo", sem realce quando
   * est com o foco e com margens internas nulas.
   * 
   * @param button boto
   * 
   * @see #makeButtonTransparent(JButton, Insets)
   */
  static void makeButtonTransparent(JButton button) {
    makeButtonTransparent(button, new Insets(0, 0, 0, 0));
  }

  /**
   * Torna um boto "transparente": sem borda, sem "fundo", e sem realce quando
   * est com o foco.
   * 
   * @param button boto
   * @param margins margens internas do boto
   * 
   * @see #makeButtonTransparent(JButton)
   */
  static void makeButtonTransparent(JButton button, Insets margins) {
    button.setBorderPainted(false);
    button.setContentAreaFilled(false);
    button.setMargin(margins);
    button.setFocusPainted(false);
  }

  /**
   * Verifica se um evento  um evento de expanso de um painel.
   * 
   * @param event evento
   * @return <code>true</code> se o evento corresponde  expanso de um painel
   */
  public static boolean isExpandEvent(ActionEvent event) {
    return event.getActionCommand().equals(EXPAND_ACTION_ID);
  }

  /**
   * Constri um painel que no possui separador, no identa seu contedo e cujo
   * <i>layout manager</i> default  o {@link GridBagLayout}.
   * 
   * @param title ttulo do painel
   * @param controlPosition posicionamento do controle
   */
  public ExpandablePanel(String title, Position controlPosition) {
    this(title, controlPosition, false, false);
  }

  /**
   * Constri um painel cujo <i>layout manager</i> default  o
   * {@link GridBagLayout}.
   * 
   * @param title ttulo do painel
   * @param controlPosition posicionamento do controle
   * @param showSeparator <code>true</code> para exibir um separador acima do
   *        ttulo
   * @param identLeft <code>true</code> para identar o contedo de forma a
   *        alinh-lo com o ttulo (til para posicionamento
   *        {@link Position#CONTROL_LEFT})
   */
  public ExpandablePanel(String title, Position controlPosition,
    boolean showSeparator, boolean identLeft) {
    this(title, showSeparator, controlPosition, identLeft, new GridBagLayout());
  }

  /**
   * Construtor.
   * 
   * @param title ttulo do painel
   * @param controlPosition posicionamento do controle
   * @param showSeparator <code>true</code> para exibir um separador acima do
   *        ttulo
   * @param identLeft <code>true</code> para identar o contedo de forma a
   *        alinh-lo com o ttulo (til para posicionamento
   *        {@link Position#CONTROL_LEFT})
   * @param layout <i>layout manager</i>
   */
  public ExpandablePanel(String title, boolean showSeparator,
    Position controlPosition, boolean identLeft, LayoutManager layout) {
    super.setLayout(new GridBagLayout());
    contentPanel = new JPanel(layout);
    contentPanel.setVisible(isExpanded);
    controlButton =
      new JButton(new AbstractAction(null, GUIResources.BUTTON_EXPAND_ICON) {

        @Override
        public void actionPerformed(ActionEvent e) {
          isExpanded = !isExpanded;
          stateChanged(true, true);
        }
      });
    makeButtonTransparent(controlButton);
    JPanel headerPanel = new JPanel(new GridBagLayout());
    int vOffset = 0;
    if (showSeparator) {
      headerPanel.add(new JSeparator(), new GBC(0, vOffset++).horizontal()
        .bottom(3).width(2));
    }
    panelTitle = new JLabel(title);
    switch (controlPosition) {
      case CONTROL_LEFT:
        headerPanel.add(controlButton, new GBC(0, vOffset).right(5));
        headerPanel.add(panelTitle, new GBC(1, vOffset++).west().horizontal());
        break;

      case CONTROL_RIGHTMOST:
        headerPanel.add(panelTitle, new GBC(0, vOffset).west().horizontal());
        headerPanel.add(controlButton, new GBC(1, vOffset++).east().right(0)
          .left(5));
        break;

      case CONTROL_RIGHT:
        headerPanel.add(panelTitle, new GBC(0, vOffset).west());
        headerPanel.add(controlButton, new GBC(1, vOffset++).west().right(0)
          .left(5).pushx());
        break;

      default:
        throw new AssertionError("posicionamento desconhecido: "
          + controlPosition.name());
    }
    /*
     * temos que chamar super.addImp() aqui, seno adicionaramos no
     * contentPanel
     */
    super.addImpl(headerPanel, new GBC(0, 0).northwest().horizontal(), -1);
    GBC gbc = new GBC(0, 1).northwest().both();
    if (identLeft) {
      gbc.left(LEFT_INSET);
    }
    super.addImpl(contentPanel, gbc, -1);
  }

  /**
   * Obtm o estado do painel.
   * 
   * @return <code>true</code> se o painel est expandido
   */
  public boolean isExpanded() {
    return isExpanded;
  }

  /**
   * Define o estado do painel.
   * 
   * @param expanded <code>true</code> para expandir o painel,
   *        <code>false</code> para fech-lo
   * @param notifyParent <code>true</code> para revalidar o pai do painel. Usar
   *        <code>false</code> quando esta operao fizer parte de um lote, onde
   *        o pai s deve ser revalidado aps a ltima operao (e no a cada
   *        operao)
   * @param notifyListener <code>true</code> se os listeneres devem ser
   *        notificados
   * @return <code>true</code> se o estado foi alterado, <code>false</code> se o
   *         painel j estava no estado solicitado
   * 
   * @see #setExpanded(boolean)
   */
  public boolean setExpanded(boolean expanded, boolean notifyParent,
    boolean notifyListener) {
    if (expanded != isExpanded) {
      isExpanded = expanded;
      stateChanged(notifyParent, notifyListener);
      return true;
    }
    return false;
  }

  /**
   * Define o estado do painel, revalidando o seu pai.
   * 
   * @param expanded <code>true</code> para expandir o painel,
   *        <code>false</code> para fech-lo
   * @return <code>true</code> se o estado foi alterado, <code>false</code> se o
   *         painel j estava no estado solicitado
   * 
   * @see #setExpanded(boolean, boolean, boolean)
   */
  public boolean setExpanded(boolean expanded) {
    return setExpanded(expanded, true, true);
  }

  /**
   * Altera o estado do painel.
   * 
   * @param notifyParent <code>true</code> para revalidar o pai do painel. Usar
   *        <code>false</code> quando esta operao fizer parte de um lote, onde
   *        o pai s deve ser revalidado aps a ltima operao (e no a cada
   *        operao)
   * @param notifyListener <code>true</code> para notificar os listeners
   */
  private void stateChanged(boolean notifyParent, boolean notifyListener) {
    contentPanel.setVisible(isExpanded);
    controlButton.setIcon(isExpanded ? GUIResources.BUTTON_COLLAPSE_ICON
      : GUIResources.BUTTON_EXPAND_ICON);
    if (getParent() != null) {
      /*
       * avisamos ao pai que o painel mudou de estado
       */
      adjustParent(notifyParent);
      if (notifyListener && listener != null) {
        /*
         * notificamos o listener
         */
        listener.actionPerformed(new ActionEvent(this, (int) System
          .currentTimeMillis(), isExpanded ? EXPAND_ACTION_ID
          : COLLAPSE_ACTION_ID));
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void addImpl(Component comp, Object constraints, int index) {
    /*
     * basta redefinirmos este mtodo, pois todos os add(...) o chamam
     */
    contentPanel.add(comp, constraints, index);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void remove(int index) {
    contentPanel.remove(index);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void remove(Component comp) {
    contentPanel.remove(comp);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeAll() {
    contentPanel.removeAll();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setLayout(LayoutManager layout) {
    if (contentPanel == null) {
      return;
    }
    contentPanel.setLayout(layout);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setEnabled(boolean enabled) {
    controlButton.setEnabled(enabled);
    panelTitle.setEnabled(enabled);
    for (Component child : contentPanel.getComponents()) {
      child.setEnabled(enabled);
    }
  }

  /**
   * Redefine o ttulo do painel.
   * 
   * @param title ttulo
   * @return o prprio painel, para encadeamento
   */
  public ExpandablePanel setTitle(String title) {
    panelTitle.setText(title);
    adjustParent(true);
    return this;
  }

  /**
   * Ajusta o pai aps mudanas no painel.
   * 
   * @param notifyParent <code>true</code> para notificar o pai,
   *        <code>false</code> para apenas revalidar o painel
   */
  private void adjustParent(boolean notifyParent) {
    Container parent = getParent();
    if (parent != null && parent.isShowing()) {
      revalidate();
      if (notifyParent) {
        parent.repaint();
      }
    }
  }

  /**
   * Define um listener para mudanas de estado do painel.
   * 
   * @param newListener listener. Se for <code>null</code>, remove o listener
   *        corrente
   * @return listener cadastrado anteriormente (ou <code>null</code> se no
   *         havia um listener)
   */
  public ActionListener setListener(ActionListener newListener) {
    ActionListener oldListener = listener;
    listener = newListener;
    return oldListener;
  }

  /**
   * Remove o listener. Equivale a {@link #setListener(ActionListener)} com
   * parmetro <code>null</code>.
   * 
   * @return listener cadastrado anteriormente (ou <code>null</code> se no
   *         havia um listener)
   */
  public ActionListener removeListener() {
    return setListener(null);
  }
}
