/*
 * $Id$
 */
package tecgraf.javautils.gui.print;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.print.PageFormat;
import java.awt.print.Paper;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;
import java.util.List;

/**
 * Implementao padro do relatrio a ser impresso.
 * 
 * @see DefaultPrintableReportSample
 */
public class DefaultPrintableReport implements PrintableReport {

  /** Margem vertical, em centmetros */
  private double verticalMargin;
  /** Margem horizontal, em centmetros */
  private double horizontalMargin;
  /** Orientao do papel */
  private int orientation;
  /** Texto que deve ser impresso em todas as pginas do relatrio no topo */
  protected PrintableReportItem header;
  /** Lista dos item a serem impressos no relatrio sequencialmente */
  protected List<PrintableReportItem> printables;
  /** Texto que deve ser impresso em todas as pginas do relatrio no inferior */
  protected PrintableReportItem footer;
  /** ltima pgina impressa */
  private int currentPage;
  /** ltimo item impresso */
  private int nextPrintable;
  /** Antigo ltimo item impresso */
  private int oldLastPrinted;
  /** Indica se o ltimo item foi totalmente impresso */
  private boolean fullPrinted;
  /** Antigo indicador se o ltimo item foi totalmente impresso */
  private boolean oldFullPrinted;
  /** Configurao de impresso dos itens */
  private PrintConfiguration configuration;

  /**
   * Construtor.
   * 
   * @param printables itens a serem impressos.
   */
  public DefaultPrintableReport(List<PrintableReportItem> printables) {
    this.printables = printables;
    this.horizontalMargin = 2.5;
    this.verticalMargin = 1.5;
    this.orientation = PageFormat.PORTRAIT;
    this.configuration = new PrintConfiguration();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setHorizontalMargin(double horizontalMargin) {
    this.horizontalMargin = horizontalMargin;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setVerticalMargin(double verticalMargin) {
    this.verticalMargin = verticalMargin;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setOrientation(int orientation) {
    this.orientation = orientation;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setHeader(PrintableReportItem header) {
    this.header = header;
    if (header != null) {
      header.initPrinting(configuration);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setPrintables(List<PrintableReportItem> printables) {
    this.printables = printables;
    for (PrintableReportItem item : printables) {
      item.initPrinting(configuration);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setFooter(PrintableReportItem footer) {
    this.footer = footer;
    if (footer != null) {
      footer.initPrinting(configuration);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getCurrentPage() {
    return currentPage;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int print(Graphics graphics, PageFormat pageFormat, int pageIndex)
    throws PrinterException {
    if (pageIndex == 0) {
      this.oldLastPrinted = 0;
      this.currentPage = 0;
      this.oldFullPrinted = false;
    }
    /* Est imprimindo uma pgina mais de uma vez */
    if (pageIndex == currentPage) {
      nextPrintable = oldLastPrinted;
      fullPrinted = oldFullPrinted;
    }

    if (fullPrinted && (nextPrintable >= (printables.size() - 1))) {
      return NO_SUCH_PAGE;
    }

    currentPage = pageIndex;
    oldLastPrinted = nextPrintable;
    oldFullPrinted = fullPrinted;

    Graphics2D g2 = (Graphics2D) graphics;
    g2.setColor(Color.BLACK);

    PageFormat printableFormat = (PageFormat) pageFormat.clone();
    if (header != null) {
      header.print(g2, printableFormat, pageIndex);
      adjustPrintable(printableFormat, header.getHeight(), true);
    }

    if (footer != null) {
      printFooter(g2, printableFormat, pageIndex, true);
    }

    for (int i = nextPrintable; i < printables.size(); i++) {
      nextPrintable = i;
      PrintableReportItem printable = printables.get(i);
      g2.setColor(Color.BLACK);
      fullPrinted = printable.print(g2, printableFormat, pageIndex);
      if (!fullPrinted) {
        break;
      }
      if (!adjustPrintable(printableFormat, printable.getHeight(), true)) {
        if ((i + 1) < printables.size()) {
          nextPrintable =
            (printables.get(i + 1) instanceof PageSeparator) ? i + 2 : i + 1;
        }
        break;
      }
    }

    return PAGE_EXISTS;
  }

  /**
   * Imprime o footer da pgina.
   * 
   * @param g2 contexto grfico.
   * @param format formato da pgina.
   * @param pageIndex ndice da pgina.
   * @param print indica se o footer deve ser impresso (<code>true</code>) ou se
   *        o mesmo somente ter sua impresso simulada (<code>false</code>).
   */
  protected void printFooter(Graphics2D g2, PageFormat format, int pageIndex,
    boolean print) {
    footer.simulatePrint(g2, format, pageIndex);

    double footerHeight = footer.getHeight();

    PageFormat footerFormat = (PageFormat) format.clone();
    Paper paper = footerFormat.getPaper();
    if (format.getOrientation() == PageFormat.PORTRAIT) {
      paper.setImageableArea(paper.getImageableX(), paper.getImageableY()
        + paper.getImageableHeight() - footerHeight, paper.getImageableWidth(),
        footerHeight);
    }
    else if (format.getOrientation() == PageFormat.LANDSCAPE) {
      paper.setImageableArea(paper.getImageableX() + paper.getImageableWidth()
        - footerHeight, paper.getImageableY(), footerHeight, paper
        .getImageableHeight());
    }
    footerFormat.setPaper(paper);

    if (print) {
      footer.print(g2, footerFormat, pageIndex);
    }
    adjustPrintable(format, (float) footerHeight, false);
  }

  /**
   * Ajusta o formato do pgina, reduzindo a altura do item impresso. O papel
   * sempre  visto na vertical, independente do formato da pgina.
   * 
   * @param format formato do papel a ser ajustado.
   * @param printableHeight altura do item impresso.
   * @param adjustY indica se deve ajustar a coordenada vertical.
   * 
   * @return verdadeiro se a pgina continua vlida.
   */
  protected boolean adjustPrintable(PageFormat format, float printableHeight,
    boolean adjustY) {
    Paper paper = format.getPaper();
    double paperHeight =
      (format.getOrientation() == PageFormat.PORTRAIT) ? paper
        .getImageableHeight() : paper.getImageableWidth();
    double availableHeight = paperHeight - printableHeight;
    if (availableHeight <= 0.1) {
      return false;
    }
    if (format.getOrientation() == PageFormat.PORTRAIT) {
      paper.setImageableArea(paper.getImageableX(), paper.getImageableY()
        + (adjustY ? printableHeight : 0), paper.getImageableWidth(), paper
        .getImageableHeight()
        - printableHeight);
    }
    else if (format.getOrientation() == PageFormat.LANDSCAPE) {
      paper.setImageableArea(paper.getImageableX()
        + (adjustY ? printableHeight : 0), paper.getImageableY(), paper
        .getImageableWidth()
        - printableHeight, paper.getImageableHeight());
    }
    format.setPaper(paper);
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getNumberOfPages() {
    return UNKNOWN_NUMBER_OF_PAGES;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PageFormat getPageFormat(int pageIndex)
    throws IndexOutOfBoundsException {
    PageFormat pageFormat = PrinterJob.getPrinterJob().defaultPage();
    Paper paper = pageFormat.getPaper();
    double hMargin = horizontalMargin * PrintableReportItem.CM_TO_INCH;
    double vMargin = verticalMargin * PrintableReportItem.CM_TO_INCH;
    paper.setImageableArea(hMargin, vMargin, paper.getWidth() - (2 * hMargin),
      paper.getHeight() - (2 * vMargin));
    pageFormat.setPaper(paper);
    pageFormat.setOrientation(orientation);
    return PrinterJob.getPrinterJob().validatePage(pageFormat);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Printable getPrintable(int pageIndex) throws IndexOutOfBoundsException {
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initPrinting(PrintConfiguration configuration) {
    this.configuration = configuration;
    if (header != null) {
      header.initPrinting(configuration);
    }
    if (footer != null) {
      footer.initPrinting(configuration);
    }
    for (PrintableReportItem item : printables) {
      item.initPrinting(configuration);
    }
  }
}
