package tecgraf.javautils.gui;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.util.Timer;
import java.util.TimerTask;

import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;

/**
 * Representa uma barra de status a ser utilizada em interfaces de usurio. Obs.
 * Uma barra de status  apenas um label com borda.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class StatusBar extends JPanel {

  /**
   * String usada para limpar a barra de status
   */
  private static final String _CLEAR_STATUS_STRING = " ";

  /**
   * Temporizador usado para limpar a barra de status
   */
  private Timer clearTimer;

  /**
   * Tarefa do temporizador que efetivamente limpa a barra de status
   */
  private TimerTask clearTask;

  /**
   * Label
   */
  final private JLabel label = new JLabel();

  /**
   * ltimo ndice usado (coordenada X)
   */
  private int lastX = 0;

  /**
   * Construtor
   */
  public StatusBar() {
    super();
    super.setVisible(true);
    setLayout(new GridBagLayout());
    addComponent(label, 1.0);
    label.setFont(label.getFont().deriveFont(Font.PLAIN));
    label.setText(" ");
    clearTimer = new Timer();
    clearTask = null;
    hideStatusBar();
  }

  /**
   * Adio de componente
   * 
   * @param component component
   * @param weightX peso de expanso em X (normalizado com 1 do label de texto).
   */
  public void addComponent(JComponent component, double weightX) {
    add(component, new GBC(lastX++, 0).horizontal(weightX).insets(0, 1, 0, 0));
    final Dimension ldim = label.getPreferredSize();
    final int hcurr = (int) ldim.getHeight();
    final int hcmp = (int) component.getPreferredSize().getHeight();
    final int height = Math.max(hcurr, hcmp);
    final Dimension dim = new Dimension((int) ldim.getWidth(), height + 2);
    label.setMinimumSize(dim);
  }

  /**
   * Faz com que o componente seja visvel ou invisvel.
   * 
   * @param visible indicativo de visibilidade.
   * 
   * @deprecated use {@link #showStatusBar()} ou {@link #hideStatusBar()}
   * @see #showStatusBar()
   * @see #hideStatusBar()
   */
  @Override
  @Deprecated
  public void setVisible(boolean visible) {
    super.setVisible(visible);
  }

  /**
   * Mostra a barra de status se a condio dada for verdadeira.
   * 
   * @param condition condio.
   */
  public void showStatusBar(final boolean condition) {
    setStatus(condition && isHidden(), _CLEAR_STATUS_STRING);
  }

  /**
   * Mostra a barra de status.
   */
  public void showStatusBar() {
    setStatus(true, _CLEAR_STATUS_STRING);
  }

  /**
   * Esconde a barra de status se a condio dada for verdadeira.
   * 
   * @param condition condio.
   */
  public void hideStatusBar(final boolean condition) {
    setStatus(condition, null);
  }

  /**
   * Esconde a barra de status.
   */
  public void hideStatusBar() {
    setStatus(true, null);
  }

  /**
   * @return true se a barra de status estiver escondida.
   */
  public boolean isHidden() {
    final String text = label.getText();
    return text == null || text.equals("");
  }

  /**
   * Limpa a barra de status se a condio dada for verdadeira. Obs. Esse mtodo
   * no faz com que a barra de status aparea caso ela esteja escondida.<br>
   * 
   * @param condition condio.
   */
  public void clearStatus(final boolean condition) {
    setStatus(condition && !isHidden(), _CLEAR_STATUS_STRING);
  }

  /**
   * Limpa a barra de status.
   */
  public void clearStatus() {
    clearStatus(true);
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone se a
   * condio dada for verdadeira e com um timeout para que a mensagem
   * desaparea.<br>
   * Obs. caso a barra de status estivesse escondida, ela ir aparecer para
   * apresentar a mensagem.<br>
   * Obs. se a mensagem for vazia (<code>""</code>) ou nula (<code>null</code>)
   * e se o cone for nulo (<code>null</code>), ser o mesmo que chamar
   * {@link #hideStatusBar()}.
   * 
   * @param condition condio.
   * @param icon cone
   * @param message mensagem
   * @param timeoutSecs timeout (em segundos)
   */
  public void setStatus(final boolean condition, final ImageIcon icon,
    final String message, final int timeoutSecs) {
    if (this.clearTask != null) {
      this.clearTask.cancel();
    }
    String msg = message;
    if (message == null) {
      msg = " --- ";
    }
    if (condition) {
      label.setIcon(icon);
      label.setText(message);
    }
    if (timeoutSecs > 0 && !msg.equals(_CLEAR_STATUS_STRING)) {
      clearStatusIn(timeoutSecs);
    }
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone se a
   * condio dada for verdadeira.<br>
   * Obs. caso a barra de status estivesse escondida, ela ir aparecer para
   * apresentar a mensagem.<br>
   * Obs. se a mensagem for vazia (<code>""</code>) ou nula (<code>null</code>)
   * e se o cone for nulo (<code>null</code>), ser o mesmo que chamar
   * {@link #hideStatusBar()}.
   * 
   * @param condition condio.
   * @param icon cone
   * @param message mensagem
   */
  public void setStatus(final boolean condition, final ImageIcon icon,
    final String message) {
    this.setStatus(condition, icon, message, 0);
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone. Obs. caso a
   * barra de status estivesse escondida, ela ir aparecer para apresentar a
   * mensagem.<br>
   * Obs. se a mensagem for vazia (<code>""</code>) ou nula (<code>null</code>)
   * e se o cone for nulo (<code>null</code>), ser o mesmo que chamar
   * {@link #hideStatusBar()}.
   * 
   * @param icon cone
   * @param message mensagem
   */
  public void setStatus(final ImageIcon icon, final String message) {
    this.setStatus(true, icon, message, 0);
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone e com um
   * timeout para que ela desaparea. Obs. caso a barra de status estivesse
   * escondida, ela ir aparecer para apresentar a mensagem.<br>
   * Obs. se a mensagem for vazia (<code>""</code>) ou nula (<code>null</code>)
   * e se o cone for nulo (<code>null</code>), ser o mesmo que chamar
   * {@link #hideStatusBar()}.
   * 
   * @param icon cone
   * @param message mensagem
   * @param timeoutSecs timeout (em segundos)
   */
  public void setStatus(final ImageIcon icon, final String message,
    final int timeoutSecs) {
    this.setStatus(true, icon, message, timeoutSecs);
  }

  /**
   * Escreve uma mensagem na barra de status se a condio dada for verdadeira.
   * Obs. caso a barra de status estivesse escondida, ela ir aparecer para
   * apresentar a mensagem.<br>
   * Obs. se a mensagem for vazia (<code>""</code>) ou nula (<code>null</code>)
   * e se o cone for nulo (<code>null</code>), ser o mesmo que chamar
   * {@link #hideStatusBar()}.
   * 
   * @param condition condio.
   * @param message mensagem.
   */
  public void setStatus(final boolean condition, final String message) {
    this.setStatus(condition, null, message, 0);
  }

  /**
   * Escreve uma mensagem na barra de status se a condio dada for verdadeira e
   * com um timeout para que a mensagem desaparea. Obs. caso a barra de status
   * estivesse escondida, ela ir aparecer para apresentar a mensagem.<br>
   * Obs. se a mensagem for vazia (<code>""</code>) ou nula (<code>null</code>)
   * e se o cone for nulo (<code>null</code>), ser o mesmo que chamar
   * {@link #hideStatusBar()}.
   * 
   * @param condition condio.
   * @param message mensagem.
   * @param timeoutSecs timeout (em segundos)
   */
  public void setStatus(final boolean condition, final String message,
    final int timeoutSecs) {
    this.setStatus(condition, null, message, timeoutSecs);
  }

  /**
   * Escreve uma mensagem na barra de status. Obs. caso a barra de status
   * estivesse escondida, ela ir aparecer para apresentar a mensagem.<br>
   * Obs. se a mensagem for vazia (<code>""</code>) ou nula (<code>null</code>)
   * e se o cone for nulo (<code>null</code>), ser o mesmo que chamar
   * {@link #hideStatusBar()}.
   * 
   * @param message mensagem.
   */
  public void setStatus(final String message) {
    this.setStatus(true, message);
  }

  /**
   * Escreve uma mensagem na barra de status com um timeout para que a mensagem
   * desaparea. Obs. caso a barra de status estivesse escondida, ela ir
   * aparecer para apresentar a mensagem.<br>
   * Obs. se a mensagem for vazia (<code>""</code>) ou nula (<code>null</code>)
   * e se o cone for nulo (<code>null</code>), ser o mesmo que chamar
   * {@link #hideStatusBar()}.
   * 
   * @param message mensagem.
   * @param timeoutSecs timeout (em segundos)
   */
  public void setStatus(final String message, final int timeoutSecs) {
    this.setStatus(true, message, timeoutSecs);
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone de
   * informao se a condio dada for verdadeira. Obs. caso a barra de status
   * estivesse escondida, ela ir aparecer para apresentar a mensagem.<br>
   * 
   * @param condition condio.
   * @param message mensagem.
   */
  public void setInfo(final boolean condition, final String message) {
    setStatus(GUIResources.STATUS_INFO_ICON, message);
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone de
   * informao se a condio dada for verdadeira e com um timeout para que a
   * mensagem desaparea. Obs. caso a barra de status estivesse escondida, ela
   * ir aparecer para apresentar a mensagem.<br>
   * 
   * @param condition condio.
   * @param message mensagem.
   * @param timeoutSecs timeout (em segundos)
   */
  public void setInfo(final boolean condition, final String message,
    final int timeoutSecs) {
    setStatus(GUIResources.STATUS_INFO_ICON, message, timeoutSecs);
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone de
   * informao. Obs. caso a barra de status estivesse escondida, ela ir
   * aparecer para apresentar a mensagem.<br>
   * 
   * @param message mensagem.
   */
  public void setInfo(final String message) {
    setInfo(true, message);
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone de
   * informao, com um timeout para que a mensagem desaparea. Obs. caso a
   * barra de status estivesse escondida, ela ir aparecer para apresentar a
   * mensagem.<br>
   * 
   * @param message mensagem.
   * @param timeoutSecs timeout (em segundos)
   */
  public void setInfo(final String message, final int timeoutSecs) {
    setInfo(true, message, timeoutSecs);
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone de aviso se
   * a condio dada for verdadeira. Obs. caso a barra de status estivesse
   * escondida, ela ir aparecer para apresentar a mensagem.<br>
   * 
   * @param condition condio.
   * @param message mensagem.
   */
  public void setWarning(final boolean condition, final String message) {
    setStatus(condition, GUIResources.STATUS_WARNING_ICON, message, 0);
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone de aviso se
   * a condio dada for verdadeira e com um timeout para que a mensagem
   * desaparea. Obs. caso a barra de status estivesse escondida, ela ir
   * aparecer para apresentar a mensagem.<br>
   * 
   * @param condition condio.
   * @param message mensagem.
   * @param timeoutSecs timeout (em segundos)
   */
  public void setWarning(final boolean condition, final String message,
    final int timeoutSecs) {
    setStatus(condition, GUIResources.STATUS_WARNING_ICON, message, timeoutSecs);
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone de aviso.
   * Obs. caso a barra de status estivesse escondida, ela ir aparecer para
   * apresentar a mensagem.<br>
   * 
   * @param message mensagem.
   */
  public void setWarning(final String message) {
    setWarning(true, message);
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone de aviso,
   * com um timeout para que a mensagem desaparea. Obs. caso a barra de status
   * estivesse escondida, ela ir aparecer para apresentar a mensagem.<br>
   * 
   * @param message mensagem.
   * @param timeoutSecs timeout (em segundos)
   */
  public void setWarning(final String message, final int timeoutSecs) {
    setWarning(true, message, timeoutSecs);
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone de erro se a
   * condio dada for verdadeira. Obs. caso a barra de status estivesse
   * escondida, ela ir aparecer para apresentar a mensagem.<br>
   * 
   * @param condition condio.
   * @param message mensagem.
   */
  public void setError(final boolean condition, final String message) {
    setStatus(condition, GUIResources.STATUS_ERROR_ICON, message, 0);
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone de erro se a
   * condio dada for verdadeira e com um timeout para que a mensagem
   * desaparea. Obs. caso a barra de status estivesse escondida, ela ir
   * aparecer para apresentar a mensagem.<br>
   * 
   * @param condition condio.
   * @param message mensagem.
   * @param timeoutSecs timeout (em segundos)
   */
  public void setError(final boolean condition, final String message,
    final int timeoutSecs) {
    setStatus(condition, GUIResources.STATUS_ERROR_ICON, message, timeoutSecs);
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone de erro.
   * Obs. caso a barra de status estivesse escondida, ela ir aparecer para
   * apresentar a mensagem.<br>
   * 
   * @param message mensagem.
   */
  public void setError(final String message) {
    setError(true, message);
  }

  /**
   * Escreve uma mensagem na barra de status precedida por um cone de erro, com
   * um timeout para que a mensagem desaparea. Obs. caso a barra de status
   * estivesse escondida, ela ir aparecer para apresentar a mensagem.<br>
   * 
   * @param message mensagem.
   * @param timeoutSecs timeout (em segundos)
   */
  public void setError(final String message, final int timeoutSecs) {
    setError(true, message, timeoutSecs);
  }

  /**
   * Limpa a barra de status aps timeoutSecs, utilizando uma thread em
   * background (Timer e TimerTask)
   * 
   * @param timeoutSecs timeout (em segundos)
   */
  private void clearStatusIn(int timeoutSecs) {
    this.clearTask = new TimerTask() {
      @Override
      public void run() {
        clearStatus();
      }
    };

    clearTimer.schedule(clearTask, timeoutSecs * 1000);
  }

  /**
   * Cancela tasks de atualizao da statusbar que ainda estejam pendentes.
   * <p>
   * <b>Este mtodo s pode ser chamado quando no se pretender mais usar o
   * timer (p.ex. quando a tela for fechada), pois no ser possvel criar novas
   * tasks aps sua execuo.</b>
   */
  public void shutdownTimer() {
    clearTimer.cancel();
  }

  /**
   * Consulta o texto da status bar.
   * 
   * @return texto
   */
  public String getText() {
    return label.getText();
  }

  /**
   * Ajusta o texto da status bar.
   * 
   * @param message a mensagem.
   */
  public void setText(String message) {
    label.setText(message);
  }
}
