/**
 * $Id$
 */
package tecgraf.javautils.gui.table;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.JComponent;
import javax.swing.JTable;
import javax.swing.JTree;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.plaf.basic.BasicTreeUI;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableModel;
import javax.swing.tree.TreeSelectionModel;

/**
 * Uma tabela que renderiza uma rvore em uma de suas colunas. A rvore ser
 * renderizada na coluna que retornar TreeTableModel.class no mtodo
 * getColumnClass no modelo.
 *
 * Os eventos do mouse so repassados para a rvore somente se ocorrerem na
 * coluna que contm a rvore, indicada pelo modelo. Esse comportamento pode ser
 * desfeito se o mtodo createMouseListenerProxy() for sobrescrito.
 *
 * Tambm  feita uma sincronizao da seleo da tabela com a rvore (a
 * recproca no  verdadeira, ento se a seleo da rvore for alterada por
 * fora, isso no refletir na seleo da tabela. Essa sincronizao pode no
 * ser configurada sobrescrevendo o mtodo createSelectionSynchronizer.
 *
 * A tabela possui uma implementao padro da rvore que  utilizada
 * internamente, mas  possvel utilizar outra, sobrescrevendo o mtodo
 * createTable. Recomenda-se que a rvore utilizada herde da classe utilizada
 * internamente pelo componente. Se isso no ocorrer,  importante observar que
 * o mtodo createTreeTableCellRenderer tambm precisar ser sobrescrito, com um
 * renderizador que utilize a rvore utilizada.
 *
 * @author Tecgraf/PUC-Rio
 */
public class TreeTable extends JTable {
  /** rvore da Tabela */
  private JTree tree;
  /** Modelo da rvore */
  private TreeTableModel treeModel;
  /** Modelo da tabela */
  private TableModel tableModel;

  /**
   * Construtor
   *
   * @param treeTableModel o modelo que ser utilizado.
   *
   */
  public TreeTable(TreeTableModel treeTableModel) {
    super();

    treeModel = treeTableModel;

    tree = createTree(treeModel);
    tableModel = createModelAdapter(tree, treeTableModel);
    super.setModel(tableModel);

    createSelectionSynchronizer();

    createMouseListenerProxy();

    // Configura renderer e editor
    setDefaultRenderer(TreeTableModel.class, createTreeTableCellRenderer(tree,
      treeTableModel));

    // Exibe o as linhas da tabela.
    setShowGrid(true);

    // D uma margem para que o texto do n da rvore no fique expremido.
    setIntercellSpacing(new Dimension(1, 1));
  }

  /**
   * Cria o renderizador da rvore a ser utilizado na coluna que contiver a
   * rvore. O comportamento padro espera que a rvore criada implemente
   * {@link TableCellRenderer}.
   *
   * @param tree a rvore utilizada no componente.
   * @param treeTableModel o modelo do componente.
   * @return o renderizador das clulas da coluna da rvore.
   */
  protected TableCellRenderer createTreeTableCellRenderer(JTree tree,
    TreeTableModel treeTableModel) {
    return (TableCellRenderer) this.tree;
  }

  /**
   * Instancia o adaptador de modelo da rvore para o modelo da tabela. O padro
   *  {@link TreeTableModelAdapter}.
   *
   * @param jtree a rvore que ser utilizada no componente.
   * @param tableTreeModel o modelo do componente.
   *
   * @return o adaptador do modelo da rvore.
   */
  protected TableModel createModelAdapter(JTree jtree,
    TreeTableModel tableTreeModel) {
    return new TreeTableModelAdapter(jtree, tableTreeModel);
  }

  /**
   * Cria a rvore que ser utilizada no componente. O padro utilizado 
   * {@link TreeTableCellRenderer}.
   *
   * @param tableTreeModel o modelo da rvore.
   * @return a rvore.
   */
  protected JTree createTree(TreeTableModel tableTreeModel) {
    // Cria a TreeTableTree.
    return new TreeTableCellRenderer(this, tableTreeModel);
  }

  /**
   * Obtm a rvore da tabela.
   *
   * @return a rvore.
   */
  public JTree getTree() {
    return tree;
  }

  /**
   * Garante que a seleo da tabela est em sincronia com a seleo da rvore.
   */
  protected void createSelectionSynchronizer() {
    getSelectionModel().addListSelectionListener(new ListSelectionListener() {
      @Override
      public void valueChanged(ListSelectionEvent e) {
        //Garante que a seleo da rvore corresponde  seleo da tabela.
        int[] rows = TreeTable.this.getSelectedRows();
        tree.setSelectionRows(rows);
      }
    });
  }

  /**
   * Cria um listener que apenas repassa os eventos de mouse para a rvore.
   */
  protected void createMouseListenerProxy() {
    addMouseListener(new MouseAdapter() {

      @Override
      public void mouseClicked(MouseEvent e) {
        copyEventToTree(e);
      }

      @Override
      public void mouseReleased(MouseEvent e) {
        copyEventToTree(e);
      }

      @Override
      public void mouseEntered(MouseEvent e) {
        copyEventToTree(e);
      }

      @Override
      public void mouseExited(MouseEvent e) {
        copyEventToTree(e);
      }

      @Override
      public void mousePressed(MouseEvent e) {
        Point p = e.getPoint();
        int rowNumber = TreeTable.this.rowAtPoint(p);
        ListSelectionModel model = TreeTable.this.getSelectionModel();
        if (e.isPopupTrigger() || SwingUtilities.isRightMouseButton(e)) {

          if (!model.isSelectedIndex(rowNumber)) {
            //garante que a linha clicada est selecionada.
            model.setSelectionInterval(rowNumber, rowNumber);
          }
        }

        copyEventToTree(e);
        //Garante que a seleo da rvore corresponde  seleo da tabela.
        int[] rows = TreeTable.this.getSelectedRows();
        tree.setSelectionRows(rows);
      }

      protected void copyEventToTree(MouseEvent e) {
        int column = columnAtPoint(e.getPoint());
        int modelColumn = convertColumnIndexToModel(column);

        if (column < 0 || modelColumn < 0) {
          //failsafe
          return;
        }
        // s repassa os eventos se o click ocorreu na coluna da rvore.
        if (tableModel.getColumnClass(modelColumn) == TreeTableModel.class) {
          // A rvore no ext sendo exibida de fato, ento preciso passar uma
          //posio no eixo X relativa ao 0,0 da rvore e no da tabela.
          int xAdjustment = 0;
          for (int i = 0; i < column; i++) {
            xAdjustment += getColumnModel().getColumn(i).getWidth();
          }
          MouseEvent newEvent = new MouseEvent(tree, e.getID(), e.getWhen(), e
            .getModifiers(), e.getX() - xAdjustment, e.getY(), e.getXOnScreen(),
            e.getYOnScreen(), e.getClickCount(), e.isPopupTrigger(), e
              .getButton());
          tree.dispatchEvent(newEvent);
        }
      }
    });
  }

  /**
   * Essa classe representa uma rvore com os arquivos do repositrio de
   * arquivos. Cada n possui um boto popup com as funcionalidades que existem
   * em cada tipo de n.
   */
  public static class TreeTableCellRenderer extends JTree implements
    TableCellRenderer {

    /** Tabela que contm esta tree */
    private JTable treeTable;

    /** A ltima linha que foi renderizada */
    protected int visibleRow;

    /**
     * Construtor.
     *
     * @param treeTable a tabela que conter esta rvore.
     * @param model o modelo da rvore.
     */
    public TreeTableCellRenderer(JTable treeTable, TreeTableModel model) {
      this(treeTable, model, true);
    }

    /**
     * Construtor.
     *
     * @param treeTable a tabela que conter esta rvore.
     * @param model o modelo da rvore.
     * @param showLines indica se as linhas da rvore devem ser apresentadas.
     */
    public TreeTableCellRenderer(JTable treeTable, TreeTableModel model,
      boolean showLines) {
      super(model);
      this.treeTable = treeTable;

      getSelectionModel().setSelectionMode(
        TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);

      setRowHeight(getRowHeight() + 2);
      setDragEnabled(false);
      setComponentPopupMenu(null);

      if (!showLines) {
        setNoLineTreeUI();
      }
    }

    /**
     * Altera a UI do renderizador da rvore para que as linhas verticais e
     * horizontais da mesma no sejam desenhadas.
     */
    private void setNoLineTreeUI() {
      this.setUI(new BasicTreeUI() {
        @Override
        protected void paintHorizontalLine(Graphics g, JComponent c, int y,
          int left, int right) {
        }

        @Override
        protected void paintVerticalLine(Graphics g, JComponent c, int x,
          int top, int bottom) {
        }
      });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setRowHeight(int rowHeight) {
      // A linha da tabela deve possuir a mesma altura de um n da rvore
      if (rowHeight > 0) {
        super.setRowHeight(rowHeight);
        if (treeTable != null && treeTable.getRowHeight() != rowHeight) {
          treeTable.setRowHeight(getRowHeight());
        }
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setBounds(int x, int y, int w, int h) {
      // Ns da rvore devem ter a mesma altura da clula da tabela.
      super.setBounds(x, 0, w, treeTable.getHeight());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void paint(Graphics g) {
      // Prov indentao de diretrios.
      g.translate(0, (-visibleRow * getRowHeight()));

      super.paint(g);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Component getTableCellRendererComponent(JTable table, Object value,
      boolean isSelected, boolean hasFocus, int row, int column) {
      // Retorna o renderizador com a cor de background apropriada.
      if (isSelected) {
        setBackground(table.getSelectionBackground());
      }
      else {
        setBackground(table.getBackground());
      }
      visibleRow = row;
      return this;
    }
  }

}
