package busexplorer.desktop.dialog;

import busexplorer.utils.Utils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTree;
import javax.swing.UIManager;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreeSelectionModel;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

/**
 * Janela que exibe o detalhamento de uma exceo.
 * 
 * @author Tecgraf PUC-Rio
 */
public abstract class ExceptionDialog extends JDialog {

  /**
   * Erro ou exceo a ser visualizada na janela
   */
  protected final Throwable _throwable;

  /**
   * Conjunto de informaes adicionais que sero mostradas na janela de erro e
   * includas em um possvel email ao administrador.
   */
  protected String[] additionalInfo;

  /**
   * Construtor auxiliar com <code>JDialog</code>.
   * 
   * @param owner a janela pai
   * @param title o ttulo
   * @param throwable o erro ou exceo
   */
  protected ExceptionDialog(Window owner, String title, Throwable throwable) {
    this(owner, title, throwable, null);
  }

  /**
   * Construtor auxiliar com <code>JFrame</code>.
   * 
   * @param owner a janela pai
   * @param title o ttulo
   * @param throwable o erro ou exceo
   * @param additionalInfo - lista de informaes adicionais que sero mostradas
   *        na janela de erro e includas em um possvel email.
   */
  protected ExceptionDialog(Window owner, String title, Throwable throwable,
    String[] additionalInfo) {
    super(owner, title);
    this._throwable = throwable;
    this.additionalInfo = additionalInfo;
    setWindowClosingMethod();
  }

  /**
   * Mtodo chamado pelos construtores para alterar o comportamento do dilogo
   * clicando no boto marcado por um <b>X</b> localizado no canto superior
   * direito. Por padro, a super classe ignora esse evento, ao chamar este
   * mtodo, o construtor ir garantir que a janela seja fechada.
   */
  private void setWindowClosingMethod() {
    addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        dispose();
      }
    });
  }

  /**
   * Mtodo esttico de criao de um dilogo <b>detalhado</b> com base em
   * parmetros passados pelo mdulo-cliente para uso sem construtor.
   * 
   * @param window a janela-me.
   * @param title o ttulo.
   * @param throwable a exceo.
   * @return um dialogo de erro.
   */
  public static ExceptionDialog createDialog(final Window window,
    final String title, final Throwable throwable) {
    return createDialog(window, title, throwable, (String[]) null);
  }

  /**
   * Mtodo esttico de criao de um dilogo com base em parmetros passados
   * pelo mdulo-cliente para uso sem construtor.<br>
   *  criado um ExceptionDialog <b>simplificado</b> pois o foco deste 
   * apresentar a mensagem ao usurio. Caso o usurio queira saber mais sobre o
   * erro/exceo, este dilogo apresenta uma opo para detalhar.
   * 
   * @param window a janela-me.
   * @param title o ttulo.
   * @param throwable a exceo.
   * @param message mensagem a ser apresentada ao usurio
   * @return um dialogo de erro.
   */
  public static ExceptionDialog createDialog(final Window window,
    final String title, final Throwable throwable, String message) {
    return createDialog(window, title, throwable, message, null);
  }

  /**
   * Mtodo esttico de criao de um dilogo <b>detalhado</b> com base em
   * parmetros passados pelo mdulo-cliente para uso sem construtor.
   * 
   * @param window a janela-me.
   * @param title o ttulo.
   * @param throwable a exceo.
   * @param additionalInfo - lista de informaes adicionais que sero mostradas
   *        na janela de erro e includas em um possvel email.
   * @return um dialogo de erro.
   */
  public static ExceptionDialog createDialog(final Window window,
    final String title, final Throwable throwable, String[] additionalInfo) {
    return new DetailedExceptionDialog(window, title, throwable, additionalInfo);
  }

  /**
   * Mtodo esttico de criao de um dilogo com base em parmetros passados
   * pelo mdulo-cliente para uso sem construtor.<br>
   *  criado um ExceptionDialog <b>simplificado</b> pois o foco deste 
   * apresentar a mensagem ao usurio. Caso o usurio queira saber mais sobre o
   * erro/exceo, este dilogo apresenta uma opo para detalhar.
   * 
   * @param window a janela-me.
   * @param title o ttulo.
   * @param throwable a exceo.
   * @param message mensagem a ser apresentada ao usurio
   * @param additionalInfo - lista de informaes adicionais que sero mostradas
   *        na janela de erro e includas em um possvel email.
   * @return um dialogo de erro.
   */
  public static ExceptionDialog createDialog(final Window window,
    final String title, final Throwable throwable, String message,
    String[] additionalInfo) {
    return new SimpleExceptionDialog(window, title, throwable, message,
      additionalInfo);
  }

  /**
   * Ajuste de um conjunto de elementos de interface para o mesmo tamanho.
   * 
   * @param comps os widgets que sero ajustados.
   */
  public static void adjustEqualSizes(JComponent... comps) {
    final Dimension dim = new Dimension(0, 0);
    for (int i = 0; i < comps.length; i++) {
      final Dimension pref = comps[i].getPreferredSize();
      final double h = Math.max(dim.getHeight(), pref.getHeight());
      final double w = Math.max(dim.getWidth(), pref.getWidth());
      dim.setSize(w, h);
    }
    for (int i = 0; i < comps.length; i++) {
      comps[i].setPreferredSize(dim);
    }
  }

  /**
   * Centralizao
   */
  public void center() {
    this.center(getOwner());
  }

  /**
   * Centralizao
   * 
   * @param window a janela de referncia para centralizao
   */
  public void center(Window window) {
    if (window == null) {
      GUIUtils.centerOnScreen(this);
      return;
    }

    if (window instanceof JFrame) {
      // getSize(), getX() e getY() no levam em considerao o fato da janela
      // de referncia estar maximizada; neste caso, centralizamos na tela.
      JFrame jframe = (JFrame) window;
      final int windowState = jframe.getExtendedState();
      if ((windowState & Frame.MAXIMIZED_BOTH) != 0) {
        GUIUtils.centerOnScreen(this);
        return;
      }
    }
    Dimension currentSize = this.getSize();
    Dimension windowSize = window.getSize();
    Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    int minX = window.getX() + (windowSize.width - currentSize.width) / 2;
    if ((minX + currentSize.width) > screenSize.width) {
      minX = screenSize.width - currentSize.width;
    }
    if (minX < 0) {
      minX = 0;
    }
    int minY = window.getY() + (windowSize.height - currentSize.height) / 2;
    if ((minY + currentSize.height) > screenSize.height) {
      minY = screenSize.height - currentSize.height;
    }
    if (minY < 0) {
      minY = 0;
    }
    this.setLocation(minX, minY);
  }

  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------

  /**
   * Classe que modela um dilogo <b>detalhado</b> de exibio de um
   * erro/exceo detectado no sistema. Este dilogo apresenta a pilha do
   * erro/exceo para o usurio.
   * 
   * @see ExceptionDialog dilogo que apresenta um erro/exceo, detectado no
   *      sistema, para o usurio.
   */
  private static final class DetailedExceptionDialog extends ExceptionDialog {

    /** Nmero de colunas */
    private static final int _MESSAGE_COLUMNS = 80;

    /** Nmero de linhas */
    private static final int _MESSAGE_ROWS = 4;

    /** Nmero de colunas */
    private static final int _STACK_TRACE_COLUMNS = _MESSAGE_COLUMNS;

    /** Nmero de linhas */
    private static final int _STACK_TRACE_ROWS = 14;

    /** Largura da rea da rvore */
    private static final int _TROUBLE_TREE_WIDTH = 800;

    /** Altura da rea da rvore */
    private static final int _TROUBLE_TREE_HEIGHT = 100;

    /** rea de texto de mensagens */
    private JTextArea _messageTextArea;

    /** rea de texto de mensagens da pilha. */
    private JTextArea _stackTraceTextArea;

    /** rvore de excees aninhadas */
    private JTree _throwableTree;

    /**
     * Construtor auxiliar com <code>JDialog</code>.
     * 
     * @param owner a janela pai
     * @param title o ttulo
     * @param throwable o erro ou exceo
     * @param additionalInfo - lista de informaes adicionais que sero
     *        mostradas na janela de erro e includas em um possvel email.
     */
    public DetailedExceptionDialog(final Window owner, final String title,
      final Throwable throwable, String[] additionalInfo) {
      super(owner, title, throwable, additionalInfo);
      createComponents();
    }

    /**
     * Mtodo interno de construo da inteface grfica.
     */
    private void createComponents() {

      // Define a cor que ser utilizada como cor de fundo para componentes
      // inativos
      Color inactiveColor =
        UIManager.getDefaults().getColor("TextField.inactiveBackground");

      // Cria uma rvore de erros/excees
      final JLabel troubleTreeLabel = new
        JLabel(Utils.getString(this.getClass(), "exceptionTree"));
      this._throwableTree = new JTree(new ThrowableTreeNode(this._throwable));
      this._throwableTree.addTreeSelectionListener(new TreeSelectionListener() {
        @Override
        public void valueChanged(TreeSelectionEvent e) {
          updateFields();
        }
      });
      this._throwableTree.getSelectionModel().setSelectionMode(
        TreeSelectionModel.SINGLE_TREE_SELECTION);
      final JScrollPane scrollTroubleTree =
        new JScrollPane(this._throwableTree);
      scrollTroubleTree.setPreferredSize(new Dimension(_TROUBLE_TREE_WIDTH,
        _TROUBLE_TREE_HEIGHT));

      // Cria a rea p/ mostrar a mensagem da exceo selecionada na rvore
      final JLabel messageLabel = new
        JLabel(Utils.getString(this.getClass(), "exceptionMessage"));
      this._messageTextArea =
        new JTextArea(this._throwable.getLocalizedMessage());
      this._messageTextArea.setColumns(_MESSAGE_COLUMNS);
      this._messageTextArea.setRows(_MESSAGE_ROWS);
      this._messageTextArea.setEditable(false);
      this._messageTextArea.setBackground(inactiveColor);

      // Cria a rea p/ mostrar a stack da exceo
      final JLabel stackTraceLabel = new
        JLabel(Utils.getString(this.getClass(), "stackTrace"));
      this._stackTraceTextArea =
        new JTextArea(getStackTraceText(this._throwable.getStackTrace()));
      this._stackTraceTextArea.setColumns(_STACK_TRACE_COLUMNS);
      this._stackTraceTextArea.setRows(_STACK_TRACE_ROWS);
      this._stackTraceTextArea.setEditable(false);
      this._stackTraceTextArea.setBackground(inactiveColor);

      // Cria o painel com os botes do dilogo.
      JPanel buttonsPanel = createButtonPanel();

      setLayout(new GridBagLayout());
      JScrollPane msgScrollPane = new JScrollPane(this._messageTextArea);
      JScrollPane stackTraceScroll = new JScrollPane(this._stackTraceTextArea);

      Insets li = new Insets(5, 6, 0, 10);
      Insets fi = new Insets(3, 10, 5, 10);

      int y = 0;

      // rvore
      add(troubleTreeLabel, new GBC(0, y++).west().insets(li));
      add(scrollTroubleTree, new GBC(0, y++).both().insets(fi));

      // mensagem
      add(messageLabel, new GBC(0, y++).west().insets(li));
      add(msgScrollPane, new GBC(0, y++).both().insets(fi));

      // pilha de execuo
      add(stackTraceLabel, new GBC(0, y++).west().insets(li));
      add(stackTraceScroll, new GBC(0, y++).both().insets(fi));

      add(buttonsPanel, new GBC(0, y++).center().insets(7));

      pack();
      center();
    }

    /**
     * Criao do painel de botes.
     * 
     * @return o painel.
     */
    private JPanel createButtonPanel() {
      final JPanel panel = new JPanel();
      final JButton closeButton = new JButton(new CloseAction(this));

      adjustEqualSizes(new JButton[] { closeButton });

      panel.add(closeButton);
      return panel;
    }

    /**
     * Atualizao dos campos relativos a exceo selecionada.
     */
    private void updateFields() {
      ThrowableTreeNode node =
        (ThrowableTreeNode) this._throwableTree.getSelectionPath()
          .getLastPathComponent();

      if (node == null) {
        node = (ThrowableTreeNode) this._throwableTree.getModel().getRoot();
      }
      this._messageTextArea.setText(node.getThrowable().getLocalizedMessage());
      this._stackTraceTextArea.setText(getStackTraceText(node.getThrowable()
        .getStackTrace()));
    }

    /**
     * Busca de uma string que representa a pilha de execuo.
     * 
     * @param stackTrace a pilha.
     * @return uma string.
     */
    private String getStackTraceText(final StackTraceElement[] stackTrace) {
      String text = "";
      for (int i = 0; i < stackTrace.length; i++) {
        text += stackTrace[i] + "\n";
      }
      return text;
    }
  }

  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------

  /**
   * Classe que modela um dilogo <b>simplificado</b> de exibio de um
   * erro/exceo detectado no sistema. Este dilogo apresenta uma mensagem do
   * sistema para o usurio e d a possibilidade de abrir um
   * DetailedExceptionDialog.
   * 
   * @see ExceptionDialog dilogo que apresenta um erro/exceo, detectado no
   *      sistema, para o usurio.
   * @see DetailedExceptionDialog dilogo que detalha o erro/exceo
   */
  private static final class SimpleExceptionDialog extends ExceptionDialog {

    /**
     * Construtor auxiliar com <code>JDialog</code>.
     * 
     * @param owner a janela pai
     * @param title o ttulo
     * @param message as mensagens de erro
     * @param throwable o erro ou exceo
     * @param additionalInfo - lista de informaes adicionais que sero
     *        mostradas na janela de erro e includas em um possvel email.
     */
    public SimpleExceptionDialog(final Window owner, final String title,
      final Throwable throwable, final String message,
      final String[] additionalInfo) {
      super(owner, title, throwable, additionalInfo);
      buildGui(message);
    }

    /**
     * Inicializao interna do dilogo.
     * 
     * @param message mensagem que deve aparecer no dilogo.
     */
    private void buildGui(String message) {
      getContentPane().setLayout(new BorderLayout());
      getContentPane().add(makeIconPanel(), BorderLayout.WEST);
      getContentPane().add(makeMainPanel(message), BorderLayout.CENTER);
      pack();
      center();
    }

    /**
     * Cria o painel que contm o cone de erro
     * 
     * @return o painel
     */
    private JPanel makeIconPanel() {
      JPanel panel = new JPanel();
      panel.add(new JLabel(UIManager.getIcon("OptionPane.errorIcon")));

      return panel;
    }

    /**
     * Cria o painel principal
     * 
     * @param message mensagem a ser exibida
     * @return o painel
     */
    private JPanel makeMainPanel(String message) {

      JPanel mainPanel = new JPanel(new GridBagLayout());
      GridBagConstraints c = new GridBagConstraints();
      c.insets = new Insets(12, 12, 11, 11);
      c.anchor = GridBagConstraints.CENTER;
      c.weightx = 1;
      c.weighty = 1;
      mainPanel.add(makeInfoPanel(message), c);
      c.gridy = 1;
      c.anchor = GridBagConstraints.SOUTHEAST;
      c.insets = new Insets(17, 12, 11, 11);
      mainPanel.add(makeButtonsPanel(), c);

      return mainPanel;
    }

    /**
     * Cria o painel que contm a mensagem de erro
     * 
     * @param message mensagem a ser exibida
     * @return o painel
     */
    private JPanel makeInfoPanel(String message) {

      JPanel panel = new JPanel(new GridBagLayout());
      GridBagConstraints c = new GridBagConstraints();
      c.insets = new Insets(0, 0, 0, 0);

      c.fill = GridBagConstraints.HORIZONTAL;
      if (message == null || message.isEmpty() || message.startsWith("<html>")) {
        JLabel label = new JLabel(message);
        panel.add(label, c);
      }
      else {
        JMultilineLabel label = new JMultilineLabel();
        label.setMaxWidth(550);
        label.setJustified(false);
        label.setText(message);
        panel.add(label, c);
      }

      c.gridy = 1;
      c.insets = new Insets(12, 0, 0, 0);
      JLabel jwhere = new JLabel(Utils.getString(this.getClass(),
        "executionError"));
      panel.add(jwhere, c);
      c.gridy = 2;
      c.insets = new Insets(0, 0, 0, 0);
      panel.add(new JLabel(Utils.getString(this.getClass(), "contactError")),
        c);
      return panel;
    }

    /**
     * Cria o painel com os botes
     * 
     * @return o painel
     */
    private JPanel makeButtonsPanel() {
      final JPanel panel = new JPanel();
      final JButton detailButton = new JButton(new DetailThrowableAction(this));
      final JButton closeButton = new JButton(new CloseAction(this));

      adjustEqualSizes(new JButton[] { detailButton, closeButton });

      panel.add(detailButton);
      panel.add(closeButton);
      return panel;
    }
  }

  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------

  /**
   * Ao que ao ser requisitada fecha a janela corrente.
   */
  class CloseAction extends AbstractAction {

    /**
     * Janela corrente que contm esta ao amarrada a um de seus componentes.
     */
    protected ExceptionDialog owner;

    /**
     * Construtor
     * 
     * @param owner janela corrente
     */
    protected CloseAction(ExceptionDialog owner) {
      this(owner, LNG.get(ExceptionDialog.class.getSimpleName() + ".close"));
    }

    /**
     * Construtor
     * 
     * @param owner janela corrente
     * @param text texto.
     */
    protected CloseAction(ExceptionDialog owner, String text) {
      super(text);
      this.owner = owner;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      this.owner.dispose();
    }
  }

  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------

  /**
   * Ao que ao ser requisitada abre um dilogo detalhando o erro/exceo
   * apresentado pela janela corrente. Em seguida fecha a janela corrente.
   */
  final class DetailThrowableAction extends CloseAction {

    /**
     * Construtor
     * 
     * @param owner janela corrente
     */
    protected DetailThrowableAction(ExceptionDialog owner) {
      super(owner, LNG.get(ExceptionDialog.class.getSimpleName() +
        ".errorDetails"));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      super.actionPerformed(e);
      ExceptionDialog dialog =
        ExceptionDialog.createDialog(owner, owner.getTitle(), owner._throwable,
          owner.additionalInfo);
      dialog.setVisible(true);
    }
  }
}
