/**
 * $Id: FlowConfiguratorImpl.java 132621 2012-08-23 16:30:23Z fpina $
 */
package tecgraf.openbus.algorithmservice.v1_1;

import java.security.InvalidParameterException;
import java.util.HashMap;
import java.util.Vector;

import tecgraf.opendreams.commontypes.v1_0.Property;

/**
 * A classe <code>FlowConfiguratorImpl</code> implementa um valutype
 * <code>FlowConfigurator</code>
 *
 * @author Tecgraf
 */
public class FlowConfiguratorImpl extends FlowConfigurator {
  /**
   * Separador entre o ID do algoritmo e o nome do parmetro
   */
  private static final String ID_SEPARATOR = ".";

  /**
   * Mapa dos algoritmo do fluxo com o identificador do algoritmo como chave.
   */
  private HashMap<Integer, FlowNode> algorithmsIdMap = null;

  /**
   * Construtr padro.
   */
  public FlowConfiguratorImpl() {
    super();
    this.algorithmName = "";
    this.algorithmVersion = "";
    this.type = AlgorithmType.FLOW;
    this.algorithms = new FlowNode[0];
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setParameterValueById(int algorithmId, String parameterName,
    IParameterValue value) throws AlgorithmIdDoesNotExistsInFlowException,
    ParameterDoesNotExistsException, IncorrectParamTypeException,
    InvalidParameterException {
    SimpleConfigurator sConfig = findAlgoConfigById(algorithmId);
    //XXX verificar tratamento adequado
    try {
      sConfig.setParameterValue(parameterName, value);
    }
    catch (tecgraf.openbus.algorithmservice.v1_1.InvalidParameterException e) {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setParameterValueAsTextById(int algorithmId,
    String parameterName, String valueAsText)
    throws ParameterDoesNotExistsException,
    AlgorithmIdDoesNotExistsInFlowException {
    SimpleConfigurator sConfig = findAlgoConfigById(algorithmId);
    sConfig.setParameterValueAsText(parameterName, valueAsText);
  }

  /**
   * Busca o configurador de um algoritmo dentro do fluxo pelo nome
   * identificador do algoritmo.
   *
   * @param id identificador do algoritmo
   *
   * @return configurador do algoritmo
   *
   * @throws AlgorithmIdDoesNotExistsInFlowException caso o algoritmo no exista
   *         dentro do fluxo
   */
  private SimpleConfigurator findAlgoConfigById(int id)
    throws AlgorithmIdDoesNotExistsInFlowException {
    if (algorithmsIdMap == null) {
      buildNodeIdMap();
    }

    if (!this.algorithmsIdMap.containsKey(id)) {
      throw new AlgorithmIdDoesNotExistsInFlowException();
    }
    else {
      FlowNode node = this.algorithmsIdMap.get(id);
      return SimpleConfigurator.class.cast(node.config);
    }
  }

  /**
   * Constri o mapa de algoritmos utilizando o identificador do algoritmo como
   * chave.
   */
  private void buildNodeIdMap() {
    this.algorithmsIdMap =
      new HashMap<Integer, FlowNode>(this.algorithms.length);

    for (FlowNode node : this.algorithms) {
      this.algorithmsIdMap.put(node.id, node);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Property[] getParameters() throws ConversionToTextException {
    Vector<Property> paramVector = new Vector<Property>();

    for (Integer id : algorithmsIdMap.keySet()) {
      FlowNode node = algorithmsIdMap.get(id);
      SimpleConfigurator algoConfig =
        SimpleConfigurator.class.cast(node.config);
      for (int i = 0; i < algoConfig.parameters.length; i++) {
        Parameter param = algoConfig.parameters[i];
        if (param.value != null) {
          paramVector.add(new Property(id + ID_SEPARATOR + param.name,
            param.value.convertToText()));
        }
      }
    }

    Property[] dictionary = new Property[paramVector.size()];
    return paramVector.toArray(dictionary);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] getAlgoritmInfo() {
    return new String[] { "-name", this.algorithmName, "-version",
        this.algorithmVersion };
  }
}
