package br.pucrio.tecgraf.soma.serviceapi.persistence.repository.impl;

import com.github.tennaito.rsql.misc.ArgumentFormatException;
import com.github.tennaito.rsql.misc.ArgumentParser;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

public class ExtendedArgumentParser implements ArgumentParser {
    private static final Logger LOG = Logger.getLogger(ExtendedArgumentParser.class.getName());

    @Override
    public <T> T parse(String argument, Class<T> type) throws ArgumentFormatException, IllegalArgumentException {
        LOG.log(Level.INFO, "Parsing argument ''{0}'' as type {1}, thread {2}", new Object[] {argument, type.getSimpleName(), Thread.currentThread().getName()});

        // Nullable object
        if (argument == null || "null".equals(argument.trim().toLowerCase())) {
            return (T) null;
        }

        // common types
        try {
            if (type.equals(String.class)) return (T) argument;
            if (type.equals(Integer.class) || type.equals(int.class)) return (T) Integer.valueOf(argument);
            if (type.equals(Boolean.class) || type.equals(boolean.class)) return (T) Boolean.valueOf(argument);
            if (type.isEnum()) return (T) Enum.valueOf((Class<Enum>)type, argument.toUpperCase());
            if (type.equals(Float.class)   || type.equals(float.class)) return (T) Float.valueOf(argument);
            if (type.equals(Double.class)  || type.equals(double.class)) return (T) Double.valueOf(argument);
            if (type.equals(Long.class)    || type.equals(long.class)) return (T) Long.valueOf(argument);
            if (type.equals(BigDecimal.class) ) return (T) new BigDecimal(argument);
        } catch (IllegalArgumentException ex) {
            throw new ArgumentFormatException(argument, type);
        }

        // date
        if (type.equals(Date.class)) {
            return (T) parseDate(argument, type);
        }

        // localdatetime
        if (type.equals(LocalDateTime.class)) {
            return (T) parseLocalDateTime(argument, type);
        }

        // try to parse via valueOf(String s) method
        try {
            LOG.log(Level.INFO, "Trying to get and invoke valueOf(String s) method on {0}", type);
            Method method = type.getMethod("valueOf", String.class);
            return (T) method.invoke(type, argument);
        } catch (InvocationTargetException ex) {
            throw new ArgumentFormatException(argument, type);
        } catch (ReflectiveOperationException ex) {
            LOG.log(Level.WARNING, "{0} does not have method valueOf(String s) or method is inaccessible", type);
            throw new IllegalArgumentException("Cannot parse argument type " + type);
        }
    }

    private <T> Date parseDate(String argument, Class<T> type) {
        LocalDateTime dateTime = parseLocalDateTime(argument, type);
        return Date.from(dateTime.atZone(ZoneId.systemDefault()).toInstant());
    }

    private <T> LocalDateTime parseLocalDateTime(String argument, Class<T> type) {
        try {
            // Allowed patterns: "yyyy/MM/ddx" , "yyyy/MM/dd HH:mmx" , "yyyy/MM/dd HH:mm:ssx" , "yyyy/MM/dd HH:mm:ss.SSSx" //ISO 8601
            // x  zone-offset. Allowed patterns:  -hh; -hhmm; +hh; +hhmm
            // Date-hour separators can be " " or "T".
            DateTimeFormatter formatter = new DateTimeFormatterBuilder()
                    .appendPattern("yyyy/MM/dd[[' ']['T']HH:mm[:ss][.SSS]]x")
                    .parseDefaulting(ChronoField.HOUR_OF_DAY, 0)
                    .parseDefaulting(ChronoField.MINUTE_OF_HOUR, 0)
                    .parseDefaulting(ChronoField.SECOND_OF_MINUTE, 0)
                    .toFormatter();
            // Applying timezone
            ZonedDateTime zonedDateTime = ZonedDateTime.parse(argument, formatter);
            return (LocalDateTime.ofInstant(zonedDateTime.toInstant(), ZoneOffset.UTC));

        } catch (DateTimeParseException ex) {
            throw new ArgumentFormatException(argument, type);
        }
    }

    @Override
    public <T> List<T> parse(List<String> arguments, Class<T> type) throws ArgumentFormatException, IllegalArgumentException {
        List<T> castedArguments = new ArrayList<T>(arguments.size());
        for (String argument : arguments) {
            castedArguments.add(this.parse(argument, type));
        }
        return castedArguments;
    }
}
