package br.pucrio.tecgraf.soma.job.application.appservice;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.ws.rs.ForbiddenException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.fasterxml.jackson.databind.ObjectMapper;

import br.pucrio.tecgraf.soma.job.api.model.Multiflow;
import br.pucrio.tecgraf.soma.job.api.model.MultiflowBasicResponse;
import br.pucrio.tecgraf.soma.job.api.model.MultiflowFullResponse;
import br.pucrio.tecgraf.soma.job.application.service.FlowService;
import br.pucrio.tecgraf.soma.job.application.service.MultiflowService;
import br.pucrio.tecgraf.soma.job.application.service.ProjectService;
import br.pucrio.tecgraf.soma.job.application.service.ReplicaFileService;
import br.pucrio.tecgraf.soma.job.domain.model.MultiflowReplicaFile;
import br.pucrio.tecgraf.soma.job.domain.model.ReplicaFile;

enum ResponseModel {
  InitialParams,
  RequiredParams,
  OptionalParams
}
@Service
public class MultiflowAppService {

  @Autowired private ProjectService projectService;
  @Autowired private MultiflowService multiflowService;
  @Autowired private FlowService flowService;
  @Autowired private ReplicaFileService replicaFileService;


   /**
   * Persiste um multifluxo.
   *
   * @param multiflowData dados do multifluxo.
   * @return o id do multifluxo
   */
  @Transactional
  public MultiflowBasicResponse createMultiflow(Multiflow multiflowData) {

    if (!projectService.hasPermission(multiflowData.getProjectId())) {
      throw new ForbiddenException("User has no permission to add a Multiflow to this project");
    }

    br.pucrio.tecgraf.soma.job.domain.model.Flow flow = new br.pucrio.tecgraf.soma.job.domain.model.Flow();
    flow.setName(multiflowData.getFlowName());
    flow.setFlowData(multiflowData.getFlowData());
    flow.setLayoutData(multiflowData.getLayoutData());
    flowService.createFlow(flow);

    br.pucrio.tecgraf.soma.job.domain.model.Multiflow multiflow = new br.pucrio.tecgraf.soma.job.domain.model.Multiflow();
    String name = multiflowData.getName();
    if(name == null || name.trim().length() == 0) {
      // Usa o nome do fluxo caso não tenha sido definido um nome específico para o Multifluxo
      name = multiflowData.getFlowName();
    }
    multiflow.setName(name);
    multiflow.setFlow(flow);
    multiflow.setProjectId(multiflowData.getProjectId());
    br.pucrio.tecgraf.soma.job.domain.model.Multiflow created = multiflowService.createMultiflow(multiflow);
    return convertToBasicRESTModel(created);
  }

  @Transactional(readOnly = true)
  public List<MultiflowFullResponse> findMultiflowsByProjectId(String projectId) {
    if (!projectService.hasPermission(projectId)) {
      throw new ForbiddenException("User has no permission to get Multiflows from this project");
    }

    List<br.pucrio.tecgraf.soma.job.domain.model.Multiflow> multiflows = multiflowService.findMultiflowsByProjectId(projectId);
    return multiflows.stream().map(mf -> MultiflowAppService.convertToFullRESTModel(mf)).collect(Collectors.toList());
  }

  @Transactional(readOnly = true)
  public MultiflowFullResponse findMultiflowById(Long multiflowId) {
    br.pucrio.tecgraf.soma.job.domain.model.Multiflow multiflow = multiflowService.findMultiflowById(multiflowId);

    if (!projectService.hasPermission(multiflow.getProjectId())) {
      throw new ForbiddenException("User has no permission to get this Multiflow");
    }

    return convertToFullRESTModel(multiflow);
  }

  @Transactional
  public MultiflowFullResponse updateMultiflow(Long multiflowId, String updatedPath, String updatedName, Object parameterMapping) {
    // TODO: Avaliar possibilidade de adaptar método para receber um objeto e alterar quaisquer de seus parâmetros
    br.pucrio.tecgraf.soma.job.domain.model.Multiflow multiflow = multiflowService.findMultiflowById(multiflowId);

    if (!projectService.hasPermission(multiflow.getProjectId())) {
      throw new ForbiddenException("User has no permission to update this Multiflow");
    }

    List<MultiflowReplicaFile> multiflowReplicaFiles = multiflow.getMultiflowReplicaFiles();
    boolean shouldCreateNewReplicaFile = (updatedPath != null && !updatedPath.trim().isEmpty());

    if (shouldCreateNewReplicaFile) {
      // Como está alterando o path do arquivo de réplicas, adiciona um novo arquivo e uma nova associação no banco de dados e deleta as demais associações válidas (isDeleted=true).
      // TODO: Validar - A nova associação é criada contendo apenas as informações passadas. Cabe ao cliente passar todas as informações que queira persistir
      this.setAllNonDeletedMultiflowReplicaFilesAsDeleted(multiflowReplicaFiles);

      br.pucrio.tecgraf.soma.job.domain.model.ReplicaFile newReplicaFile = new br.pucrio.tecgraf.soma.job.domain.model.ReplicaFile();
      newReplicaFile.setName(updatedName);
      newReplicaFile.setPath(updatedPath);
      replicaFileService.createReplicaFile(newReplicaFile);
  
      br.pucrio.tecgraf.soma.job.domain.model.MultiflowReplicaFile fileAssociation = new br.pucrio.tecgraf.soma.job.domain.model.MultiflowReplicaFile();
      fileAssociation.setReplicaFile(newReplicaFile);
      fileAssociation.setIsDeleted(false);

      fileAssociation.setParameterMapping(parameterMapping);
      multiflow.addMultiflowReplicaFile(fileAssociation);
    } else {
      // Como NÃO está alterando o path do arquivo de réplicas, apenas atualiza o arquivo e a associação já existente.
      MultiflowReplicaFile currentMultiflowReplicaFile = this.setOnlyCurrentMultiflowReplicaFileAsNonDeleted(multiflowReplicaFiles);

      if (parameterMapping != null) {
        currentMultiflowReplicaFile.setParameterMapping(parameterMapping);
      }
      if (updatedName != null) {
        currentMultiflowReplicaFile.getReplicaFile().setName(updatedName);
      }
    }

    multiflow.setLastModifiedTime(LocalDateTime.now(ZoneOffset.UTC));  
    multiflow.setIsDeleted(false);
    br.pucrio.tecgraf.soma.job.domain.model.Multiflow updatedMultiflow = multiflowService.updateMultiflow(multiflow);
    return convertToFullRESTModel(updatedMultiflow);
  }

  @Transactional
  public void deleteMultiflow(Long multiflowId) {
    br.pucrio.tecgraf.soma.job.domain.model.Multiflow multiflow = multiflowService.findMultiflowById(multiflowId);

    if (!projectService.hasPermission(multiflow.getProjectId())) {
      throw new ForbiddenException("User has no permission to delete this Multiflow");
    }

    // O delete vai somente marcar o multifluxo com is_deleted=true, mas o multifluxo vai permanecer na tabela
    multiflow.setIsDeleted(true);
    multiflowService.updateMultiflow(multiflow);
  }

  private static MultiflowBasicResponse convertToBasicRESTModel(br.pucrio.tecgraf.soma.job.domain.model.Multiflow dbModel) {
    MultiflowBasicResponse restModel = new MultiflowBasicResponse();
    restModel.setId(dbModel.getId());
    restModel.setFlowName(dbModel.getFlow().getName());
    restModel.setName(dbModel.getName());
    restModel.setCreationTime(Date.from(dbModel.getCreationTime().toInstant(ZoneOffset.UTC)));
    restModel.setLastModifiedTime(Date.from(dbModel.getLastModifiedTime().toInstant(ZoneOffset.UTC)));
    return restModel;
  }

  private static MultiflowFullResponse convertToFullRESTModel(br.pucrio.tecgraf.soma.job.domain.model.Multiflow dbModel) {
    MultiflowFullResponse restModel = new MultiflowFullResponse();
    restModel.setId(dbModel.getId());
    restModel.setFlowName(dbModel.getFlow().getName());
    restModel.setName(dbModel.getName());
    restModel.setCreationTime(Date.from(dbModel.getCreationTime().toInstant(ZoneOffset.UTC)));
    restModel.setLastModifiedTime(Date.from(dbModel.getLastModifiedTime().toInstant(ZoneOffset.UTC)));
    restModel.setFlowData(dbModel.getFlow().getFlowData());
    restModel.setLayoutData(dbModel.getFlow().getLayoutData());
    restModel.setProjectId(dbModel.getProjectId());

    List<MultiflowReplicaFile> replicaFiles = dbModel.getMultiflowReplicaFiles();
    // Pega a associação de arquivos do multifluxo, filtrando pelas que já foram removidos e ordenando pelos mais recentes
    List<MultiflowReplicaFile> nonDeletedFiles = replicaFiles.stream()
                                                    .filter(f -> !f.isDeleted())
                                                    .sorted(Comparator.comparing(MultiflowReplicaFile::getId, (a,b) -> b.compareTo(a)))
                                                    .collect(Collectors.toList());

    if(!nonDeletedFiles.isEmpty()) {
      // Pega a associação de arquivo mais recente caso haja dois arquivos não apagados associados a um multifluxo
      MultiflowReplicaFile fileAssociation = nonDeletedFiles.get(0);
      ReplicaFile replicaFile = fileAssociation.getReplicaFile();
      restModel.setReplicaFilePath(replicaFile.getPath());
      restModel.setReplicaFileName(replicaFile.getName());
      Object parameterMapping = fileAssociation.getParameterMapping();

      //TODO: No caso de chamada ao endpoint PATCH ter sido feito com valor do parâmetro em String o retorno está
      // sendo recebido como string (não sabemos o motivo), por isso estamos forçando a conversão para um objeto
      if(parameterMapping instanceof String) {
        ObjectMapper jsonMapper = new ObjectMapper();
        try {
          parameterMapping = jsonMapper.readValue((String) parameterMapping, Map.class);
        } catch (Exception e) {
          parameterMapping = null;
          e.printStackTrace();
        }
      }
      restModel.setParameterMapping(parameterMapping);
    }

    return restModel;
  }

  private void setAllNonDeletedMultiflowReplicaFilesAsDeleted(List<MultiflowReplicaFile> allMultiflowReplicaFiles) {
    this.setNonDeletedMultiflowReplicaFilesAsDeleted(allMultiflowReplicaFiles, false);
  }

  private MultiflowReplicaFile setOnlyCurrentMultiflowReplicaFileAsNonDeleted(List<MultiflowReplicaFile> allMultiflowReplicaFiles) {
      List<MultiflowReplicaFile> validAssociations = this.setNonDeletedMultiflowReplicaFilesAsDeleted(allMultiflowReplicaFiles, true);
      return validAssociations.get(validAssociations.size() - 1);
  }

  private List<MultiflowReplicaFile> setNonDeletedMultiflowReplicaFilesAsDeleted(List<MultiflowReplicaFile> allMultiflowReplicaFiles, boolean exceptLast) {
      List<MultiflowReplicaFile> validAssociations = allMultiflowReplicaFiles.stream().filter(f -> !f.isDeleted()).collect(Collectors.toList());
      int maxIndex = validAssociations.size() - (exceptLast ? 1 : 0);
      for (int i = 0; i < maxIndex; i++) {
          MultiflowReplicaFile validadAssociation  = validAssociations.get(i);
          validadAssociation.setIsDeleted(true);
      }
      return validAssociations;
  }


}
