package br.pucrio.tecgraf.soma.job.application.controller;

import java.util.Arrays;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.ServiceUnavailableException;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.SecurityContext;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;

import br.pucrio.tecgraf.soma.job.api.JobsApi;
import br.pucrio.tecgraf.soma.job.api.model.Algorithm;
import br.pucrio.tecgraf.soma.job.api.model.AlgorithmResponse;
import br.pucrio.tecgraf.soma.job.api.model.NewComment;
import br.pucrio.tecgraf.soma.job.api.model.JobData;
import br.pucrio.tecgraf.soma.job.api.model.JobPagination;
import br.pucrio.tecgraf.soma.job.api.model.JobResponse;
import br.pucrio.tecgraf.soma.job.application.appservice.JobAppService;
import br.pucrio.tecgraf.soma.job.domain.dto.DomainMapper;

@Component
public class JobController extends JobsApi {

  @Autowired private JobAppService jobAppService;
  @Autowired private HttpServletRequest request;
  private static final int maxResultLimit = 1000;

  public JobController() {
    super(null);
  }

  @Override
  public Response jobsHistoryAlgorithmsGet(String q, String locale, SecurityContext securityContext)
      throws ServiceUnavailableException {

    AlgorithmResponse response = new AlgorithmResponse();
    response.setProcessingDate(System.currentTimeMillis());

    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if(accessToken == null || accessToken.isEmpty()) {
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }
    q = this.filterUserProjects(locale, accessToken, q);

    response.setAlgorithms(
            Arrays.asList(DomainMapper.convert(jobAppService.findDistinctAlgorithms(q), Algorithm[].class)));

    return Response.ok().entity(response).build();
  }

  @Override
  public Response jobsHistoryJobIdCommentPut(String jobId, NewComment requestBody, String locale, SecurityContext securityContext) {
    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if(accessToken == null || accessToken.isEmpty()) {
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }
    if (requestBody == null || requestBody.getNewComment() == null) {
      return Response.status(HttpStatus.BAD_REQUEST.value(), "Missing body with newComment parameter")
          .build();
    }
    List<String> userProjects = jobAppService.getUserProjects(locale, accessToken);
    jobAppService.editJobComment(jobId, requestBody.getNewComment(), userProjects);
    return Response.status(Response.Status.NO_CONTENT).build();
  }

  @Override
  public Response jobsHistoryDelete(List<String> jobIds, String locale, SecurityContext securityContext) {
    jobAppService.markJobsAsDeleted(jobIds);
    return Response.status(Response.Status.NO_CONTENT).build();
  }

  @Override
  public Response jobsHistoryGet(
      String q,
      Integer offset,
      Integer limit,
      Boolean asc,
      String attr,
      Boolean grouped,
      String locale,
      SecurityContext securityContext) {

    if (limit == null) {
      limit = maxResultLimit;
    } else if (limit <= 0 || limit > maxResultLimit) {
      return Response.status(HttpStatus.BAD_REQUEST.value(),
              "Invalid limit value").build();
    }
    if (offset == null) {
      offset = 0;
    } else if (offset < 0) {
      return Response.status(HttpStatus.BAD_REQUEST.value(),
              "Offset must be a positive number or zero").build();
    }
    if (asc == null) {
      asc = false;
    }

    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if(accessToken == null || accessToken.isEmpty()) {
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }
    q = this.filterUserProjectsAndHandleDeleted(locale, accessToken, q);

    JobData data = new JobData();
    data.processingDate(System.currentTimeMillis());
    JobPagination pagination = new JobPagination();
    pagination.setLimit(limit);
    JobResponse response = new JobResponse();
    response.data(data);
    response.pagination(pagination);

    if (grouped) {
      jobAppService.findGroupedJobs(q, limit, offset, asc, attr, response);
    }
    else {
      jobAppService.findJobs(q, limit, offset, asc, attr, response);
    }

    return Response.ok().entity(response).build();
  }

  public String filterUserProjects(String locale, String accessToken, String q) {

    String projectsQuery = this.getUserProjectsPredicate(locale, accessToken, "job.projectId");
    if (q == null || q.length() == 0) { // caso de query vazia
      q = projectsQuery;
    } else {
      q = String.format("(%s);(%s)", q, projectsQuery);
    }
    return q;
  }

  public String filterUserProjectsAndHandleDeleted(String locale, String accessToken, String q) {

    String projectsQuery = this.getUserProjectsPredicate(locale, accessToken, "projectId");
    String baseQuery = "isDeleted==false;" + projectsQuery; // por padrao só retornaremos os jobs não deletados
    if (q == null || q.length() == 0 ) { // caso de query vazia
      q = baseQuery;
    } else if( q.contains("isDeleted") == false ) { // se a busca NÃO for pelo atributo isDeleted
      // Adiciona restrição de projetos do usuário e de jobs não apagados
      q = String.format("(%s);(%s)",q, baseQuery);
    } else {
      // Adiciona apenas restrição de projetos do usuário
      q = String.format("(%s);(%s)", q, projectsQuery);
    }

    return q;
  }

  public String getUserProjectsPredicate(String locale, String accessToken, String projectColumn) {
    List<String> projectIds = jobAppService.getUserProjects(locale, accessToken);
    if(projectIds.isEmpty()) {
      throw new NotFoundException("User has access to no projects");
    }
    StringBuilder projectsQuery = new StringBuilder(projectColumn);
    projectsQuery.append("=in=(");
    for (String projectId : projectIds) {
      projectsQuery.append(projectId);
      projectsQuery.append(",");
    }
    projectsQuery.deleteCharAt(projectsQuery.lastIndexOf(","));
    projectsQuery.append(")");

    return projectsQuery.toString();
  }

}