/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2008(c).
 * 
 * file: GridCoverageDataStore.java
 * created: 17/10/2008
 */
package org.geotools.data.gridcoverage;

import java.io.IOException;

import org.geotools.coverage.grid.io.AbstractGridCoverage2DReader;
import org.geotools.data.FeatureSource;
import org.geotools.data.collection.CollectionDataStore;
import org.geotools.data.gridcoverage.exception.DataAccessGridCoverageException;
import org.geotools.feature.FeatureCollection;
import org.geotools.resources.coverage.FeatureUtilities;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.parameter.GeneralParameterValue;

/**
 * Default implementation of a {@link GridCoverageDataStore}, which bases itself
 * in {@link CollectionDataStore}. As such, this implementation works by
 * wrapping a provided {@link AbstractGridCoverage2DReader} as a
 * {@link FeatureCollection}, and then handling that collection as normally done
 * by a CollectionDataStore. In the end, the provided coverage reader will be
 * retrievable as a {@link GridCoverageFeatureSource}.
 * 
 * @author milton
 * @since TDK 3.0.0
 */
public class DefaultGridCoverageDataStore extends CollectionDataStore implements GridCoverageDataStore
{
    /**
     * Regular constructor for instantiating a GridCoverageDataStore given a
     * provided coverage reader.
     * 
     * @param reader an AbstractGridCoverage2DReader on which the DataStore will
     *            be based.
     * @throws DataAccessGridCoverageException if the provided reader could not
     *             be wrapped.
     * @since TDK 3.0.0
     */
    public DefaultGridCoverageDataStore(AbstractGridCoverage2DReader reader) throws DataAccessGridCoverageException
    {
        this(reader, null);
    }

    /**
     * Constructor for instantiating a GridCoverageDataStore given a provided
     * coverage reader and user-specified reading parameters.
     * 
     * @param reader an AbstractGridCoverage2DReader on which the DataStore will
     *            be based.
     * @param params optional GeneralParameterValue array which may be used in
     *            calls to
     *            {@link AbstractGridCoverage2DReader#read(GeneralParameterValue[])}
     *            (can be null)
     * @throws DataAccessGridCoverageException if the provided reader could not
     *             be wrapped.
     * @since TDK 3.0.0
     */
    public DefaultGridCoverageDataStore(AbstractGridCoverage2DReader reader, GeneralParameterValue[] params)
            throws DataAccessGridCoverageException
    {
        super(wrapGridCoverageReader(reader, params));
    }

    /**
     * Static method for wrapping a provided coverage reader as a
     * FeatureCollection. The final wrapped object will consist of a collection
     * with one single Feature, whose schema includes the bounds of the
     * coverage, the coverage reader itself, and the optional reading
     * parameters.
     * 
     * @param reader an AbstractGridCoverage2DReader around which the
     *            FeatureCollection will be wrapped.
     * @param params optional GeneralParameterValue array which may be used in
     *            calls to
     *            {@link AbstractGridCoverage2DReader#read(GeneralParameterValue[])}
     *            (can be null)
     * @return the wrapped FeatureCollection.
     * @throws DataAccessGridCoverageException if the provided reader could not
     *             be wrapped.
     * @since TDK 3.0.0
     */
    protected static FeatureCollection<SimpleFeatureType, SimpleFeature> wrapGridCoverageReader(
            AbstractGridCoverage2DReader reader, GeneralParameterValue[] params) throws DataAccessGridCoverageException
    {
        // checks reader argument
        if (reader == null)
            throw new NullPointerException("reader can't be null");

        // wraps reader in a FeatureCollection
        FeatureCollection<SimpleFeatureType, SimpleFeature> wrappedReader = null;
        try
        {
            wrappedReader = FeatureUtilities.wrapGridCoverageReader(reader, params);
        }
        catch (Exception e)
        {
            throw new DataAccessGridCoverageException("Could not wrap GridCoverageReader.", e);
        }
        return wrappedReader;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.tecgraf.tdk.dataaccess.gridcoverage.GridCoverageDataStore#
     * getCoverageReader()
     */
    @Override
    public AbstractGridCoverage2DReader getCoverageReader() throws DataAccessGridCoverageException
    {
        // retrieves the typeNames of the DataStore: there should be one
        // typeName, corresponding to the GridCoverageFeatureSource contained in
        // the DataStore
        String[] typeNames = getTypeNames();
        if (typeNames.length < 1)
            throw new DataAccessGridCoverageException(
                    "DataStore should contain a typeName for the contained GridCoverageFeatureSource.");

        // retrieves first FeatureSource from the DataStore
        FeatureSource<SimpleFeatureType, SimpleFeature> featSource = null;
        try
        {
            featSource = getFeatureSource(typeNames[0]);
        }
        catch (IOException e)
        {
            throw new DataAccessGridCoverageException(
                    "Could not retrieve first FeatureSource from GridCoverageDataStore.");
        }

        // checks if FeatureSource is of the appropriate type
        if (!(featSource instanceof GridCoverageFeatureSource))
            throw new DataAccessGridCoverageException(
                    "DataStore's first typeName should be a GridCoverageFeatureSource.");

        // extracts coverage reader from the FeatureSource and returns it
        return ((GridCoverageFeatureSource) featSource).getCoverageReader();
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.geotools.data.AbstractDataStore#getFeatureSource(java.lang.String)
     */
    @Override
    @SuppressWarnings("unchecked")
    public FeatureSource<SimpleFeatureType, SimpleFeature> getFeatureSource(final String typeName) throws IOException
    {
        final SimpleFeatureType featureType = getSchema(typeName);

        // returns the appropriate flavor of FeatureSource, depending on whether
        // it is read-only or writable, and whether it supports long term
        // transactions (locking)
        if (isWriteable)
        {
            if (getLockingManager() != null)
                return new DefaultGridCoverageFeatureLocking(this, featureType, getSupportedHints());
            else
                return new DefaultGridCoverageFeatureStore(this, featureType, getSupportedHints());
        }
        else
            return new DefaultGridCoverageFeatureSource(this, featureType, getSupportedHints());
    }
}
