/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2008(c).
 * 
 * file: GridCoverageFeatureSourceImpl.java
 * created: 17/10/2008
 */
package org.geotools.data.gridcoverage;

import java.awt.RenderingHints;
import java.io.IOException;
import java.util.Iterator;
import java.util.Set;

import org.geotools.coverage.grid.io.AbstractGridCoverage2DReader;
import org.geotools.data.AbstractDataStore;
import org.geotools.data.AbstractFeatureSource;
import org.geotools.data.DataStore;
import org.geotools.data.FeatureListener;
import org.geotools.data.FeatureSource;
import org.geotools.data.gridcoverage.exception.DataAccessGridCoverageException;
import org.geotools.feature.FeatureCollection;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;

/**
 * Specifies a {@link FeatureSource} that wraps and allows read access to
 * coverage data. In practice, this will correspond to a FeatureSource whose
 * schema specifies:
 * <ul>
 * <li>a "bounds" Polygon attribute that will contain the coverage bounds and
 * will serve as the FeatureSource's default geometry attribute
 * <li>a "grid" AbstractGridCoverage2DReader attribute that will contain the
 * coverage reader itself
 * <li>a "params" GeneralParameterValue[] attribute for storing optional
 * coverage reading parameters.
 * </ul>
 * As such, this FeatureSource responds normally to requests such as whether its
 * data intersects a given area (using the "bounds" attribute), while the
 * coverage data itself is made available through the "grid" attribute.
 * 
 * @author milton
 * @since TDK 3.0.0
 */
@SuppressWarnings("unchecked")
public class DefaultGridCoverageFeatureSource extends AbstractFeatureSource implements GridCoverageFeatureSource
{
    private DataStore _dataStore;
    private SimpleFeatureType _featureType;

    /**
     * Regular constructor that builds a GridCoverageFeatureSource within a
     * provided DataStore.
     * 
     * @param dataStore the DataStore that will contain the newly created
     *            FeatureSource.
     * @param featureType the SimpleFeatureType (schema) of the new
     *            GridCoverageFeatureSource, which should contain a "bounds"
     *            Polygon attribute for the coverage bounds, a "grid"
     *            AbstractGridCoverage2DReader attribute for the coverage data,
     *            and a "params" GeneralParameterValue[] attribute for storing
     *            optional coverage reading parameters.
     * @since TDK 3.0.0
     */
    public DefaultGridCoverageFeatureSource(DataStore dataStore, SimpleFeatureType featureType)
    {
        super();

        if (dataStore == null)
            throw new NullPointerException("dataStore can't be null");
        if (featureType == null)
            throw new NullPointerException("featureType can't be null");

        _dataStore = dataStore;
        _featureType = featureType;
    }

    /**
     * Constructor that builds a GridCoverageFeatureSource within a provided
     * DataStore and allows the setting of user-specified hints.
     * 
     * @param dataStore the DataStore that will contain the newly created
     *            FeatureSource.
     * @param featureType the SimpleFeatureType (schema) of the new
     *            GridCoverageFeatureSource, which should contain a "bounds"
     *            Polygon attribute for the coverage bounds, a "grid"
     *            AbstractGridCoverage2DReader attribute for the coverage data,
     *            and a "params" GeneralParameterValue[] attribute for storing
     *            optional coverage reading parameters.
     * @param hints a Hints objects containing configurations or settings that
     *            may influence the new object's behavior.
     * @since TDK 3.0.0
     */
    public DefaultGridCoverageFeatureSource(DataStore dataStore, SimpleFeatureType featureType,
            Set<RenderingHints.Key> hints)
    {
        super(hints);

        if (dataStore == null)
            throw new NullPointerException("dataStore can't be null");
        if (featureType == null)
            throw new NullPointerException("featureType can't be null");

        _dataStore = dataStore;
        _featureType = featureType;
    }

    /**
     * Generic static method for retrieving the GridCoverageReader wrapped by a
     * GridCoverageFeatureSource. This method is the one responsible for
     * retrieving the contents stored within the "grid" column.
     * 
     * @param featSource a FeatureSource that should have a "grid" column
     *            containing an AbstractGridCoverage2DReader.
     * @return the AbstractGridCoverage2DReader.
     * @throws DataAccessGridCoverageException if a "grid" column is not
     *             available, or if it does not contain a valid instance of
     *             AbstractGridCoverage2DReader.
     * @since TDK 3.0.0
     */
    protected static AbstractGridCoverage2DReader getCoverageReader(
            FeatureSource<SimpleFeatureType, SimpleFeature> featSource) throws DataAccessGridCoverageException
    {
        // retrieves the collection of Features contained in the FeatureSource
        FeatureCollection<SimpleFeatureType, SimpleFeature> featCollection = null;
        try
        {
            featCollection = featSource.getFeatures();
        }
        catch (IOException e)
        {
            throw new DataAccessGridCoverageException("Could not retrieve FeatureCollection from FeatureSource.");
        }

        // retrieves an iterator over the FeatureSource
        Iterator<SimpleFeature> featIterator = featCollection.iterator();
        if (!featIterator.hasNext())
            throw new DataAccessGridCoverageException(
                    "FeatureSource should contain one Feature with the wrapped GridCoverage.");

        // retrieves GridCoverage reader from the "grid" column
        Object readerObject = featIterator.next().getAttribute("grid");
        if (readerObject == null)
            throw new DataAccessGridCoverageException("FeatureSource should contain a 'grid' column.");
        if (!(readerObject instanceof AbstractGridCoverage2DReader))
            throw new DataAccessGridCoverageException(
                    "FeatureSource should contain an AbstractGridCoverage2DReader in its 'grid' column.");

        return (AbstractGridCoverage2DReader) readerObject;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.tecgraf.tdk.dataaccess.gridcoverage.GridCoverageFeatureSource#
     * getCoverageReader()
     */
    @Override
    public AbstractGridCoverage2DReader getCoverageReader() throws DataAccessGridCoverageException
    {
        return getCoverageReader(this);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.AbstractFeatureSource#getDataStore()
     */
    @Override
    public DataStore getDataStore()
    {
        return _dataStore;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.geotools.data.FeatureSource#addFeatureListener(org.geotools.data.
     * FeatureListener)
     */
    @Override
    public void addFeatureListener(FeatureListener listener)
    {
        // this is not ideal, but it is the most straightforward way of dealing
        // with this method
        // - DefaultGridCoverageDataStore currently extends AbstractDataStore,
        // in which case this implementation is straightforward
        // - It is not yet clear in which situation this listener will be
        // needed, and whether there will ever be DataStore with GridCoverages
        // that does not extend
        // AbstractDataStore but does need this feature
        // - If this solution is not satisfactory, the recommended solution
        // would be to develop a different implementation of
        // GridCoverageFeatureSource
        if (!(_dataStore instanceof AbstractDataStore))
            throw new UnsupportedOperationException(
                    "FeatureListenerManager is only available for DataStores that extend AbstactDataStore.");

        ((AbstractDataStore) _dataStore).listenerManager.addFeatureListener(this, listener);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.geotools.data.FeatureSource#removeFeatureListener(org.geotools.data
     * .FeatureListener)
     */
    @Override
    public void removeFeatureListener(FeatureListener listener)
    {
        // this is not ideal, but it is the most straightforward way of dealing
        // with this method
        // (see GridCoverageFeatureSourceImpl.addFeatureListener for a
        // discussion)
        if (!(_dataStore instanceof AbstractDataStore))
            throw new UnsupportedOperationException(
                    "FeatureListenerManager is only available for DataStores that extend AbstactDataStore.");

        ((AbstractDataStore) _dataStore).listenerManager.removeFeatureListener(this, listener);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.FeatureSource#getSchema()
     */
    @Override
    public SimpleFeatureType getSchema()
    {
        return _featureType;
    }
}
