package tecgraf.openbus;

import org.omg.CORBA.NO_PERMISSION;
import org.omg.CORBA.ORB;

import tecgraf.openbus.core.v2_0.OctetSeqHolder;
import tecgraf.openbus.core.v2_0.services.ServiceFailure;
import tecgraf.openbus.core.v2_0.services.access_control.AccessDenied;
import tecgraf.openbus.core.v2_0.services.access_control.LoginInfo;
import tecgraf.openbus.core.v2_0.services.access_control.LoginProcess;
import tecgraf.openbus.core.v2_0.services.access_control.MissingCertificate;
import tecgraf.openbus.core.v2_0.services.access_control.NoLoginCode;
import tecgraf.openbus.core.v2_0.services.access_control.UnknownBusCode;
import tecgraf.openbus.exception.AlreadyLoggedIn;
import tecgraf.openbus.exception.InvalidLoginProcess;

/**
 * Objeto que representa uma forma de acesso a um barramento.
 * <p>
 * Uma conexo representa uma forma de acesso a um barramento. Basicamente, uma
 * conexo  usada para representar uma identidade de acesso a um barramento. 
 * possvel uma aplicao assumir mltiplas identidades ao acessar um ou mais
 * barramentos criando mltiplas conexes para esses barramentos.
 * <p>
 * Para que as conexes possam ser efetivamente utilizadas elas precisam estar
 * autenticadas no barramento, que pode ser visto como um identificador de
 * acesso. Cada login possui um identificador nico e  autenticado em nome de
 * uma entidade, que pode representar um sistema computacional ou mesmo uma
 * pessoa. A funo da entidade  atribuir a responsabilidade s chamadas feitas
 * com aquele login.
 * <p>
 *  importante notar que a conexo s define uma forma de acesso, mas no 
 * usada diretamente pela aplicao ao realizar ou receber chamadas, pois as
 * chamadas ocorrem usando proxies e servants de um {@link ORB}. As conexes que
 * so efetivamente usadas nas chamadas do ORB so definidas atravs do
 * {@link OpenBusContext} associado ao ORB.
 * 
 * @author Tecgraf
 */
public interface Connection {

  /**
   * Recupera o {@link ORB} correspondente ao {@link OpenBusContext} a partir do
   * qual essa conexo foi criada.
   * 
   * @return o ORB
   */
  org.omg.CORBA.ORB orb();

  /**
   * Recupera o identificador do barramento ao qual essa conexo se refere.
   * 
   * @return o identificador do barramento.
   */
  String busid();

  /**
   * Recupera as Informaes do login dessa conexo ou <code>null</code> se a
   * conexo no est autenticada, ou seja, no tem um login vlido no
   * barramento.
   * 
   * @return as informaes do login.
   */
  LoginInfo login();

  /**
   * Efetua login de uma entidade usando autenticao por senha.
   * <p>
   * A autenticao por senha  validada usando um dos validadores de senha
   * definidos pelo adminsitrador do barramento.
   * 
   * @param entity Identificador da entidade a ser autenticada.
   * @param password Senha de autenticao no barramento da entidade.
   * 
   * @exception AlreadyLoggedIn A conexo j est autenticada.
   * @exception AccessDenied Senha fornecida para autenticao da entidade no
   *            foi validada pelo barramento.
   * @exception ServiceFailure Ocorreu uma falha interna nos servios do
   *            barramento que impediu a autenticao da conexo.
   */
  void loginByPassword(String entity, byte[] password) throws AccessDenied,
    AlreadyLoggedIn, ServiceFailure;

  /**
   * Efetua login de uma entidade usando autenticao por certificado.
   * <p>
   * A autenticao por certificado  validada usando um certificado de login
   * registrado pelo adminsitrador do barramento.
   * 
   * @param entity Identificador da entidade a ser autenticada.
   * @param privateKey Chave privada correspondente ao certificado registrado a
   *        ser utilizada na autenticao.
   * 
   * @exception AlreadyLoggedIn A conexo j est autenticada.
   * @exception AccessDenied A chave privada fornecida no corresponde ao
   *            certificado da entidade registrado no barramento indicado.
   * @exception MissingCertificate No h certificado para essa entidade
   *            registrado no barramento indicado.
   * @exception ServiceFailure Ocorreu uma falha interna nos servios do
   *            barramento que impediu a autenticao da conexo.
   */
  void loginByCertificate(String entity, PrivateKey privateKey)
    throws AlreadyLoggedIn, AccessDenied, MissingCertificate, ServiceFailure;

  /**
   * Inicia o processo de login por autenticao compartilhada.
   * <p>
   * A autenticao compartilhada permite criar um novo login compartilhando a
   * mesma autenticao do login atual da conexo. Portanto essa operao s
   * pode ser chamada enquanto a conexo estiver autenticada, caso contrrio a
   * exceo de sistema {@link NO_PERMISSION}[{@link NoLoginCode}]  lanada. As
   * informaes fornecidas por essa operao devem ser passadas para a operao
   * {@link #loginBySharedAuth(LoginProcess, byte[]) loginBySharedAuth} para
   * concluso do processo de login por autenticao compartilhada. Isso deve
   * ser feito dentro do tempo de lease definido pelo administrador do
   * barramento. Caso contrrio essas informaes se tornam invlidas e no
   * podem mais ser utilizadas para criar um login.
   * 
   * @param secret Segredo a ser fornecido na concluso do processo de login.
   * 
   * @return Objeto que represeta o processo de login iniciado.
   * 
   * @exception ServiceFailure Ocorreu uma falha interna nos servios do
   *            barramento que impediu a obteno do objeto de login e segredo.
   */
  LoginProcess startSharedAuth(OctetSeqHolder secret) throws ServiceFailure;

  /**
   * Efetua login de uma entidade usando autenticao compartilhada.
   * <p>
   * A autenticao compartilhada  feita a partir de informaes obtidas a
   * atravs da operao {@link #startSharedAuth(OctetSeqHolder)
   * startSharedAuth} de uma conexo autenticada.
   * 
   * @param process Objeto que represeta o processo de login iniciado.
   * @param secret Segredo a ser fornecido na concluso do processo de login.
   * 
   * @exception InvalidLoginProcess O LoginProcess informado  invlido, por
   *            exemplo depois de ser cancelado ou ter expirado.
   * @exception AlreadyLoggedIn A conexo j est autenticada.
   * @exception AccessDenied O segredo fornecido no corresponde ao esperado
   *            pelo barramento.
   * @exception ServiceFailure Ocorreu uma falha interna nos servios do
   *            barramento que impediu a autenticao da conexo.
   */
  void loginBySharedAuth(LoginProcess process, byte[] secret)
    throws AlreadyLoggedIn, InvalidLoginProcess, AccessDenied, ServiceFailure;

  /**
   * Efetua logout da conexo, tornando o login atual invlido.
   * <p>
   * Aps a chamada a essa operao a conexo fica desautenticada, implicando
   * que qualquer chamada realizada pelo ORB usando essa conexo resultar numa
   * exceo de sistema {@link NO_PERMISSION}[{@link NoLoginCode}] e chamadas
   * recebidas por esse ORB sero respondidas com a exceo
   * {@link NO_PERMISSION}[{@link UnknownBusCode}] indicando que no foi
   * possvel validar a chamada pois a conexo est temporariamente
   * desautenticada.
   * 
   * @return <code>true</code> se o processo de logout for concludo com xito e
   *         <code>false</code> se a conexo j estiver desautenticada (login
   *         invlido) ou se houver uma falha durante o processo remoto do
   *         logout.
   * 
   * @exception ServiceFailure Ocorreu uma falha interna nos servios do
   *            barramento durante chamada ao remota.
   */
  boolean logout() throws ServiceFailure;

  /**
   * Callback a ser chamada quando o login atual se tornar invlido.
   * <p>
   * Esse atributo  utilizado para definir um objeto que implementa uma
   * interface de callback a ser chamada sempre que a conexo receber uma
   * notificao de que o seu login est invlido. Essas notificaes ocorrem
   * durante chamadas realizadas ou recebidas pelo barramento usando essa
   * conexo. Um login pode se tornar invlido caso o administrador
   * explicitamente o torne invlido ou caso a thread interna de renovao de
   * login no seja capaz de renovar o lease do login a tempo. Caso esse
   * atributo seja <code>null</code>, nenhum objeto de callback  chamado na
   * ocorrncia desse evento.
   * <p>
   * Durante a execuo dessa callback um novo login pode ser restabelecido.
   * Neste caso, a chamada do barramento que recebeu a notificao de login
   * invlido  refeita usando o novo login, caso contrrio, a chamada original
   * lana a exceo de de sistema {@link NO_PERMISSION}[{@link NoLoginCode}].
   * 
   * @param callback Objeto que implementa a interface de callback a ser chamada
   *        ou <code>null</code> caso nenhum objeto deva ser chamado na
   *        ocorrncia desse evento.
   */
  void onInvalidLoginCallback(InvalidLoginCallback callback);

  /**
   * Recupera a callback a ser chamada sempre que o login se torna invlido.
   * 
   * @return a callback ou <code>null</code> caso ela no exista.
   */
  InvalidLoginCallback onInvalidLoginCallback();

}