package csbase.client.applications.imageviewer.effects;

import java.awt.RenderingHints;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.BufferedImageOp;
import java.awt.image.ColorModel;
import java.awt.image.Raster;
import java.awt.image.WritableRaster;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import tecgraf.javautils.gui.GUIUtils;
import csbase.client.applications.imageviewer.ImageViewer;

/**
 * Efeito de blur implementado com filtro de Gauss em uma matrix 3x3
 * 
 * http://en.wikipedia.org/wiki/Gaussian_filter
 * 
 * @see AbstractMatrixFilter
 * 
 * @author TecGraf/PUC-Rio
 * 
 */
public class BlurEffect extends AbstractEffect {

  /**
   * Classe para auxiliar a multiplicao de matrizes para filtro de imagem.
   * 
   * @see Matrix
   * 
   * @author TecGraf/PUC-Rio
   * 
   */
  abstract class AbstractMatrixFilter {

    /**
     * Classe que define uma matriz quadrada de nmero mpar de linhas e colunas
     * 
     * A matriz  um array 2 dimenses, onde a primeira representando as
     * posies de X e a segunda representando as posies de Y
     * 
     * Exemplo da funcionalidade de multiplicao provida pela classe:
     * 
     * <pre>
     *  _ _ _     _ _ _     __ __ __
     * |1|T|1|   |0|Z|0|   | 0|TZ| 0| 
     * |T|X|T| * |Z|Y|Z| = |TZ|XY|TZ| 
     * |1|T|1|   |0|Z|0|   | 0|TZ| 0|
     * </pre>
     * 
     * Note que a multiplicaes  elemento a elemento e no como se espera de
     * multiplcao de matrizes em lgebra linear.
     * 
     * @author TecGraf/PUC-Rio
     * 
     */
    class Matrix {

      /**
       * Array de duas dimenses que guarda os valores da matriz
       */
      final private double[][] matrix2dArray;

      /**
       * Constri uma matriz com um nmero mpar de linhas e colunas
       * 
       * @param oddRowsCols nmero mpar de nmeros e colunas
       */
      Matrix(int oddRowsCols) {
        if (oddRowsCols % 2 == 0) {
          throwNotOddNumberIllegalArgumentException();
        }

        matrix2dArray = newSquared2dArray(oddRowsCols);
      }

      /**
       * Constri uma matriz para manipular a matriz 2D passada por parmetro
       * 
       * NOTA: _No_  feita cpia da matrix 2D passada
       * 
       * @param matrix2d vetor 2D a ser usado como referncia pela matriz
       *        retornada
       */
      Matrix(double[][] matrix2d) {
        squarenessCheck(matrix2d);
        if (oddRowCheck(matrix2d) == false || oddColumnCheck(matrix2d) == false) {
          throwNotOddNumberIllegalArgumentException();
        }
        matrix2dArray = matrix2d;
      }

      /**
       * Constri um vetor double de duas dimenses para guardar os valores da
       * matriz
       * 
       * @param n nmero de colunas e linhas
       * @return vetor double de duas dimenses
       */
      private double[][] newSquared2dArray(final int n) {
        double[][] resultMatrix = new double[n][];
        for (int i = 0; i < n; ++i) {
          resultMatrix[i] = new double[n];
        }
        return resultMatrix;
      }

      /**
       * Retorna o nmero de linhas na matriz
       * 
       * @return o nmero de linhas na matriz
       */
      int getRowNum() {
        return matrix2dArray.length;
      }

      /**
       * Retorna soma de todos os valores da matriz
       * 
       * @return soma de todos os valores da matriz
       */
      double sumValues() {
        double totalSum = 0;
        for (int x = 0; x < getRowNum(); ++x) {
          for (int y = 0; y < getRowNum(); ++y) {
            totalSum += getValue(x, y);
          }
        }
        return totalSum;
      }

      /**
       * Testa para saber se a matriz 2D passada tem o nmero de colunas mpar
       * 
       * @param matrix2dArray
       * @return true se o nmero de colunas for mpar
       */
      private boolean oddColumnCheck(double[][] matrix2dArray) {
        return matrix2dArray.length % 2 == 1;
      }

      /**
       * Testa para saber se a matriz 2D passada tem o nmero de linhas
       * 
       * @param matrix2dArray
       * @return true se o nmero de linhas for mpar
       */
      private boolean oddRowCheck(double[][] matrix2dArray) {
        for (int i = 0; i < matrix2dArray.length; ++i) {
          if (matrix2dArray[i].length % 2 == 0) {
            return false;
          }
        }
        return true;
      }

      /**
       * Levanta uma exceo de IllegalArgumentException descrevendo que a
       * matriz deve ser quadrada e ter um nmero mpar de linhas e colunas
       */
      private void throwNotOddNumberIllegalArgumentException() {
        throw new IllegalArgumentException(
          "A matriz deve ser quadrada e ter um nmero mpar de linhas e colunas");
      }

      /**
       * Define o valor de value ao elemento de coluna x e linha y da matriz
       * 
       * @param x coluna da matriz ser definido o valor
       * @param y linha a ser definido o valor
       * @param value valor a ser definido
       */
      private void setValue(int x, int y, double value) {
        matrix2dArray[x][y] = value;
      }

      /**
       * Retorna o valor do elemento da posio de coluna de ndice x e linha de
       * ndice y na matriz
       * 
       * @param x ndice x da coluna da matriz do valor a ser retornado
       * @param y ndice y da coluna da matriz do valor a ser retornado
       * @return valor da posio ndices x e y da matriz
       */
      double getValue(int x, int y) {
        return matrix2dArray[x][y];
      }

      /**
       * Multiplica os valores da matriz com outra usando as posies
       * correspondentes dos elementos, e no como na multiplicao de matrizes
       * como  conhecida na matemtica
       * 
       * @param otherMatrix outra matriz que ser multiplicada a esta
       * @return nova instncia da matriz com o resultado das multiplicaes
       */
      Matrix multiply(Matrix otherMatrix) {
        if (getRowNum() != otherMatrix.getRowNum()) {
          throw new IllegalArgumentException(
            "Matrizes precisam ser do mesmo temanho para a multiplicao componente a componente");
        }
        Matrix resultMatrix = new Matrix(getRowNum());
        for (int x = 0; x < getRowNum(); ++x) {
          for (int y = 0; y < getRowNum(); ++y) {
            final double result = getValue(x, y) * otherMatrix.getValue(x, y);
            resultMatrix.setValue(x, y, result);
          }
        }
        return resultMatrix;
      }
    }

    /**
     * Testa um vetor 2D para saber se ele  quadrado
     * 
     * @param shouldBeSquaredMatrix vetor 2D a ser testado
     */
    private void squarenessCheck(double[][] shouldBeSquaredMatrix) {
      final int matrixTotalColumns = shouldBeSquaredMatrix.length;
      for (int i = 0; i < matrixTotalColumns; ++i) {
        final int matrixTotalRows = shouldBeSquaredMatrix[i].length;
        if (matrixTotalColumns != matrixTotalRows) {
          throw new IllegalArgumentException("Vetor 2D passado no  quadrada");
        }
      }
    }

    /**
     * Constri uma matriz com os valores dos pixels do raster centrados na
     * coordenada x, y e pixels adjacentes. O tamanho da matriz  definido por
     * totalRowCol
     * 
     * @param raster Raster do qual ser extrada a matriz
     * @param x coordenada x do pixel do centro da matriz
     * @param y coordenada y do pixel do centro da matriz
     * @param componentIndex ndice da componente no Raster
     * @param totalRowCol nmero de linhas e colunas da matriz a ser gerada
     * @return matriz com os valores do pixel do raster
     */
    protected Matrix matrixFrom(Raster raster, int componentIndex, int x,
      int y, int totalRowCol) {
      Matrix resultMatrix = new Matrix(totalRowCol);
      final int edgeOffset = (totalRowCol - 1) / 2;
      int xOffset = -edgeOffset;
      for (int mx = 0; mx < totalRowCol; ++mx) {
        int yOffset = -edgeOffset;
        for (int my = 0; my < totalRowCol; ++my) {
          resultMatrix.setValue(mx, my, raster.getSampleDouble(xOffset + x,
            yOffset + y, componentIndex));
          ++yOffset;
        }
        ++xOffset;
      }
      return resultMatrix;
    }

    /**
     * Testa para saber se o pixel passado est na borda do raster usando como
     * referncia a matriz de filtro que ser aplicada
     * 
     * @param raster Raster que ser manipulado pela matriz
     * @param x coordenada x do pixel a ser testado
     * @param y coordenada y do pixel a ser testado
     * @param matrix matriz de filtro para ser usada como refer
     */
    private void borderPixelCheck(Raster raster, int x, int y, Matrix matrix) {
      final int edgeOffset = (matrix.getRowNum() - 1) / 2;
      if (x - edgeOffset < 0 || y - edgeOffset < 0
        || x + edgeOffset >= raster.getWidth()
        || y + edgeOffset >= raster.getHeight()) {
        throw new IllegalArgumentException(
          "Pixel passado na borda do raster e este mtodo no suporta valores de pixel na borda");
      }
    }
  }

  /**
   * Operador do filtro de Gauss
   * 
   * @author TecGraf/PUC-Rio
   * 
   */
  class GaussOperator extends AbstractMatrixFilter implements BufferedImageOp {

    /**
     * Intensidade do filtro, 0  sem efeito e 100  o maior efeito suportado
     * por este filtro
     */
    final private double intensityFactor;

    /**
     * Cria um operador de filtro de Gauss com uma intensidade de 0 a 100
     * 
     * @param instensity de 0 a 100
     */
    GaussOperator(int intensity) {
      if (intensity < 0 || intensity > 100) {
        throw new IllegalArgumentException("Intensidade deve ser de 0 a 100");
      }

      intensityFactor = (double) intensity / 100;

    }

    /**
     * Aplica o operador de Gauss em uma matriz 3x3 na BufferedImage passada
     * como src retornando a BufferedImage resultado.
     * 
     * Se dest for null, cria uma BufferedImage nova para ser retornada
     * 
     * @param src BufferedImage origem
     * @param dest BufferedImage destino
     * @return BufferedImage destino ou uma nova caso dest seja null
     */
    private BufferedImage applyGaussOp(BufferedImage src, BufferedImage dest) {
      Raster raster = src.getData();
      int width = raster.getWidth();
      int height = raster.getHeight();

      if (dest == null) {
        dest = new BufferedImage(width, height, src.getType());
      }
      WritableRaster writableRaster = dest.getRaster();

      double max = 0;

      int numComponents = src.getColorModel().getNumColorComponents();
      Matrix gaussMatrix = gaussMatrixFromIntensity(intensityFactor);
      for (int x = 1; x < width - 1; ++x) {
        for (int y = 1; y < height - 1; ++y) {
          double[] components = new double[numComponents];
          for (int c = 0; c < components.length; ++c) {
            Matrix rasterMatrix =
              matrixFrom(raster, c, x, y, gaussMatrix.getRowNum());
            Matrix multResult = rasterMatrix.multiply(gaussMatrix);
            components[c] = multResult.sumValues();
          }
          writableRaster.setPixel(x, y, components);
        }
      }

      return dest;
    }

    /**
     * Retorna uma matriz de Gauss de acordo com a intensidade
     * 
     * 0 - sem efeito 100 - efeito mximo suportado por esta implementao
     * 
     * @param intensityFactor
     * @return uma matriz de Gauss de acordo com a intensidade
     */
    private Matrix gaussMatrixFromIntensity(double intensityFactor) {
      double[][] matrixRet = new double[3][];
      for (int x = 0; x < matrixRet.length; ++x) {
        matrixRet[x] = new double[matrixRet.length];
        for (int y = 0; y < matrixRet.length; ++y) {
          matrixRet[x][y] = 1;
        }
      }

      /**
       * Center pixel
       * 
       * <pre>
       *  _ _ _ 
       * |_|_|_|
       * |_|X|_|
       * |_|_|_|
       * </pre>
       */
      matrixRet[1][1] *= 1. - intensityFactor;

      /**
       * Cross pixels
       * 
       * <pre>
       *  _ _ _
       * |_|X|_|
       * |X|_|X|
       * |_|X|_|
       * </pre>
       */
      matrixRet[0][1] *= intensityFactor;
      matrixRet[1][0] *= intensityFactor;
      matrixRet[1][2] *= intensityFactor;
      matrixRet[2][1] *= intensityFactor;

      /**
       * Diagonal edges pixels
       * 
       * <pre>
       * _ _ _ 
       * |X|_|X|
       * |_|_|_|
       * |X|_|X|
       * </pre>
       */
      matrixRet[0][0] *= intensityFactor * intensityFactor;
      matrixRet[0][2] *= intensityFactor * intensityFactor;
      matrixRet[2][0] *= intensityFactor * intensityFactor;
      matrixRet[2][2] *= intensityFactor * intensityFactor;

      double sum = 0;
      for (int x = 0; x < matrixRet.length; ++x) {
        for (int y = 0; y < matrixRet.length; ++y) {
          sum += matrixRet[x][y];
        }
      }

      for (int x = 0; x < matrixRet.length; ++x) {
        for (int y = 0; y < matrixRet.length; ++y) {
          matrixRet[x][y] /= sum;
        }
      }

      return new Matrix(matrixRet);
    }

    @Override
    public BufferedImage createCompatibleDestImage(BufferedImage src,
      ColorModel destCM) {
      return null;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public BufferedImage filter(BufferedImage src, BufferedImage dest) {
      return applyGaussOp(src, dest);
    }

    @Override
    public Rectangle2D getBounds2D(BufferedImage src) {
      return null;
    }

    @Override
    public Point2D getPoint2D(Point2D srcPt, Point2D dstPt) {
      return null;
    }

    @Override
    public RenderingHints getRenderingHints() {
      return null;
    }
  }

  /**
   * Slider que regular a intensidade do filtro
   */
  final JSlider intensitySlider = new JSlider(JSlider.HORIZONTAL);

  /**
   * Constri o efeito de gauss dado a aplicao
   * 
   * @param application aplicao qual pertence o filtro
   */
  public BlurEffect(ImageViewer application) {
    super(application);
    intensitySlider.addChangeListener(new ChangeListener() {

      @Override
      public void stateChanged(ChangeEvent arg0) {
        if (intensitySlider.isEnabled()) {
          updatePreview();
        }
      }
    });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JPanel getParameterPanel() {
    JLabel intensityLabel = new JLabel(getString("BlurEffect.intensity.label"));
    JComponent[][] row = { { intensityLabel, intensitySlider } };
    return GUIUtils.createBasicGridPanel(row);
  }

  /**
   * Aplica o filtro de Gauss na BufferedImage passada por parmetro e retorna
   * uma nova.
   * 
   * @return BufferedImage com o filtro de Gauss aplicado
   * 
   */
  @Override
  protected BufferedImage transformImage(BufferedImage image) {
    return new GaussOperator(intensitySlider.getValue()).filter(image, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void resetParameters() {
    intensitySlider.setValue(0);
  }
}
