/*
 * $Id:$
 */

package csbase.client.util.gui.log;

import java.awt.Color;
import java.awt.Font;
import java.awt.GridBagLayout;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ScrollPaneConstants;

import tecgraf.javautils.gui.GBC;
import csbase.client.util.event.EventListener;
import csbase.client.util.gui.LinedTextArea;
import csbase.client.util.gui.log.LogPanelReloader.StatusChangedEvent;
import csbase.client.util.gui.log.enums.FontSize;
import csbase.client.util.gui.log.enums.FontType;

/**
 * rea de texto a ser utilizada pelo painel de log para mostrar a pgina
 * corrente.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class LogPanelTextArea extends JPanel {

  /**
   * Fonte padro da rea de texto.
   */
  private static final Font FONT_DEFAULT = new Font(Font.MONOSPACED,
    Font.PLAIN, 12);

  /**
   * Pede ao {@link LogPanel}, atravs do mtodo
   * {@link LogPanel#getString(String, Object...)}, um texto com a chave formada
   * por "TextArea."+tag.
   * 
   * @param tag o tag
   * @param args os argumentos.
   * 
   * @return um texto sem espaos antes nem depois caso a chave tenha sido
   *         encontrada ou nulo caso contrrio.
   */
  private static final String getString(final String tag, final Object... args) {
    return LogPanel.getString("TextArea." + tag, args);
  }

  /**
   * Textarea aonde o texto  exibido.
   */
  private LinedTextArea textArea;

  /**
   * Texto do cabealho.
   */
  private JLabel header;

  /**
   * Consulta o tamanho da fonte
   * 
   * @return o tamanho da fonte
   */
  public final FontSize getFontSize() {
    final Font font = textArea.getFont();
    final int isize = font.getSize();
    return FontSize.getFontSizeByInt(isize);
  }

  /**
   * Consulta o tipo de fonte.
   * 
   * @return o tipo de fonte.
   */
  public final FontType getFontType() {
    final Font font = textArea.getFont();
    final String name = font.getName();
    return FontType.getFontTypeByName(name);
  }

  /**
   * Posiciona o caret para o fim da pgina.
   */
  public final void goToHead() {
    final String text = textArea.getText();
    if (text == null) {
      return;
    }
    textArea.setCaretPosition(0);
  }

  /**
   * Posiciona o caret para o fim da pgina.
   */
  public final void goToTail() {
    final String text = textArea.getText();
    if (text == null) {
      return;
    }
    final int textLength = text.length();
    if (textLength == 0) {
      return;
    }
    textArea.setCaretPosition(textLength - 1);
  }

  /**
   * Initicializa os componentes.
   * 
   * @param logPanel o painel de log.
   * @param scrollable indica se o painel deve ser "rolvel".
   * @param headerText texto do cabealho.
   */
  private void initializeComponents(final AutoReloadable logPanel,
    boolean scrollable, String headerText) {
    textArea = new LinedTextArea();
    textArea.setEditable(true);
    textArea.setFont(LogPanelTextArea.FONT_DEFAULT);
    textArea.setEditable(false);
    textArea.setColumns(132);

    if (headerText != null) {
      header = new JLabel(headerText);
      header.setFont(LogPanelTextArea.FONT_DEFAULT);
    }

    /* Atualiza a cor de fundo da textArea. */
    updateBgColor(logPanel.getReloader().isRunning());
    logPanel.getReloader().addStatusChangedEventListener(
      new EventListener<LogPanelReloader.StatusChangedEvent>() {
        @Override
        public void eventFired(final StatusChangedEvent event) {
          updateBgColor(event.isRunning());
        }
      });

    setLayout(new GridBagLayout());

    int i = 0;
    if (header != null) {
      add(header, new GBC(0, i++).horizontal());
    }

    if (scrollable) {
      JScrollPane scrollPane = new JScrollPane(textArea);
      final int mode = ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS;
      scrollPane.setVerticalScrollBarPolicy(mode);
      add(scrollPane, new GBC(0, i).both());
    }
    else {
      add(textArea, new GBC(0, i).both());
    }
  }

  /**
   * Ajusta o tamanho de fonte
   * 
   * @param size o tamanho
   */
  public final void setFontSize(final FontSize size) {
    final Font oldFont = textArea.getFont();
    final int isize = size.getSize();
    final int style = oldFont.getStyle();
    final String name = oldFont.getName();

    final Font newFont = new Font(name, style, isize);
    textArea.setFont(newFont);
    if (header != null) {
      header.setFont(newFont);
    }
  }

  /**
   * Ajusta o tipo de fonte
   * 
   * @param type o tipo
   */
  public final void setFontType(final FontType type) {
    final Font oldFont = textArea.getFont();
    final int size = oldFont.getSize();
    final int style = oldFont.getStyle();
    final String typeName = type.getTypeName();

    final Font newFont = new Font(typeName, style, size);
    textArea.setFont(newFont);
    if (header != null) {
      header.setFont(newFont);
    }

  }

  /**
   * Atualiza o prefixo de acordo com a pgina corrente.
   * 
   * @param page a pgina
   */
  public void setPrefix(final long page) {
    if (LogPanel.FIRST_PAGE >= page) {
      textArea.setPrefix(null);
    }
    else {
      final String prefix = LogPanelTextArea.getString("prefix", page);
      textArea.setPrefix(prefix);
    }
  }

  /**
   * Atribui o texto a ser mostrado.
   * 
   * @param text o texto
   */
  final public void setText(final String text) {
    textArea.setText(text);
  }

  /**
   * Atualiza a cor de fundo levando em conta o fato da auto-recarga do logPanel
   * estar ligada ou no.
   * 
   * @param reloading indicativo de reload.
   */
  private void updateBgColor(final boolean reloading) {
    final Color on = new Color(200, 200, 200);
    final Color off = new Color(220, 220, 220);
    final Color clr = reloading ? on : off;
    textArea.setBackground(clr);
    if (header != null) {
      header.setOpaque(true);
      header.setBackground(clr);
    }
  }

  /**
   * Construtor.
   * 
   * @param logPanel painel que detm esta rea de texto.
   * @param scrollable indica se o painel deve ser "rolvel".
   * @param headerText texto do cabealho.
   */
  public LogPanelTextArea(final AutoReloadable logPanel, boolean scrollable,
    String headerText) {
    super();
    initializeComponents(logPanel, scrollable, headerText);
    setFontType(FontType.MONOSPACED);
  }

  /**
   * Construtor.
   * 
   * @param logPanel painel que detm esta rea de texto.
   * @param scrollable indica se o painel deve ser "rolvel".
   */
  public LogPanelTextArea(final AutoReloadable logPanel, boolean scrollable) {
    this(logPanel, scrollable, null);
  }

}
