package csbase.client.applications.statsviewer;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import csbase.logic.eventlogservice.LogsInfo;

/**
 * Classe que representa uma viso dos arquivos de log obtidos do servidor,
 * conforme as configuraes de exibio dos usurios.
 *
 * @author Tecgraf/PUC-Rio
 */
public class LogsView {

  /**
   * Tipos de logs com informaes de estatsticas.
   */
  public enum LogsDataType {

    /**
     * Estatsticas de Acesso ao Sistema.
     */
    LoginData(new String[][] { { "client", "Desktop" } }),

    /**
     * Estatsticas da Utilizao das Aplicaes.
     */
    ApplicationsData(new String[][] { { "client", "Applications" } }),

    /**
     * Estatsticas da Execuo de Comandos (simples e de fluxo).
     */
    ExecutionsData(new String[][] { { "server", "commands", "simple" },
        { "server", "commands", "flow" } });

    /**
     * Fila que indicam o path dos arquivos deste tipo no servio de logs.
     */
    public String[][] queues;

    /**
     * @param queues filas de armazenamento
     */
    private LogsDataType(String[][] queues) {
      this.queues = queues;
    }
  }

  /**
   * Tipo de exibio para datas.
   */
  public enum DateViewType {

    /** Por ano */
    YEAR,

    /** Por Ms */
    MONTH,

    /** Por Dia */
    DAY;

    /**
     * Obtm a viso da data, conforme o tipo.
     *
     * @param date string com data no formato "yyyy/MM/dd"
     * @return substring que representa ano, ano/ms ou ano/mes/dia
     */
    public String getDateByView(String date) {
      switch (this) {
        case YEAR:
          return date.substring(0, 4);
        case MONTH:
          return date.substring(0, 7);
        case DAY:
        default:
          return date.substring(0, 10);
      }
    }

  }

  /**
   * Filtro das aplicaes selecionadas para exibio.
   */
  private List<String> selectedApplications;

  /**
   * Filtro dos algoritmos selecionados para exibio.
   */
  private List<String> selectedAlgorithms;

  /**
   * Filtro do agregador de datas selecionado para exibio.
   */
  private DateViewType selectedDateView;

  /**
   * Tipo de estatstica de interesse selecionado.
   */
  private LogsDataType selectedLogsType;

  /**
   * Dados obtidos dos logs para um perodo especfico.
   */
  private LogsInfo info;

  /**
   * Construtor.
   */
  public LogsView() {
    this(DateViewType.YEAR, LogsDataType.LoginData);
  }

  /**
   * Construtor.
   *
   * @param dateView viso para exibir colunas de data
   * @param logsType tipo de log relacionado
   */
  public LogsView(DateViewType dateView, LogsDataType logsType) {
    this.selectedDateView = dateView;
    this.selectedLogsType = logsType;
    this.selectedApplications = new ArrayList<String>();
    this.selectedAlgorithms = new ArrayList<String>();
    this.info = new LogsInfo();
  }

  /**
   * Limpa os dados da viso dos logs, mantendo configurao de exibio.
   */
  public void clear() {
    this.selectedApplications = new ArrayList<String>();
    this.selectedAlgorithms = new ArrayList<String>();
    this.info = new LogsInfo();
  }

  /**
   * @param applications lista de aplicaes
   */
  public void setSelectedApplications(List<String> applications) {
    this.selectedApplications = applications;
  }

  /**
   * @param algorithms lista de algoritmos
   */
  public void setSelectedAlgorithms(List<String> algorithms) {
    this.selectedAlgorithms = algorithms;
  }

  /**
   * Modifica o resultado dos arquivos de logs.
   *
   * @param info informaes de log
   */
  public void setInfo(LogsInfo info) {
    this.info = info;
  }

  /**
   * Modifica o tipo de log.
   *
   * @param type tipo de arquivo de log
   */
  public void setLogType(LogsDataType type) {
    this.selectedLogsType = type;
  }

  /**
   * Modifica a viso para agrupar as datas.
   *
   * @param dateViewType viso para exidir datas
   */
  public void setDateViewType(DateViewType dateViewType) {
    this.selectedDateView = dateViewType;
  }

  /**
   * @return tipo de log atual
   */
  public LogsDataType getLogType() {
    return selectedLogsType;
  }

  /**
   * @return informaes de log atuais
   */
  public LogsInfo getInfo() {
    return info;
  }

  /**
   * Obtm a tabela de resultados, conforme o tipo de log atual.
   *
   * @return tabela de dados para o tipo de log
   */
  public String[][] getTable() {
    List<String[]> lines;
    switch (selectedLogsType) {
      case LoginData:
        lines = info.getLoginTable();
        break;
      case ApplicationsData:
        lines = info.getApplicationsTable();
        break;
      case ExecutionsData:
        lines = info.getExecutionsTable();
      default:
        return null;
    }
    return lines.toArray(new String[lines.size()][]);
  }

  // Mtodos auxiliares para acessar informaes de estatsticas coletadas
  // //////////////////////////////////////////////////////////////

  /**
   * Obtm um mapa com o nmero total de utilizaes de aplicao, tendo a data
   * como chave. Considera na pesquisa apenas as execues das aplicaes
   * previamente selecionadas em infoview.
   *
   * @return mapa com utilizaes das aplicaes <DATA, <APLICACAO, TOTAL>>
   */
  public Map<String, Map<String, Integer>> getApplicationsMap() {

    Map<String, Map<String, Integer>> map =
      new TreeMap<String, Map<String, Integer>>();

    for (String[] line : info.getApplicationsTable()) {

      String strDate = selectedDateView.getDateByView(line[0]);
      String appId = line[1];
      String appDesc = line[2];

      if (!selectedApplications.contains(appDesc)) {
        continue;
      }

      Map<String, Integer> currentMap = map.get(strDate);
      if (currentMap == null) {
        currentMap = new TreeMap<String, Integer>();
        map.put(strDate, currentMap);
      }

      Integer countApplications = currentMap.get(appDesc);
      if (countApplications == null) {
        countApplications = 0;
      }
      currentMap.put(appDesc, countApplications + 1);

    }
    return map;
  }

  /**
   * Obtm um mapa com o nmero total de execues por algoritimo, tendo a data
   * como chave. Considera na pesquisa apenas as execues dos algoritmos
   * previamente selecionados em infoview.
   *
   * @return mapa com execues dos algoritmos <DATA, <ALGORITMO, TOTAL>>
   */
  public Map<String, Map<String, Integer>> getAlgorithmsByDate() {
    Map<String, Map<String, Integer>> map =
      new TreeMap<String, Map<String, Integer>>();

    for (String[] line : info.getExecutionsTable()) {

      String strDate = selectedDateView.getDateByView(line[0]);
      String algorithm = line[1];

      if (!selectedAlgorithms.contains(algorithm)) {
        continue;
      }

      Map<String, Integer> currentMap = map.get(strDate);
      if (currentMap == null) {
        currentMap = new TreeMap<String, Integer>();
        map.put(strDate, currentMap);
      }

      Integer countApp = currentMap.get(algorithm);
      if (countApp == null) {
        countApp = 0;
      }
      currentMap.put(algorithm, countApp + 1);

    }
    return map;

  }

  /**
   * Obtm um mapa com o nmero total de execues, tendo a data como chave.
   *
   * @return mapa com execues dos algoritmos <DATA, TOTAL>
   */
  public Map<String, Integer> getExecutionsByDate() {
    Map<String, Integer> map = new TreeMap<String, Integer>();
    for (String[] line : info.getExecutionsTable()) {
      String strDate = selectedDateView.getDateByView(line[0]);
      Integer value = map.get(strDate);
      if (value == null) {
        value = 0;
      }
      map.put(strDate, value + 1);
    }
    return map;
  }

  /**
   * Obtm um mapa com o nmero total de execues por data (chave). Agrupa as
   * informaes de data conforme previamente configurado em infoview.
   *
   * @return mapa com nmero de logins por <DATA, TOTAL>
   */
  public Map<String, Integer> getLoginsByDate() {
    Map<String, Integer> map = new TreeMap<String, Integer>();
    for (String[] line : info.getLoginTable()) {
      String strDate = selectedDateView.getDateByView(line[0]);
      Integer value = map.get(strDate);
      if (value == null) {
        value = 0;
      }
      map.put(strDate, value + 1);
    }
    return map;
  }

  /**
   * Consulta os totais de execues por algoritmos. Considera na pesquisa
   * apenas as execues dos algoritmos selecionados pelo usurio. Retorna no
   * mximo os 10 maiores resultados.
   *
   * @return mapa ordenado com nmero de execues por algoritmo at 10 valores
   */
  public Map<String, Integer> getTop10AlgorithmExecutions() {
    Map<String, Integer> executionsByAlgorithmMap =
      info.getExecutionsByAlgorithm(selectedAlgorithms);
    return sortByValue(executionsByAlgorithmMap, 10);
  }

  /**
   * Consulta os totais de execues por usurios. Considera na pesquisa apenas
   * as execues dos algoritmos selecionados pelo usurio. Retorna no mximo os
   * 10 maiores resultados.
   *
   * @return mapa ordenado com nmero de execues por usurio at 10 valores
   */
  public Map<String, Integer> getTop10UserExecutions() {
    Map<String, Integer> executionsByUserMap =
      info.getExecutionsByUser(selectedAlgorithms);
    return sortByValue(executionsByUserMap, 10);
  }

  /**
   * Consulta os totais de logins por usurios. Retorna no mximo os 10 maiores
   * nmeros de acesso.
   *
   * @return mapa ordenado com nmero de logins por usurio at 10 valores
   */
  public Map<String, Integer> getTop10UserLogins() {
    Map<String, Integer> loginsByUserMap = info.getLoginsByUser();
    return sortByValue(loginsByUserMap, 10);
  }

  /**
   * Retorna um mapa ordenado por valor, em ordem decrescente. O novo mapa
   * obtido contm apenas os n maiores valores encontrados do mapa de origem
   * aps sua ordenao.
   *
   * @param map mapa de origem que se deseja ordenar por valor
   * @param n total de valores que se deseja obter aps ordenao
   *
   * @return mapa ordenado por valor, com no n valores
   */
  private Map<String, Integer> sortByValue(Map<String, Integer> map, int n) {

    // ordena mapa por valor
    List<Map.Entry<String, Integer>> entries =
      new ArrayList<Map.Entry<String, Integer>>(map.entrySet());
    Collections.sort(entries, new Comparator<Map.Entry<String, Integer>>() {
      @Override
      public int compare(Map.Entry<String, Integer> a,
        Map.Entry<String, Integer> b) {
        return -(a.getValue().compareTo(b.getValue()));
      }
    });

    // obtm apenas os resultados dentro do limite pesquisado
    Map<String, Integer> sortedMap = new LinkedHashMap<String, Integer>();
    int size = 0;
    for (Map.Entry<String, Integer> entry : entries) {
      sortedMap.put(entry.getKey(), entry.getValue());
      size++;
      if (size >= n) {
        break;
      }
    }

    return sortedMap;
  }

}
