package csbase.client.applications.statsviewer;

import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Observer;

import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JPanel;
import javax.swing.JSplitPane;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationAboutAction;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.statsviewer.LogsView.DateViewType;
import csbase.client.applications.statsviewer.LogsView.LogsDataType;
import csbase.client.remote.srvproxies.EventLogProxy;
import csbase.client.util.csvpanel.CSVFileUtils;
import csbase.client.util.filechooser.ClientLocalFileChooserUtil;
import csbase.client.util.filechooser.util.ClientFileChooserUtils;
import csbase.logic.ClientFile;
import csbase.logic.eventlogservice.LogsInfo;

/**
 * Aplicao para Visualizao das Estatsticas do Sistema. Faz a leitura dos
 * dados de auditagem gerados pelo Servio de Log de Eventos e exibe grficos
 * com estatsticas de acesso ao sistema, estatsticas de uso das aplicaes e
 * estatsticas de execuo de comandos.
 */
public class StatsViewer extends Application {

  /**
   * Lista de observadores dos dados da aplicao.
   */
  private List<Observer> observers;

  /**
   * Viso dos dados de log que so exibidos pela aplicao. Contm o resultado
   * da pesquisa e as configuraes de como so exibidos.
   */
  private LogsView logsView;

  /**
   * Painel com parmetros para configurar busca nos logs.
   */
  private SearchPanel searchPanel;

  /**
   * Painel com filtro para exibio dos dados no grfico, sua exibio varia
   * conforme a busca que for especificada em {@link SearchPanel}.
   */
  private FilterPanel filterPanel;

  /**
   * Painel que exibe os grficos, sua exibio varia conforme a busca que for
   * especifica em {@link SearchPanel}.
   */
  private GraphicTabsPanel graphicsTabPanel;

  /**
   * Construtor.
   *
   * @param id identificador da aplicao
   */
  public StatsViewer(String id) {
    super(id);
    this.logsView = new LogsView();
    this.observers = new ArrayList<Observer>();
    buildInterface();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addObserver(Observer o) {
    observers.add(o);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void notifyObservers() {
    for (Observer observer : observers) {
      observer.update(this, logsView);
    }
  }

  /**
   * Limpa os resultados da busca corrente e notifica os observadores.
   */
  public void clearInfo() {
    logsView.clear();
    filterPanel.updateInfo(logsView.getInfo());
    notifyObservers();
  }

  /**
   * Realiza uma busca nos logs com o perodo fornecido e as configuraes de
   * fila de armazenamentos atualmente selecionadas.
   *
   * @param start data de incio
   * @param end data de trmino
   */
  public void find(Date start, Date end) {
    LogsInfo info =
      EventLogProxy.find(logsView.getLogType().queues, start, end);
    logsView.setInfo(info);
    filterPanel.updateInfo(info);
    notifyObservers();
  }

  /**
   * Exportar os dados da busca corrente como arquivo CSV.
   */
  public void exportCSV() {

    List<String> fileCodes = new ArrayList<String>();
    fileCodes.add("CSV");

    String title =
      String.format(getString("export.csv.title"), getString(logsView
        .getLogType().name()));
    ClientFile csvLocalFile =
      ClientLocalFileChooserUtil.browseSingleFileInSaveMode(
        getApplicationFrame(), fileCodes, "CSV", title, true,
        ClientFileChooserUtils.getLocalHomeDirectory());

    if (csvLocalFile == null) {
      // operao cancelada
      return;
    }

    try {

      if (csvLocalFile.exists()) {
        int answerDialog =
          StandardDialogs.showYesNoDialog(getApplicationFrame(),
            getString("warning.title"), getString("override.question"));
        if (answerDialog == StandardDialogs.NO_OPTION) {
          // no sobrescrever, cancelar operao
          return;
        }
      }

      String[][] table = logsView.getTable();
      if (table == null) {
        showInformation(getString("emptyTable"));
        return;
      }
      CSVFileUtils.writeCSVFile(getApplicationFrame(), table, csvLocalFile);
    }
    catch (Exception e) {
      showException(getString("exportFileError"), e.getCause());
    }
  }

  /**
   * Atualiza a configurao de busca atual e os painis conforme o novo tipo de
   * dado de interesse fornecido.
   *
   * @param logType novo tipo de log de interesse selecionado
   */
  public void setLogType(LogsDataType logType) {
    logsView.setLogType(logType);
    filterPanel.setLogType(logType);
    graphicsTabPanel.setLogType(logType);
  }

  /**
   * Filtra os dados exibidos nos grficos para agrupar os resultados conforme o
   * {@link DateViewType} fornecido.
   *
   * @param dateViewType tipo de data para exibio do resultado, pode ser por
   *        ano, por ms ou por dia.
   */
  public void filterDateViewType(DateViewType dateViewType) {
    logsView.setDateViewType(dateViewType);
    notifyObservers();
  }

  /**
   * Filtra os dados exibidos nos grficos para exibir somente os dados
   * relacionados a lista de aplicaoes fornecida.
   *
   * @param applications nova lista de aplicaes de interesse
   */
  public void filterApplications(List<String> applications) {
    logsView.setSelectedApplications(applications);
    notifyObservers();
  }

  /**
   * Filtra os dados exibidos nos grficos para exibir somente os dados
   * relacionados a lista de algoritmos fornecida.
   *
   * @param algorithms nova lista de algoritmos de interesse
   */
  public void filterAlgorithms(List<String> algorithms) {
    logsView.setSelectedAlgorithms(algorithms);
    notifyObservers();
  }

  /** Constroi a interface da aplicao. */
  private void buildInterface() {
    final JFrame mainFrame = getApplicationFrame();
    mainFrame.setJMenuBar(buildMenuBar());
    mainFrame.getContentPane().add(buildSplitPane());
    mainFrame.setSize(new Dimension(800, 600));
    getApplicationFrame().getStatusBar().showStatusBar();
  }

  /**
   * montagem do split pane principal.
   *
   * @return o painel.
   */
  private JSplitPane buildSplitPane() {
    filterPanel = new FilterPanel(this);
    graphicsTabPanel = new GraphicTabsPanel(this);
    searchPanel = new SearchPanel(this);

    JPanel leftPanel = new JPanel(new GridBagLayout());
    leftPanel.add(searchPanel, new GBC(0, 0));
    leftPanel.add(filterPanel, new GBC(0, 1).both());

    JSplitPane splitPane =
      new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, leftPanel, graphicsTabPanel);
    splitPane.setDividerLocation(200);
    splitPane.setOneTouchExpandable(true);
    return splitPane;
  }

  /**
   * Montagem do menu.
   *
   * @return o menu do programa.
   */
  private JMenuBar buildMenuBar() {
    final JMenuBar menuBar = new JMenuBar();
    menuBar.add(buildApplicationMenu());
    menuBar.add(buildHelpMenu());
    return menuBar;
  }

  /**
   * Montagem do menu da aplicao
   *
   * @return um menu da aplicao.
   */
  private JMenu buildApplicationMenu() {
    final JMenu configMenu = new JMenu(getString("menu.stats"));
    configMenu.add(new ApplicationExitAction(this));
    return configMenu;
  }

  /**
   * Montagem do menu de ajuda
   *
   * @return um menu de about.
   */
  private JMenu buildHelpMenu() {
    final JMenu helpMenu = new JMenu(getString("menu.help"));
    helpMenu.add(new ApplicationAboutAction(this));
    return helpMenu;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() throws ApplicationException {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean userCanKillApplication() {
    return true;
  }

}
