/**
 * $Id: StandaloneLocalFile.java 114508 2011-01-13 09:44:42Z cassino $
 */

package csbase.client.externalresources;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * Implementao da <code>LocalFile</code> para execues em que a API JNLP no
 * esteja disponvel (execuo <i>standalone</i>).
 * 
 * @author Tecgraf/PUC-Rio
 */
public class StandaloneLocalFile implements LocalFile {

  /** Referncia lgica para o arquivo . */
  private File file;

  /**
   * Cria o objeto que encapsula um arquivo local.
   * 
   * @param file referncia para o arquivo.
   */
  public StandaloneLocalFile(File file) {
    if (file == null) {
      throw new IllegalArgumentException("file == null");
    }
    this.file = file;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getLength() {
    return file.length();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canRead() {
    return file.canRead();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canWrite() {
    return file.canWrite();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputStream getInputStream() throws IOException {
    return new FileInputStream(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public OutputStream getOutputStream(boolean append) throws IOException {
    return new FileOutputStream(file, append);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return file.getName();
  }

  /**
   * Informa se o caminho representado por este objeto aponta para um diretrio.
   * 
   * @return true se for um diretrio, false se for um arquivo.
   */
  public boolean isDirectory() {
    return file.isDirectory();
  }

  /**
   * Informa se o caminho representado por este objeto aponta para um arquivo.
   * 
   * @return true se for um arquivo, false se for um diretrio.
   */
  public boolean isFile() {
    return file.isFile();
  }

  /**
   * Obtm um array de filhos deste diretrio. Caso este objeto no represente
   * um diretrio, <code>null</code> ser retornado.
   * 
   * @return um array com os filhos deste diretrio, ou <code>null</code> caso
   *         este objeto no represente um diretrio.
   */
  public StandaloneLocalFile[] listFiles() {
    File[] children = file.listFiles();
    if (children == null) {
      return null;
    }
    StandaloneLocalFile[] wrappedChildren =
      new StandaloneLocalFile[children.length];
    for (int i = 0; i < children.length; i++) {
      wrappedChildren[i] = new StandaloneLocalFile(children[i]);
    }
    return wrappedChildren;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean exists() {
    return file.exists();
  }

  /**
   * Cria um novo arquivo (vazio) caso este no exista.
   * 
   * @return <code>true</code> se o arquivo no existia e foi criado com sucesso
   * 
   * @throws IOException se houve algum erro
   * 
   * @see java.io.File#createNewFile()
   */
  public boolean createNewFile() throws IOException {
    return file.createNewFile();
  }

  /**
   * Cria o diretrio caso este no exista.
   * 
   * @return <code>true</code> se o diretrio foi criado com sucesso
   * 
   * @see java.io.File#mkdir()
   */
  public boolean mkdir() {
    return file.mkdir();
  }

  /**
   * Obtm o path absoluto para o arquivo.
   * 
   * @return path absoluto para o arquivo
   * 
   * @see java.io.File#getAbsolutePath()
   */
  public String getAbsolutePath() {
    return file.getAbsolutePath();
  }
}
