package csbase.client.util;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTree;
import javax.swing.UIManager;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreeSelectionModel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.DesktopFrame;
import csbase.client.util.gui.JMultilineLabel;
import csbase.logic.CommonClientProject;
import csbase.logic.User;
import csbase.logic.Version;

/**
 * @see DesktopComponentDialog dilogo de componentes do desktop.
 */
public abstract class ExceptionDialog extends DesktopComponentDialog {

  /**
   * Mensagem de erro utilizada para o caso de passarem uma janela-me invlida
   * durante a requisio de um ExceptionDialog
   */
  private static final String _ILLEGAL_WINDOW_EXCEPTION_MESSAGE =
    "A classe do parmetro window  %s, porm deveria ser %s ou %s.";

  /**
   * Erro ou exceo a ser visualizada na janela
   */
  protected final Throwable _throwable;

  /**
   * Conjunto de informaes adicionais que sero mostradas na janela de erro e
   * includas em um possvel email ao administrador.
   */
  protected String[] additionalInfo;

  /**
   * Construtor auxiliar com <code>JDialog</code>.
   * 
   * @param owner a janela pai
   * @param title o ttulo
   * @param throwable o erro ou exceo
   */
  protected ExceptionDialog(Window owner, String title, Throwable throwable) {
    this(owner, title, throwable, null);
  }

  /**
   * Construtor auxiliar com <code>JFrame</code>.
   * 
   * @param owner a janela pai
   * @param title o ttulo
   * @param throwable o erro ou exceo
   * @param additionalInfo - lista de informaes adicionais que sero mostradas
   *        na janela de erro e includas em um possvel email.
   */
  protected ExceptionDialog(Window owner, String title, Throwable throwable,
    String[] additionalInfo) {
    super(owner, title);
    this._throwable = throwable;
    this.additionalInfo = additionalInfo;
    setWindowClosingMethod();
  }

  /**
   * Mtodo chamado pelos construtores para alterar o comportamento do dilogo
   * clicando no boto marcado por um <b>X</b> localizado no canto superior
   * direito. Por padro, a super classe ignora esse evento, ao chamar este
   * mtodo, o construtor ir garantir que a janela seja fechada.
   */
  private void setWindowClosingMethod() {
    addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        close();
      }
    });
  }

  /**
   * Mtodo esttico de criao de um dilogo <b>detalhado</b> com base em
   * parmetros passados pelo mdulo-cliente para uso sem construtor.
   * 
   * @param window a janela-me.
   * @param title o ttulo.
   * @param throwable a exceo.
   * @return um dialogo de erro.
   */
  public static ExceptionDialog createDialog(final Window window,
    final String title, final Throwable throwable) {
    return createDialog(window, title, throwable, (String[]) null);
  }

  /**
   * Mtodo esttico de criao de um dilogo com base em parmetros passados
   * pelo mdulo-cliente para uso sem construtor.<br>
   *  criado um ExceptionDialog <b>simplificado</b> pois o foco deste 
   * apresentar a mensagem ao usurio. Caso o usurio queira saber mais sobre o
   * erro/exceo, este dilogo apresenta uma opo para detalhar.
   * 
   * @param window a janela-me.
   * @param title o ttulo.
   * @param throwable a exceo.
   * @param message mensagem a ser apresentada ao usurio
   * @return um dialogo de erro.
   */
  public static ExceptionDialog createDialog(final Window window,
    final String title, final Throwable throwable, String message) {
    return createDialog(window, title, throwable, message, null);
  }

  /**
   * Mtodo esttico de criao de um dilogo <b>detalhado</b> com base em
   * parmetros passados pelo mdulo-cliente para uso sem construtor.
   * 
   * @param window a janela-me.
   * @param title o ttulo.
   * @param throwable a exceo.
   * @param additionalInfo - lista de informaes adicionais que sero mostradas
   *        na janela de erro e includas em um possvel email.
   * @return um dialogo de erro.
   */
  public static ExceptionDialog createDialog(final Window window,
    final String title, final Throwable throwable, String[] additionalInfo) {
    return new DetailedExceptionDialog(window, title, throwable, additionalInfo);
  }

  /**
   * Mtodo esttico de criao de um dilogo com base em parmetros passados
   * pelo mdulo-cliente para uso sem construtor.<br>
   *  criado um ExceptionDialog <b>simplificado</b> pois o foco deste 
   * apresentar a mensagem ao usurio. Caso o usurio queira saber mais sobre o
   * erro/exceo, este dilogo apresenta uma opo para detalhar.
   * 
   * @param window a janela-me.
   * @param title o ttulo.
   * @param throwable a exceo.
   * @param message mensagem a ser apresentada ao usurio
   * @param additionalInfo - lista de informaes adicionais que sero mostradas
   *        na janela de erro e includas em um possvel email.
   * @return um dialogo de erro.
   */
  public static ExceptionDialog createDialog(final Window window,
    final String title, final Throwable throwable, String message,
    String[] additionalInfo) {
    return new SimpleExceptionDialog(window, title, throwable, message,
      additionalInfo);
  }

  /**
   * @return comentrios sobre o erro/exceo apresentado pelo dilogo.
   */
  protected abstract String getComments();

  /**
   * @param comments comentrios sobre o erro/exceo apresentado pelo dilogo.
   */
  protected abstract void setComments(String comments);

  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------

  /**
   * Classe que modela um dilogo <b>detalhado</b> de exibio de um
   * erro/exceo detectado no sistema. Este dilogo apresenta a pilha do
   * erro/exceo para o usurio.
   * 
   * @see ExceptionDialog dilogo que apresenta um erro/exceo, detectado no
   *      sistema, para o usurio.
   */
  private static final class DetailedExceptionDialog extends ExceptionDialog {

    /** Nmero de colunas */
    private static final int _MESSAGE_COLUMNS = 80;

    /** Nmero de linhas */
    private static final int _MESSAGE_ROWS = 4;

    /** Nmero de colunas */
    private static final int _STACK_TRACE_COLUMNS = _MESSAGE_COLUMNS;

    /** Nmero de linhas */
    private static final int _STACK_TRACE_ROWS = 14;

    /** Nmero de colunas da rea de comentrios */
    private static final int _COMMENT_COLUMNS = _MESSAGE_COLUMNS;

    /** Nmero de linhas da rea de comentrios */
    private static final int _COMMENT_ROWS = _MESSAGE_ROWS;

    /** Largura da rea da rvore */
    private static final int _TROUBLE_TREE_WIDTH = 800;

    /** Altura da rea da rvore */
    private static final int _TROUBLE_TREE_HEIGHT = 100;

    /** rea de texto de mensagens */
    private JTextArea _messageTextArea;

    /** rea de texto de mensagens da pilha. */
    private JTextArea _stackTraceTextArea;

    /** rea de texto de comentrios feitos pelo usurio */
    protected JTextArea _commentTextArea;

    /** rvore de excees aninhadas */
    private JTree _throwableTree;

    /**
     * Construtor auxiliar com <code>JDialog</code>.
     * 
     * @param owner a janela pai
     * @param title o ttulo
     * @param throwable o erro ou exceo
     * @param additionalInfo - lista de informaes adicionais que sero
     *        mostradas na janela de erro e includas em um possvel email.
     */
    public DetailedExceptionDialog(final Window owner, final String title,
      final Throwable throwable, String[] additionalInfo) {
      super(owner, title, throwable, additionalInfo);
      createComponents();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected String getComments() {
      return this._commentTextArea.getText();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void setComments(String comments) {
      if (null == comments) {
        this._commentTextArea.setText("");
      }
      else {
        this._commentTextArea.setText(comments);
      }
    }

    /**
     * Mtodo interno de construo da inteface grfica.
     */
    private void createComponents() {

      // Define a cor que ser utilizada como cor de fundo para componentes
      // inativos
      Color inactiveColor =
        UIManager.getDefaults().getColor("TextField.inactiveBackground");

      // Cria uma rvore de erros/excees
      final JLabel troubleTreeLabel =
        new JLabel(LNG.get("UTIL_EXCEPTION_TREE"));
      this._throwableTree = new JTree(new ThrowableTreeNode(this._throwable));
      this._throwableTree.addTreeSelectionListener(new TreeSelectionListener() {
        @Override
        public void valueChanged(TreeSelectionEvent e) {
          updateFields();
        }
      });
      this._throwableTree.getSelectionModel().setSelectionMode(
        TreeSelectionModel.SINGLE_TREE_SELECTION);
      final JScrollPane scrollTroubleTree =
        new JScrollPane(this._throwableTree);
      scrollTroubleTree.setPreferredSize(new Dimension(_TROUBLE_TREE_WIDTH,
        _TROUBLE_TREE_HEIGHT));

      // Cria a rea p/ mostrar a mensagem da exceo selecionada na rvore
      final JLabel messageLabel = new JLabel(LNG.get("UTIL_EXCEPTION_MESSAGE"));
      this._messageTextArea =
        new JTextArea(this._throwable.getLocalizedMessage());
      this._messageTextArea.setColumns(_MESSAGE_COLUMNS);
      this._messageTextArea.setRows(_MESSAGE_ROWS);
      this._messageTextArea.setEditable(false);
      this._messageTextArea.setBackground(inactiveColor);

      // Cria a rea p/ mostrar a stack da exceo
      final JLabel stackTraceLabel = new JLabel(LNG.get("UTIL_STACK_TRACE"));
      this._stackTraceTextArea =
        new JTextArea(getStackTraceText(this._throwable.getStackTrace()));
      this._stackTraceTextArea.setColumns(_STACK_TRACE_COLUMNS);
      this._stackTraceTextArea.setRows(_STACK_TRACE_ROWS);
      this._stackTraceTextArea.setEditable(false);
      this._stackTraceTextArea.setBackground(inactiveColor);

      // Cria a rea para possibilitar ao usurio inserir comentrios sobre
      // aquele erro/exceo
      final JLabel commentLabel = new JLabel(LNG.get("UTIL_COMMENT"));
      this._commentTextArea = new JTextArea();
      this._commentTextArea.setColumns(_COMMENT_COLUMNS);
      this._commentTextArea.setRows(_COMMENT_ROWS);

      // Cria o painel com os botes do dilogo.
      JPanel buttonsPanel = createButtonPanel();

      setLayout(new GridBagLayout());
      JScrollPane msgScrollPane = new JScrollPane(this._messageTextArea);
      JScrollPane stackTraceScroll = new JScrollPane(this._stackTraceTextArea);
      JScrollPane commentScrollPane = new JScrollPane(this._commentTextArea);

      Insets li = new Insets(5, 6, 0, 10);
      Insets fi = new Insets(3, 10, 5, 10);

      // rea com as informaes adicionais: 
      JLabel label = new JLabel(LNG.get("UTIL_ADDITIONAL_INFO"));
      int y = 0, rows;
      if (additionalInfo == null || additionalInfo.length == 0) {
        rows = 3; // projeto, usurio e verso
      }
      else {
        rows = Math.min(3 + additionalInfo.length, 4);
      }
      JTextArea additionalInfoTextArea = new JTextArea(rows, 80);
      additionalInfoTextArea.setEditable(false);
      additionalInfoTextArea.setBackground(inactiveColor);
      JScrollPane scrollPane = new JScrollPane(additionalInfoTextArea);
      additionalInfoTextArea.setText(createAdditionalInfoText());
      add(label, new GBC(0, y++).west().insets(li));
      add(scrollPane, new GBC(0, y++).both().insets(fi));

      // rvore
      add(troubleTreeLabel, new GBC(0, y++).west().insets(li));
      add(scrollTroubleTree, new GBC(0, y++).both().insets(fi));

      // mensagem
      add(messageLabel, new GBC(0, y++).west().insets(li));
      add(msgScrollPane, new GBC(0, y++).both().insets(fi));

      // pilha de execuo
      add(stackTraceLabel, new GBC(0, y++).west().insets(li));
      add(stackTraceScroll, new GBC(0, y++).both().insets(fi));

      // comentrio do usurio para enviar no email 
      add(commentLabel, new GBC(0, y++).west().insets(li));
      add(commentScrollPane, new GBC(0, y++).both().insets(fi));

      add(buttonsPanel, new GBC(0, y++).center().insets(7));

      pack();
      center();
    }

    /**
     * @return texto formatado com as informaes adicionais sobre o contexto do
     *         erro.
     */
    private String createAdditionalInfoText() {
      // usurio:
      String info = LNG.get("UTIL_ADDITIONAL_INFO_USER");
      info += " " + User.getLoggedUser().getLogin();
      info += " (" + User.getLoggedUser().getName() + ")\n";

      // projeto atual:
      if (DesktopFrame.getInstance() != null) {
        CommonClientProject project = DesktopFrame.getInstance().getProject();
        if (project == null) {
          info += LNG.get("UTIL_ADDITIONAL_INFO_NO_PROJECT");
        }
        else {
          info += LNG.get("UTIL_ADDITIONAL_INFO_PROJECT");
          info += " " + project.getName();
        }
        info += "\n";
      }

      // Verso:
      final Version version = Version.getInstance();
      if (version != null) {
        info += LNG.get("UTIL_ADDITIONAL_INFO_VERSION");
        info += " " + version.getVersion();
      }

      // informaes adicionais especficas:
      if (additionalInfo != null && additionalInfo.length > 0) {
        for (int i = 0; i < additionalInfo.length; i++) {
          info += "\n" + additionalInfo[i];
        }
      }
      return info;
    }

    /**
     * Criao do painel de botes.
     * 
     * @return o painel.
     */
    private JPanel createButtonPanel() {
      final JPanel panel = new JPanel();
      final JButton sendButton = new JButton(new MailAction(this));
      final JButton closeButton = new JButton(new CloseAction(this));

      ClientUtilities
        .adjustEqualSizes(new JButton[] { sendButton, closeButton });

      panel.add(sendButton);
      panel.add(closeButton);
      return panel;
    }

    /**
     * Atualizao dos campos relativos a exceo selecionada.
     */
    private void updateFields() {
      ThrowableTreeNode node =
        (ThrowableTreeNode) this._throwableTree.getSelectionPath()
          .getLastPathComponent();

      if (node == null) {
        node = (ThrowableTreeNode) this._throwableTree.getModel().getRoot();
      }
      this._messageTextArea.setText(node.getThrowable().getLocalizedMessage());
      this._stackTraceTextArea.setText(getStackTraceText(node.getThrowable()
        .getStackTrace()));
    }

    /**
     * Busca de uma string que representa a pilha de execuo.
     * 
     * @param stackTrace a pilha.
     * @return uma string.
     */
    private String getStackTraceText(final StackTraceElement[] stackTrace) {
      String text = "";
      for (int i = 0; i < stackTrace.length; i++) {
        text += stackTrace[i] + "\n";
      }
      return text;
    }
  }

  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------

  /**
   * Classe que modela um dilogo <b>simplificado</b> de exibio de um
   * erro/exceo detectado no sistema. Este dilogo apresenta uma mensagem do
   * sistema para o usurio e d a possibilidade de abrir um
   * DetailedExceptionDialog.
   * 
   * @see ExceptionDialog dilogo que apresenta um erro/exceo, detectado no
   *      sistema, para o usurio.
   * @see DetailedExceptionDialog dilogo que detalha o erro/exceo
   */
  private static final class SimpleExceptionDialog extends ExceptionDialog {

    /** rea de comentrios */
    private JTextArea _commentTextArea;

    /**
     * Construtor auxiliar com <code>JDialog</code>.
     * 
     * @param owner a janela pai
     * @param title o ttulo
     * @param message as mensagens de erro
     * @param throwable o erro ou exceo
     * @param additionalInfo - lista de informaes adicionais que sero
     *        mostradas na janela de erro e includas em um possvel email.
     */
    public SimpleExceptionDialog(final Window owner, final String title,
      final Throwable throwable, final String message,
      final String[] additionalInfo) {
      super(owner, title, throwable, additionalInfo);
      buildGui(message);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected String getComments() {
      return this._commentTextArea.getText();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void setComments(String comments) {
      if (null == comments) {
        this._commentTextArea.setText("");
      }
      else {
        this._commentTextArea.setText(comments);
      }
    }

    /**
     * Inicializao interna do dilogo.
     * 
     * @param message mensagem que deve aparecer no dilogo.
     */
    private void buildGui(String message) {
      getContentPane().setLayout(new BorderLayout());
      getContentPane().add(makeIconPanel(), BorderLayout.WEST);
      getContentPane().add(makeMainPanel(message), BorderLayout.CENTER);
      pack();
      center();
    }

    /**
     * Cria o painel que contm o cone de erro
     * 
     * @return o painel
     */
    private JPanel makeIconPanel() {
      JPanel panel = new JPanel();
      panel.add(new JLabel(UIManager.getIcon("OptionPane.errorIcon")));

      return panel;
    }

    /**
     * Cria o painel principal
     * 
     * @param message mensagem a ser exibida
     * @return o painel
     */
    private JPanel makeMainPanel(String message) {

      JPanel mainPanel = new JPanel(new GridBagLayout());
      GridBagConstraints c = new GridBagConstraints();
      c.insets = new Insets(12, 12, 11, 11);
      c.anchor = GridBagConstraints.CENTER;
      c.weightx = 1;
      c.weighty = 1;
      mainPanel.add(makeInfoPanel(message), c);
      c.gridy = 1;
      c.anchor = GridBagConstraints.SOUTHEAST;
      c.insets = new Insets(17, 12, 11, 11);
      mainPanel.add(makeButtonsPanel(), c);

      return mainPanel;
    }

    /**
     * Cria o painel que contm a mensagem de erro
     * 
     * @param message mensagem a ser exibida
     * @return o painel
     */
    private JPanel makeInfoPanel(String message) {

      JPanel panel = new JPanel(new GridBagLayout());
      GridBagConstraints c = new GridBagConstraints();
      c.insets = new Insets(0, 0, 0, 0);

      c.fill = GridBagConstraints.HORIZONTAL;
      if (message == null || message.isEmpty() || message.startsWith("<html>")) {
        JLabel label = new JLabel(message);
        panel.add(label, c);
      }
      else {
        JMultilineLabel label = new JMultilineLabel();
        label.setMaxWidth(550);
        label.setJustified(false);
        label.setText(message);
        panel.add(label, c);
      }

      c.gridy = 1;
      c.insets = new Insets(12, 0, 0, 0);
      JLabel jwhere = new JLabel(LNG.get("UTIL_EXECUTION_ERROR"));
      panel.add(jwhere, c);
      c.gridy = 2;
      c.insets = new Insets(0, 0, 0, 0);
      panel.add(new JLabel(LNG.get("UTIL_CONTACT_ERROR")), c);
      c.gridy = 3;
      c.insets = new Insets(12, 0, 0, 0);
      panel.add(new JLabel(LNG.get("UTIL_COMMENT")), c);
      c.gridy = 4;
      c.insets = new Insets(0, 0, 0, 0);
      c.fill = GridBagConstraints.BOTH;
      _commentTextArea = new JTextArea("", 5, 30);
      panel.add(new JScrollPane(_commentTextArea), c);

      return panel;
    }

    /**
     * Cria o painel com os botes
     * 
     * @return o painel
     */
    private JPanel makeButtonsPanel() {
      final JPanel panel = new JPanel();
      final JButton sendButton = new JButton(new MailAction(this));
      final JButton detailButton = new JButton(new DetailThrowableAction(this));
      final JButton closeButton = new JButton(new CloseAction(this));

      ClientUtilities.adjustEqualSizes(new JButton[] { sendButton,
          detailButton, closeButton });

      panel.add(sendButton);
      panel.add(detailButton);
      panel.add(closeButton);
      return panel;
    }
  }

  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------

  /**
   * Ao que ao ser requisitada fecha a janela corrente.
   */
  class CloseAction extends AbstractAction {

    /**
     * Janela corrente que contm esta ao amarrada a um de seus componentes.
     */
    protected ExceptionDialog owner;

    /**
     * Construtor
     * 
     * @param owner janela corrente
     */
    protected CloseAction(ExceptionDialog owner) {
      this(owner, LNG.get("UTIL_CLOSE"));
    }

    /**
     * Construtor
     * 
     * @param owner janela corrente
     * @param text texto.
     */
    protected CloseAction(ExceptionDialog owner, String text) {
      super(text);
      this.owner = owner;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      this.owner.close();
    }
  }

  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------

  /**
   * Ao que ao ser requisitada envia um em-mail para o suporte contendo o
   * erro/exceo apresentado pela janela corrente e um comentrio. Em seguida
   * fecha a janela corrente.
   */
  final class MailAction extends CloseAction {

    /**
     * Construtor
     * 
     * @param owner janela corrente
     */
    protected MailAction(ExceptionDialog owner) {
      super(owner, LNG.get("UTIL_SEND"));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      final ExceptionMessenger instance = ExceptionMessenger.getInstance();
      final String comments = this.owner.getComments();
      final Throwable throwable = this.owner._throwable;
      boolean sent = instance.send(throwable, comments, owner.additionalInfo);

      if (sent) {
        final int INFO = JOptionPane.INFORMATION_MESSAGE;
        final String sentMsg = LNG.get("UTIL_SENT");
        JOptionPane.showMessageDialog(this.owner, sentMsg, "", INFO);
        super.actionPerformed(e);
      }
    }
  }

  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------
  //----------------------------------------------------------------------------

  /**
   * Ao que ao ser requisitada abre um dilogo detalhando o erro/exceo
   * apresentado pela janela corrente. Em seguida fecha a janela corrente.
   */
  final class DetailThrowableAction extends CloseAction {

    /**
     * Construtor
     * 
     * @param owner janela corrente
     */
    protected DetailThrowableAction(ExceptionDialog owner) {
      super(owner, LNG.get("UTIL_ERROR_DETAILS"));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      super.actionPerformed(e);
      ExceptionDialog dialog =
        ExceptionDialog.createDialog(owner, owner.getTitle(), owner._throwable,
          owner.additionalInfo);
      dialog.setVisible(true);
    }
  }
}
