/*
 * $Id: ClientProjectFileChooserFacilities.java 110158 2010-09-14 23:00:41Z
 * clinio $
 */
package csbase.client.util.filechooser;

import java.awt.Window;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationFrame;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.tasks.ChangeFileTypeTask;
import csbase.client.project.tasks.CreateFileTask;
import csbase.client.project.tasks.GetFileTask;
import csbase.client.util.filechooser.filters.ClientFileAllFilter;
import csbase.client.util.filechooser.filters.ClientFileFilterInterface;
import csbase.client.util.filechooser.filters.ClientFileMultipleTypesFilter;
import csbase.client.util.filechooser.filters.ClientFileSingleTypeFilter;
import csbase.logic.ClientFile;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileType;

/**
 * Utilitrio para escolha de arquivos na rea de projetos.
 * 
 * @author Tecgraf/PUC-Rio.
 */
public class ClientProjectFileChooserUtil {

  /**
   * Estrutura queindica o resultado de uma navegao do tipo SAVE em arquivo.
   * 
   * @author Tecgraf
   */
  static public class OperationResult {
    /**
     * Indicativo se arquivo escolhido j existe
     */
    final private boolean existsFile;

    /**
     * Arquivo de projeto associado.
     */
    final private ClientProjectFile file;

    /**
     * Path retornado
     */
    final private String[] path;

    /**
     * Chooser.
     */
    private final ClientFileChooser chooser;

    /**
     * Retorna indicativo de existncia de arquivo selecionado.
     * 
     * @return existsFile indicativo
     */
    public final boolean existsFile() {
      return existsFile;
    }

    /**
     * Retorna o arquivo associado (se existir).
     * 
     * @return file
     */
    public final ClientProjectFile getClientProjectFile() {
      return file;
    }

    /**
     * Mtodo utilitrio para montagem de path como string usando o array
     * interno definido (ver {@link #getPath()}).
     * 
     * @return o texto
     */
    final public String getStringPath() {
      final String[] pth = getPath();
      if ((pth == null) || (pth.length <= 0)) {
        return null;
      }
      StringBuilder builder = new StringBuilder();
      final int endIdx = pth.length;
      for (int i = 0; i < endIdx - 2; i++) {
        builder.append(pth[i]);
        builder.append('/');
      }
      builder.append(pth[endIdx - 1]);
      final String text = builder.toString();
      return text;
    }

    /**
     * Retorna
     * 
     * @return path
     */
    public final String[] getPath() {
      return path;
    }

    /**
     * Construtor.
     * 
     * @param chooser chooser
     * @param project projeto.
     * @param path path retornado
     * @param createChosen indicativo de criao automtica do arquivo
     */
    public OperationResult(final ClientFileChooser chooser,
      final CommonClientProject project, final String[] path,
      boolean createChosen) {
      this.chooser = chooser;
      this.path = path;
      ClientProjectFile tmpFile =
        GetFileTask.runTask(chooser.getOwner(), project, path);

      if (createChosen) {
        if (tmpFile == null) {
          this.file = createFromPath();
          this.existsFile = ((this.file == null) ? false : true);
        }
        else {
          this.file = tmpFile;
          this.existsFile = true;
          final String fileName = path[path.length - 1];
          final ProjectFileType fileType = getChosenFileType(fileName);
          ChangeFileTypeTask.runTask(chooser.getOwner(), file, fileType);
        }
      }
      else {
        this.file = tmpFile;
        this.existsFile = ((this.file == null) ? false : true);
      }
    }

    /**
     * Construtor
     * 
     * @param chooser chooser
     * @param file arquivo.
     */
    public OperationResult(final ClientFileChooser chooser,
      final ClientProjectFile file) {
      this.chooser = chooser;
      if (file == null) {
        this.path = null;
        this.existsFile = false;
      }
      else {
        this.path = file.getPath();
        this.existsFile = true;
      }
      this.file = file;
    }

    /**
     * Cria o arquivo.
     * 
     * @return o arquivo (novo) ou {@code null}.
     */
    final public ClientProjectFile createFromPath() {
      if (file != null) {
        return file;
      }

      final DesktopFrame desktopFrame = DesktopFrame.getInstance();
      final CommonClientProject project = desktopFrame.getProject();
      if (project == null) {
        return null;
      }

      if (path == null) {
        return null;
      }

      final int pathLength = path.length;
      final ClientProjectFile parent;
      if (pathLength == 1) {
        parent = project.getRoot();
      }
      else {
        final String[] parentPath = new String[pathLength - 1];
        for (int i = 0; i < parentPath.length; i++) {
          parentPath[i] = path[i];
        }
        parent = GetFileTask.runTask(chooser.getOwner(), project, parentPath);
      }
      final String fileName = path[pathLength - 1];

      final ProjectFileType tp = getChosenFileType(fileName);
      if (tp == null) {
        final String msg = getString("unsupported.file.type");
        final String extension = FileUtils.getFileExtension(fileName);
        final String err = msg + " - (." + extension + ")";
        StandardDialogs.showErrorDialog(chooser.getOwner(), "", err);
        return null;
      }

      final String fileType = tp.getCode();
      final ClientProjectFile newFile =
        CreateFileTask.runTask(chooser.getOwner(), project, parent, fileName,
          fileType);
      if (newFile == null) {
        return null;
      }
      return newFile;
    }

    /**
     * Busca texto de internacionalizao.
     * 
     * @param tag tag de busca.
     * @param args argumentos.
     * @return texto
     */
    private String getString(final String tag, Object... args) {
      final String prefix1 = ClientFileChooser.class.getSimpleName();
      final String prefix2 = getClass().getSimpleName();
      final String prefix = prefix1 + "." + prefix2;
      final String realTag = prefix + "." + tag;
      final String fmt = LNG.get(realTag);
      final String msg = String.format(fmt, args);
      return msg;
    }

    /**
     * Busca um tipo de arquivo com base em um nome.
     * 
     * @param fileName nome de arquivo.
     * @return o tipo.
     */
    final public ProjectFileType getChosenFileType(final String fileName) {
      final ProjectFileType filterType =
        chooser.getFileTypeFromSelectedFilter();
      if (filterType == null) {
        final ProjectFileType extensionType =
          chooser.getFileTypeFromFileName(fileName);
        return extensionType;
      }
      return filterType;
    }
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em um tipo aceitvel.
   * 
   * @param application a aplicao.
   * @param fileType tipo aceito.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleFileInOpenMode(
    final Application application, final String fileType, final boolean allowAll) {
    final ApplicationFrame frame = application.getApplicationFrame();
    final OperationResult result =
      browseSingleFileInOpenMode(frame, fileType, application.getName(),
        allowAll);
    return result;
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param application a aplicao.
   * @param fileTypes a lista de tipos a serem aceitos
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleFileInOpenMode(
    final Application application, final Collection<String> fileTypes,
    final boolean allowAll) {
    final ApplicationFrame frame = application.getApplicationFrame();
    return browseSingleFileInOpenMode(frame, fileTypes, application.getName(),
      allowAll);
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param window a janela-me (owner).
   * @param fileType tipos a ser aceito
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleFileInOpenMode(
    final Window window, final String fileType, final String title,
    final boolean allowAll) {
    final List<String> fileTypes = new ArrayList<String>();
    fileTypes.add(fileType);
    return browseSingleFileInOpenMode(window, fileTypes, title, allowAll);
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param window a janela-me (owner).
   * @param fileTypes a lista de tipos a serem aceitos
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleFileInOpenMode(
    final Window window, final Collection<String> fileTypes,
    final String title, final boolean allowAll) {
    return browseInOpenMode(window, fileTypes, title, allowAll,
      ClientFileChooserSelectionMode.FILES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param window a janela-me (owner).
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleDirectoryInOpenMode(
    final Window window, final String title, final boolean allowAll) {
    return browseInOpenMode(window, null, title, allowAll,
      ClientFileChooserSelectionMode.DIRECTORIES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param window a janela-me (owner).
   * @param fileTypes a lista de tipos a serem aceitos
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param selectionMode modo de seleo (arquivo ou diretrio).
   * @return o arquivo escolhido ou {@code null}.
   */
  private static OperationResult browseInOpenMode(final Window window,
    final Collection<String> fileTypes, final String title,
    final boolean allowAll, ClientFileChooserSelectionMode selectionMode) {

    final ClientFileChooser chooser = new ClientFileChooser(window);
    final DesktopFrame desktop = DesktopFrame.getInstance();
    final CommonClientProject project = desktop.getProject();
    if (project == null) {
      StandardDialogs.showErrorDialog(window, null, "No project!");
      return null;
    }
    final ClientProjectFile root = project.getRoot();
    chooser.setCurrentDirectory(root);
    chooser.setTitle(title);
    chooser.setSelectionMode(selectionMode);
    chooser.setSelectionType(ClientFileChooserType.OPEN);
    chooser.setLocalHomeButtonVisible(false);
    chooser.setLocalRootButtonVisible(false);
    chooser.setProjectHomeButtonVisible(true);
    chooser.setCardinality(ClientFileChooserCardinality.SINGLE_CHOOSE);

    final List<ClientFileFilterInterface> filters =
      new ArrayList<ClientFileFilterInterface>();
    if (fileTypes != null) {
      if (fileTypes.size() > 1) {
        filters.add(new ClientFileMultipleTypesFilter(fileTypes));
      }
      for (final String fileType : fileTypes) {
        final ClientFileSingleTypeFilter flt =
          new ClientFileSingleTypeFilter(fileType);
        filters.add(flt);
      }
    }
    if (allowAll) {
      filters.add(new ClientFileAllFilter());
    }

    chooser.setViewFilters(filters);
    chooser.setVisible(true);
    final List<ClientFile> selected = chooser.getChosenItens();
    if (selected == null || selected.isEmpty()) {
      return null;
    }
    final ClientProjectFile first = (ClientProjectFile) selected.get(0);
    final OperationResult result = new OperationResult(chooser, first);
    return result;
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param application a aplicao.
   * @param fileType tipo a ser aceitos
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param createChosen indicativo de criao automtica do novo arquivo
   *        escolhido, se inexistente.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleFileInSaveMode(
    final Application application, final String fileType,
    final boolean allowAll, final boolean createChosen) {
    final ApplicationFrame frame = application.getApplicationFrame();
    return browseSingleFileInSaveMode(frame, fileType, application.getName(),
      allowAll, createChosen);
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param application a aplicao.
   * @param fileTypes tipos a ser aceitos
   * @param defaultFileType tipos default aceito.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param createChosen indicativo de criao automtica do novo arquivo
   *        escolhido, se inexistente.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleFileInSaveMode(
    final Application application, final Collection<String> fileTypes,
    String defaultFileType, final boolean allowAll, final boolean createChosen) {
    final ApplicationFrame frame = application.getApplicationFrame();
    return browseSingleFileInSaveMode(frame, fileTypes, defaultFileType,
      application.getName(), allowAll, createChosen);
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param window a janela-me.
   * @param fileType tipo a ser aceito
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param createChosen indicativo de criao automtica do novo arquivo
   *        escolhido, se inexistente.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleFileInSaveMode(
    final Window window, final String fileType, final String title,
    final boolean allowAll, final boolean createChosen) {
    final List<String> fileTypes = new ArrayList<String>();
    fileTypes.add(fileType);
    return browseSingleFileInSaveMode(window, fileTypes, fileType, title,
      allowAll, createChosen);
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param window a janela-me.
   * @param fileTypes a lista de tipos a serem aceitos
   * @param defaultFileType tipo default aceito.
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param createChosen indicativo de criao automtica do novo arquivo
   *        escolhido, se inexistente.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleFileInSaveMode(
    final Window window, final Collection<String> fileTypes,
    String defaultFileType, final String title, final boolean allowAll,
    final boolean createChosen) {
    final ClientFileChooser chooser = new ClientFileChooser(window);
    final DesktopFrame desktop = DesktopFrame.getInstance();
    final CommonClientProject project = desktop.getProject();
    if (project == null) {
      StandardDialogs.showErrorDialog(window, null, "No project!");
      return null;
    }

    chooser.setTitle(title);
    chooser.setSelectionMode(ClientFileChooserSelectionMode.FILES_ONLY);
    chooser.setSelectionType(ClientFileChooserType.SAVE);
    chooser.setLocalHomeButtonVisible(false);
    chooser.setLocalRootButtonVisible(false);
    chooser.setProjectHomeButtonVisible(true);
    chooser.setCardinality(ClientFileChooserCardinality.SINGLE_CHOOSE);

    final ClientProjectFile root = project.getRoot();
    chooser.setCurrentDirectory(root);

    final List<ClientFileFilterInterface> filters =
      new ArrayList<ClientFileFilterInterface>();

    if (fileTypes.size() > 1) {
      filters.add(new ClientFileMultipleTypesFilter(fileTypes));
    }

    ClientFileFilterInterface defaultFilter = null;
    for (final String fileType : fileTypes) {
      final ClientFileSingleTypeFilter flt =
        new ClientFileSingleTypeFilter(fileType);
      filters.add(flt);
      if (defaultFileType != null && defaultFileType.equals(fileType)) {
        defaultFilter = flt;
      }
    }

    if (allowAll) {
      filters.add(new ClientFileAllFilter());
    }

    chooser.setViewFilters(filters);
    if (defaultFilter != null) {
      chooser.setSelectedFilter(defaultFilter);
    }
    chooser.setVisible(true);
    final String[] selectedSavePath = chooser.getSelectedSavePath();
    if (selectedSavePath == null || selectedSavePath.length <= 0) {
      return null;
    }
    final OperationResult result =
      new OperationResult(chooser, project, selectedSavePath, createChosen);
    return result;
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de diretrios de
   * sada na rea de projetos.
   * 
   * @param window a janela-me.
   * @param title ttulo.
   * @param createChosen indicativo de criao automtica do novo arquivo
   *        escolhido, se inexistente.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleDirectoryInSaveMode(
    Window window, String title, boolean createChosen) {

    ClientFileChooser chooser = new ClientFileChooser(window);
    DesktopFrame desktop = DesktopFrame.getInstance();
    CommonClientProject project = desktop.getProject();
    if (project == null) {
      StandardDialogs.showErrorDialog(window, null, "No project!");
      return null;
    }

    chooser.setTitle(title);
    chooser.setSelectionMode(ClientFileChooserSelectionMode.DIRECTORIES_ONLY);
    chooser.setSelectionType(ClientFileChooserType.SAVE);
    chooser.setLocalHomeButtonVisible(false);
    chooser.setLocalRootButtonVisible(false);
    chooser.setProjectHomeButtonVisible(true);
    chooser.setCardinality(ClientFileChooserCardinality.SINGLE_CHOOSE);

    ClientProjectFile root = project.getRoot();
    chooser.setCurrentDirectory(root);

    List<ClientFileFilterInterface> filters =
      new ArrayList<ClientFileFilterInterface>();

    filters.add(new ClientFileAllFilter());

    chooser.setViewFilters(filters);
    chooser.setVisible(true);
    String[] selectedSavePath = chooser.getSelectedSavePath();
    if (selectedSavePath == null || selectedSavePath.length <= 0) {
      return null;
    }
    final OperationResult result =
      new OperationResult(chooser, project, selectedSavePath, createChosen);
    return result;
  }

  /**
   * Mtodo para ser usado na escolha mtiplos itens de seleo homognicos
   * 
   * @param window a janela-pai (owner).
   * @param fileTypes tipos a ser aceitos
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param selectionMode modo de seleo (arquivo ou diretrio).
   * @return o arquivo escolhido ou {@code null}.
   */
  private static List<OperationResult> browseMultipleInOpenMode(
    final Window window, final Collection<String> fileTypes,
    final String title, final boolean allowAll,
    ClientFileChooserSelectionMode selectionMode) {

    final ClientFileChooser chooser = new ClientFileChooser(window);
    final DesktopFrame desktop = DesktopFrame.getInstance();
    final CommonClientProject project = desktop.getProject();
    if (project == null) {
      StandardDialogs.showErrorDialog(window, null, "No project!");
      return null;
    }
    final ClientProjectFile root = project.getRoot();
    chooser.setCurrentDirectory(root);
    chooser.setTitle(title);
    chooser.setSelectionMode(selectionMode);
    chooser.setSelectionType(ClientFileChooserType.OPEN);
    chooser.setLocalHomeButtonVisible(false);
    chooser.setLocalRootButtonVisible(false);
    chooser.setProjectHomeButtonVisible(true);
    chooser.setCardinality(ClientFileChooserCardinality.MULTIPLE_CHOOSE);

    final List<ClientFileFilterInterface> filters =
      new ArrayList<ClientFileFilterInterface>();
    if (fileTypes != null) {
      if (fileTypes.size() > 1) {
        filters.add(new ClientFileMultipleTypesFilter(fileTypes));
      }
      for (final String fileType : fileTypes) {
        final ClientFileSingleTypeFilter flt =
          new ClientFileSingleTypeFilter(fileType);
        filters.add(flt);
      }
    }
    if (allowAll) {
      filters.add(new ClientFileAllFilter());
    }

    chooser.setViewFilters(filters);
    chooser.setVisible(true);
    final List<ClientFile> selected = chooser.getChosenItens();
    if (selected == null || selected.isEmpty()) {
      return null;
    }

    final List<OperationResult> results = new ArrayList<OperationResult>();
    for (ClientFile sel : selected) {
      final OperationResult result =
        new OperationResult(chooser, (ClientProjectFile) sel);
      results.add(result);
    }
    return results;
  }

  /**
   * Mtodo para ser usado na escolha de mtiplos diretrios de projeto.
   * 
   * @param window a janela-pai (owner).
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return lista de diretrios escolhidos ou {@code null}.
   */
  final static public List<OperationResult> browseMultipleDirectoriesInOpenMode(
    final Window window, final String title, final boolean allowAll) {
    return browseMultipleInOpenMode(window, null, title, allowAll,
      ClientFileChooserSelectionMode.DIRECTORIES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos da projeto com base
   * em lista de tipos aceitveis. final ProjectFileType fileType =
   * ProjectFileType.getFileType(fileCode); final List<String> extensions =
   * fileType.getExtensions();
   * 
   * 
   * @param window a janela-me (owner).
   * @param fileCodes a lista de tipos a serem aceitos
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return lista de arquivos escolhidos ou {@code null}.
   */
  final static public List<OperationResult> browseMultipleFilesInOpenMode(
    final Window window, final Collection<String> fileCodes,
    final String title, final boolean allowAll) {
    return browseMultipleInOpenMode(window, fileCodes, title, allowAll,
      ClientFileChooserSelectionMode.FILES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de mtiplos
   * arquivos da rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param application a aplicao.
   * @param fileTypes a lista de tipos a serem aceitos
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return lista de arquivos escolhidos ou {@code null}.
   */
  final static public List<OperationResult> browseMultipleFilesInOpenMode(
    final Application application, final Collection<String> fileTypes,
    final boolean allowAll) {
    final ApplicationFrame frame = application.getApplicationFrame();
    return browseMultipleFilesInOpenMode(frame, fileTypes,
      application.getName(), allowAll);
  }

}
