package csbase.client.applications.flowapplication.filters;

import java.awt.Graphics2D;
import java.awt.event.MouseEvent;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import tecgraf.vix.TypeMessage;
import csbase.client.applications.flowapplication.RubberBand;
import csbase.client.applications.flowapplication.Workspace;
import csbase.client.applications.flowapplication.graph.Graph;
import csbase.client.applications.flowapplication.graph.GraphElement;
import csbase.client.applications.flowapplication.graph.GraphFileDescriptor;
import csbase.client.applications.flowapplication.graph.GraphLink;
import csbase.client.applications.flowapplication.graph.GraphNode;
import csbase.client.applications.flowapplication.messages.PickElementMessage;
import csbase.client.applications.flowapplication.messages.PickGraphMessage;
import csbase.client.applications.flowapplication.messages.PickNodeMessage;
import csbase.client.applications.flowapplication.messages.SelectElementMessage;
import csbase.client.applications.flowapplication.messages.SelectElementsMessage;

/**
 * Filtro de seleo
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class SelectElementFilter extends WorkspaceFilter {

  /**
   * Estado do filtro: incio
   */
  private static final int START = 0;

  /**
   * Estado do filtro: aguardando drag
   */
  private static final int WAITING_DRAG_TO_MOVE_ELEMENT = 1;

  /**
   * Estado do filtro: movendo
   */
  private static final int MOVING_ELEMENT = 2;

  /**
   * Estado do filtro: aguadando drag de rubber-band
   */
  private static final int WAITING_DRAG_TO_RUBBER_BAND = 3;

  /**
   * Estado do filtro: drag de rubber-band
   */
  private static final int DRAGGING_RUBBER_BAND = 4;

  /**
   * Estado do filtro: aguardando seleo de um de grupo
   */
  private static final int WAITING_SELECT_ONE_OF_GROUP = 5;

  /**
   * Estado do filtro: aguardando drag de vrios elementos.
   */
  private static final int WAITING_DRAG_TO_MOVE_MANY_ELEMENTS = 6;

  /**
   * Estado do filtro: movendo vrios elementos
   */
  private static final int MOVING_MANY_ELEMENTS = 7;

  /**
   * Estado do filtro: aguardando fechar link
   */
  private static final int WAITING_FINISH_LINK = 8;

  /**
   * Elemento corrente
   */
  private GraphElement currentElement;

  /**
   * Link corrente
   */
  private GraphLink currentLink;

  /**
   * Estado corrente
   */
  private int currentState;

  /**
   * Ponto selecionado.
   */
  private Point2D selectedPoint;

  /**
   * Rubber band
   */
  private RubberBand rubberBand;

  /**
   * Lista de elementos selecionados.
   */
  private List<GraphElement> elementCollection;

  /**
   * Construtor
   * 
   * @param workspace workspace
   */
  public SelectElementFilter(final Workspace workspace) {
    super(workspace);
    this.currentState = START;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void callbackButton(final Point2D pt, final MouseEvent ev) {
    switch (this.currentState) {
      case WAITING_DRAG_TO_MOVE_ELEMENT: {
        if (wasPressed(ev)) {
          unselectCurrentElement(START);
        }
      }
      case START: {
        if (wasPressed(ev)) {
          final PickNodeMessage pickNodeMessage = new PickNodeMessage(pt);
          pickNodeMessage.sendVO(this);
          final GraphNode node = pickNodeMessage.getNode();
          if (node != null) {
            this.currentLink = node.createLink(pt);
            if (this.currentLink != null) {
              repaint();
              this.currentState = WAITING_FINISH_LINK;
              break;
            }
          }
          final PickElementMessage pickElementMessage =
            new PickElementMessage(pt);
          pickElementMessage.sendVO(this);
          final GraphElement element = pickElementMessage.getElement();
          if (element == null) {
            this.currentElement = null;
            this.currentState = WAITING_DRAG_TO_RUBBER_BAND;
            this.rubberBand = new RubberBand(pt);
            break;
          }
          this.currentElement = element;
          if (this.currentElement != null) {
            this.selectedPoint = (Point2D) pt.clone();
            this.currentElement.setSelected(true);
            repaint();
            this.currentState = WAITING_DRAG_TO_MOVE_ELEMENT;
            break;
          }
        }
        break;
      }
      case MOVING_ELEMENT: {
        if (wasReleased(ev)) {
          this.currentElement.drop(pt);
          unselectCurrentElement(START);
        }
        break;
      }
      case WAITING_DRAG_TO_RUBBER_BAND: {
        this.rubberBand = null;
        this.currentState = START;
        break;
      }
      case DRAGGING_RUBBER_BAND: {
        final PickGraphMessage message = new PickGraphMessage();
        message.sendVO(this);
        final Rectangle2D bounds = this.rubberBand.getBounds();
        this.elementCollection =
          new LinkedList<GraphElement>(message.getGraph().getElementCollection(
            bounds));
        for (final GraphElement element : this.elementCollection) {
          element.setSelected(true);
        }
        this.rubberBand = null;
        repaint();
        if (this.elementCollection.size() > 0) {
          this.currentState = WAITING_SELECT_ONE_OF_GROUP;
        }
        else {
          reset();
        }
        break;
      }
      case WAITING_SELECT_ONE_OF_GROUP: {
        if (wasPressed(ev)) {
          final PickElementMessage pickElementMessage =
            new PickElementMessage(pt);
          pickElementMessage.sendVO(this);
          final GraphElement element = pickElementMessage.getElement();
          if (element != null && this.elementCollection.contains(element)) {
            this.selectedPoint = (Point2D) pt.clone();
            this.currentState = WAITING_DRAG_TO_MOVE_MANY_ELEMENTS;
          }
          else {
            unselectElements(START);
            this.currentElement = element;
            if (this.currentElement != null) {
              this.selectedPoint = (Point2D) pt.clone();
              this.currentElement.setSelected(true);
              repaint();
              this.currentState = WAITING_DRAG_TO_MOVE_ELEMENT;
            }
            else {
              this.currentState = WAITING_DRAG_TO_RUBBER_BAND;
              this.rubberBand = new RubberBand(pt);
            }
          }
        }
        break;
      }
      case WAITING_DRAG_TO_MOVE_MANY_ELEMENTS: {
        this.currentState = WAITING_SELECT_ONE_OF_GROUP;
        break;
      }
      case MOVING_MANY_ELEMENTS: {
        final Iterator<GraphElement> iterator =
          this.elementCollection.iterator();
        while (iterator.hasNext()) {
          final GraphElement element = iterator.next();
          element.drop();
        }
        this.currentState = WAITING_SELECT_ONE_OF_GROUP;
        break;
      }
      case WAITING_FINISH_LINK: {
        final PickNodeMessage message = new PickNodeMessage(pt);
        message.sendVO(this);
        final GraphNode node = message.getNode();
        if (node != null) {
          final GraphFileDescriptor inputFile = node.getInputFileDescriptor(pt);
          if (inputFile != null) {
            this.currentLink.increase(pt);
            if (this.currentLink.finish(inputFile)) {
              this.currentElement = this.currentLink;
              this.currentLink = null;
              this.selectedPoint = (Point2D) pt.clone();
              this.currentElement.setSelected(true);
              repaint();
              this.currentState = WAITING_DRAG_TO_MOVE_ELEMENT;
              break;
            }
          }
        }
        this.currentLink.deattach();
        this.currentLink = null;
        reset();
        repaint();
        this.currentState = START;
        break;
      }
    }
    super.callbackButton(pt, ev);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void callbackDrag(final Point2D point, final MouseEvent ev) {
    switch (this.currentState) {
      case WAITING_DRAG_TO_MOVE_ELEMENT:
      case MOVING_ELEMENT:
        this.currentElement.drag(this.selectedPoint, point);
        this.selectedPoint =
          new Point2D.Double(Math.max(0, point.getX()), Math.max(0, point
            .getY()));
        repaint();
        this.currentState = MOVING_ELEMENT;
        break;
      case DRAGGING_RUBBER_BAND:
      case WAITING_DRAG_TO_RUBBER_BAND:
        this.rubberBand.setPoint(point);
        repaint();
        this.currentState = DRAGGING_RUBBER_BAND;
        break;
      case MOVING_MANY_ELEMENTS:
      case WAITING_DRAG_TO_MOVE_MANY_ELEMENTS:
        final double startX = this.selectedPoint.getX();
        final double endX = point.getX();
        final double startY = this.selectedPoint.getY();
        final double endY = point.getY();
        final double tx = endX - startX;
        final double ty = endY - startY;
        final Iterator<GraphElement> iterator =
          this.elementCollection.iterator();
        while (iterator.hasNext()) {
          final GraphElement element = iterator.next();
          element.drag(tx, ty);
        }
        this.selectedPoint =
          new Point2D.Double(Math.max(0, point.getX()), Math.max(0, point
            .getY()));
        repaint();
        this.currentState = MOVING_MANY_ELEMENTS;
        break;
      case WAITING_FINISH_LINK:
        this.currentLink.pretentToIncrease(point);
        repaint();
        break;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void callbackRepaint(final Graphics2D g) {
    if (this.rubberBand != null) {
      this.rubberBand.paint(g);
    }
    super.callbackRepaint(g);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean msgHandlerVO(final TypeMessage message) {
    if (message instanceof SelectElementMessage) {
      handleSelectElementMessage((SelectElementMessage) message);
      return true;
    }
    if (message instanceof SelectElementsMessage) {
      handleSelectElementsMessage((SelectElementsMessage) message);
      return true;
    }
    return super.msgHandlerVO(message);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasElementRemoved(final Graph graph, final GraphElement element) {
    reset();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasReseted(final Graph graph) {
    reset();
  }

  /**
   * Tratador de mensagem de seleo.
   * 
   * @param message mensagem.
   */
  private void handleSelectElementMessage(final SelectElementMessage message) {
    reset();
    this.currentElement = message.getElement();
    this.currentElement.setSelected(true);
    this.selectedPoint = (Point2D) message.getPoint().clone();
    repaint();
    this.currentState = WAITING_DRAG_TO_MOVE_ELEMENT;
  }

  /**
   * Tratador de mensagem de seleo mltipla.
   * 
   * @param message mensagem.
   */
  private void handleSelectElementsMessage(final SelectElementsMessage message) {
    reset();
    final List<GraphElement> elements =
      new LinkedList<GraphElement>(message.getElements());
    if (!elements.isEmpty()) {
      for (final GraphElement element : elements) {
        element.setSelected(true);
      }
    }
    this.elementCollection = elements;
    repaint();
    this.currentState = WAITING_SELECT_ONE_OF_GROUP;
  }

  /**
   * Reinicializao.
   */
  private void reset() {
    this.currentLink = null;
    unselectCurrentElement(START);
    unselectElements(START);
    unmarkRubberBand();
  }

  /**
   * Desmarcao de rubber-band.
   */
  private void unmarkRubberBand() {
    this.rubberBand = null;
    this.currentState = START;
  }

  /**
   * Fim de seleo de elemento corrente.
   * 
   * @param nextState prximo estado.
   */
  private void unselectCurrentElement(final int nextState) {
    if (this.currentElement != null) {
      this.currentElement.setSelected(false);
      this.currentElement = null;
    }
    this.selectedPoint = null;
    repaint();
    this.currentState = nextState;
  }

  /**
   * Deseleo total.
   * 
   * @param nextState prximo estado.
   */
  private void unselectElements(final int nextState) {
    if (this.elementCollection != null) {
      final Iterator<GraphElement> elementIterator =
        this.elementCollection.iterator();
      while (elementIterator.hasNext()) {
        final GraphElement element = elementIterator.next();
        element.setSelected(false);
      }
      this.elementCollection = null;
    }
    this.selectedPoint = null;
    repaint();
    this.currentState = nextState;
  }

  /**
   * Indicativo de mouse pressionado.
   * 
   * @param ev evento
   * @return indicativo
   */
  private boolean wasPressed(final MouseEvent ev) {
    return (ev.getButton() == MouseEvent.BUTTON1)
      && (ev.getID() == MouseEvent.MOUSE_PRESSED);
  }

  /**
   * Indicativo de mouse solto.
   * 
   * @param ev evento
   * @return indicativo
   */
  private boolean wasReleased(final MouseEvent ev) {
    return (ev.getButton() == MouseEvent.BUTTON1)
      && (ev.getID() == MouseEvent.MOUSE_RELEASED);
  }
}
