package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.text.MessageFormat;
import java.util.Hashtable;
import java.util.List;

import javax.swing.AbstractButton;
import javax.swing.BorderFactory;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.gui.GBC;
import csbase.client.applications.algorithmsmanager.models.AlgorithmListItem;
import csbase.client.applications.algorithmsmanager.versiontree.PropertiesPanel;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmProperty;

/**
 * Painel com as informaes para criao de um novo algoritmo.
 * 
 */
public class AlgorithmCreateInfoPanel extends CommonInfoEditPanel {
  /** Campo texto com o nome do algoritmo */
  private JTextField algorithmNameField;

  /** Campo texto com o nome do algoritmo */
  private JTextField algorithmIdField;

  /**
   * Caixa de seleo que define se o identificador do algoritmo deve ser gerado
   * automaticamente.
   */
  private JCheckBox generateIdCheckBox;

  /**
   * Lista com as propriedades ordenadas dos algoritmos. Cada propriedade  um
   * mapeamento da chave ao Label (String de visualizaco).
   */
  private List<AlgorithmProperty> algorithmProperties;

  /** Painel de propriedades de algoritmos */
  private PropertiesPanel propertiesPanel;

  /** Painel principal */
  private JPanel mainPanel;

  /** Indica que est sendo feita uma edio */
  private boolean isEditing;

  /**
   * Constri o painel com as informaes bsicas para criao de um algoritmo.
   * 
   * @param algoPane painel com os dados de criao de um algoritmo
   */
  public AlgorithmCreateInfoPanel(CommonEditTabbedPanel algoPane) {
    super(algoPane);
    setCancelActionState(true);
  }

  /**
   * Obtm a janela da aplicao Gerenciador de Algoritmos.
   * 
   * @return a janela da aplcicao
   */
  private Window getWindow() {
    return getApplication().getApplicationFrame();
  }

  /**
   * Obtm o painel com os campos para alterao de uma categoria.
   * 
   * @return o painel criado
   */
  private JPanel getAlgoCreatePanel() {
    JPanel panel = new JPanel(new GridBagLayout());
    panel.setBorder(BorderFactory.createEmptyBorder());

    algorithmProperties =
      AlgorithmManagementProxy.getAlgorithmProperties(getWindow());
    propertiesPanel =
      new PropertiesPanel(algorithmProperties, new Hashtable<String, String>(),
        getString("AlgorithmCreateInfoPanel.label.properties"), getWindow(),
        true);

    if (this.algorithmProperties.size() > 0) {
      panel.add(propertiesPanel.getPropertiesPanel(), new GBC(0, 0)
        .horizontal().west().insets(0, 0, 0, 0));
    }
    panel
      .add(buildEmptyPanel(), new GBC(0, 1).both().west().insets(0, 0, 0, 0));
    return panel;
  }

  /**
   * Obtm o painel com o nome do algoritmo.
   * 
   * @return o painel criado
   */
  private JPanel getAlgorithmNamePanel() {
    JPanel panel = new JPanel(new GridBagLayout());
    String maxSize = Integer.toString(AlgorithmInfo.MAX_NAME_SIZE);
    String maxInLabel =
      MessageFormat.format(getString("AlgorithmCreateInfoPanel.msg.name_max"),
        new Object[] { maxSize });
    JLabel nameLabel =
      new JLabel(getString("AlgorithmCreateInfoPanel.label.name") + " "
        + maxInLabel);
    algorithmNameField = new JTextField(30);
    algorithmNameField.addKeyListener(new KeyAdapter() {
      @Override
      public void keyReleased(KeyEvent e) {
        if (e.getKeyCode() == KeyEvent.VK_ENTER) {
          apply();
        }
      }
    });
    algorithmNameField.addCaretListener(new CaretListener() {
      @Override
      public void caretUpdate(CaretEvent e) {
        JTextField tf = (JTextField) e.getSource();
        boolean hasName = !(tf.getText().trim().isEmpty());
        isEditing = hasName;
        setApplyActionState(hasName);
      }
    });

    String maxInId =
      MessageFormat.format(getString("AlgorithmCreateInfoPanel.msg.id_max"),
        new Object[] { maxSize });

    final JLabel idLabel =
      new JLabel(getString("AlgorithmCreateInfoPanel.label.id") + " " + maxInId);
    algorithmIdField = new JTextField(30);
    algorithmIdField.addCaretListener(new CaretListener() {
      @Override
      public void caretUpdate(CaretEvent e) {
        JTextField tf = (JTextField) e.getSource();
        boolean hasName = !(tf.getText().trim().isEmpty());
        isEditing = hasName;
      }
    });

    final JPanel idPanel = new JPanel(new GridBagLayout());
    idPanel.add(idLabel, new GBC(0, 0).none().west().insets(5, 15, 5, 5));
    idPanel.add(algorithmIdField,
      new GBC(1, 0).horizontal().west().insets(5, 5, 5, 15));
    idPanel.setVisible(false);

    JLabel generateIdLabel =
      new JLabel(getString("AlgorithmCreateInfoPanel.label.generate_id"));
    generateIdCheckBox = new JCheckBox();
    generateIdCheckBox.setSelected(true);
    generateIdCheckBox.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        AbstractButton abstractButton = (AbstractButton) e.getSource();
        boolean selected = abstractButton.getModel().isSelected();
        idPanel.setVisible(!selected);
      }
    });

    panel.add(nameLabel, new GBC(0, 0).none().west().insets(5, 5, 5, 5));
    panel.add(algorithmNameField,
      new GBC(1, 0).horizontal().west().insets(5, 5, 5, 5));
    panel.add(generateIdLabel, new GBC(0, 1).none().west().insets(5, 5, 5, 5));
    panel.add(generateIdCheckBox,
      new GBC(1, 1).horizontal().west().insets(5, 5, 5, 5));
    panel.add(idPanel, new GBC(0, 2).gridwidth(2).horizontal().west());

    return panel;
  }

  /**
   * Obtm o nome da categoria especificado pelo usurio.
   * 
   * @return o nome da categoria
   */
  protected String getFieldCategoryName() {
    return algorithmNameField.getText();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initializeData() {
    algorithmNameField.requestFocus();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JPanel buildMainInfoPanel() {
    if (mainPanel == null) {
      mainPanel = new JPanel(new GridBagLayout());

      JPanel namePanel = getAlgorithmNamePanel();
      JPanel algoPanel = getAlgoCreatePanel();
      mainPanel.add(namePanel,
        new GBC(0, 0).horizontal().west().insets(0, 0, 0, 0));
      mainPanel.add(algoPanel, new GBC(0, 1).both().west().insets(0, 0, 0, 0));
    }
    return mainPanel;
  }

  /**
   * A partir do nome do algoritmo e suas propriedades adicionadas pelo usurio,
   * o sero efetivamente criadas no servidor.
   * 
   * @return retorna true se o algoritmo foi criado com sucesso, caso contrrio,
   *         retorna false
   */
  protected boolean createAlgorithm() {
    String algoName = algorithmNameField.getText().trim();
    if (getApplication().containsAlgorithm(algoName)) {
      getApplication()
        .showError(
          getString("AlgorithmCreateInfoPanel.msg.error.create.algorithm_exists"));
      algorithmNameField.requestFocus();
      return false;
    }

    Hashtable<String, String> fieldValues = propertiesPanel.getProperties();

    String algoId = null;
    if (!generateIdCheckBox.isSelected()) {
      algoId = algorithmIdField.getText().trim();
      if (!isValidId(algoId)) {
        getApplication()
          .showError(
            getString("AlgorithmCreateInfoPanel.msg.error.create.algorithm_id_invalid"));
        algorithmIdField.requestFocus();
        return false;
      }
      if (getApplication().containsAlgorithmWithId(algoId)) {
        getApplication()
          .showError(
            getString("AlgorithmCreateInfoPanel.msg.error.create.algorithm_id_exists"));
        algorithmIdField.requestFocus();
        return false;
      }
    }
    AlgorithmInfo algorithmCreated =
      getAlgorithmCreateDataPanel().createAlgorithm(algoName, algoId,
        fieldValues);
    if (algorithmCreated == null) {
      getApplication().showError(
        getString("AlgorithmCreateInfoPanel.msg.error.create.no_sucess"));
      algorithmNameField.requestFocus();
      return false;
    }
    return true;
  }

  /**
   * Determina se um identificador de algoritmo  vlido.
   * 
   * @param algoId o identificador.
   * @return verdadeiro se o identificador  vlido ou falso, caso contrrio.
   */
  private boolean isValidId(String algoId) {
    String trimmedId = algoId.trim();
    if (trimmedId.isEmpty()) {
      return false;
    }
    if (trimmedId.length() > 50) {
      return false;
    }
    String normalizedId = FileUtils.fixDirectoryName(trimmedId);
    /*
     * Se a "normalizao" do id no tiver nenhum efeito,  porque todos os seus
     * caracteres so vlidos.
     */
    if (!normalizedId.equals(trimmedId)) {
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void apply() {
    isEditing = false;
    createAlgorithm();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void cancel() {
    boolean confirmCancelling =
      !wasModified() || (wasModified() && confirmCancelling());
    if (confirmCancelling) {
      getAlgorithmCreateDataPanel().cancel();
      AlgorithmListItem selectedAlgorithm =
        getSelectionPanel().getSelectedAlgorithm();
      if (selectedAlgorithm != null) {
        getSelectionPanel().selectAlgorithm(selectedAlgorithm.getItem());
      }
    }
  }

  /**
   * Obtm o painel de seleo de algoritmos.
   * 
   * @return o painel de seleo de algoritmos
   */
  private AlgorithmSelectionPanel getSelectionPanel() {
    return getAlgorithmCreateDataPanel().getSelectionPanel();
  }

  /**
   * Obtm o painel global de criao de dados de algoritmo.
   * 
   * @return o painel global de criao de dados de algoritmo
   */
  private AlgorithmCreateDataPanel getAlgorithmCreateDataPanel() {
    return (AlgorithmCreateDataPanel) getEditPanel();
  }

  /**
   * Confirma com o usurio se a operao de cancelamento deve ser efetivada, j
   * que perder todas as alteraes feitas sobre o dado.
   * 
   * @return retorna true, se a alterao do dado deve ser cancelada, caso
   *         contrrio, retorna false
   */
  private boolean confirmCancelling() {
    int confirm =
      getApplication().showOptionDialog(
        getString("AlgorithmCreateInfoPanel.msg.cancel.confirm"),
        new String[] {
            getString("AlgorithmCreateInfoPanel.msg.cancel.confirm.yes"),
            getString("AlgorithmCreateInfoPanel.msg.cancel.confirm.no") });
    if (confirm == 0) {
      isEditing = false;
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setDataChanged() {
    isEditing = false;
    changeOperationsState(isEditing);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getTitle() {
    return getApplication().getString("CommonEditTabbedPanel.tab.info.basic");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean wasModified() {
    return isEditing;
  }
}
