/*
 * $Id: FileExchangerConfiguration.java 106082 2010-05-26 02:36:42Z juliana $
 */

package csbase.client.applications.fileexchanger.logic;

import csbase.client.applications.fileexchanger.FileExchanger;

/**
 * Configurao do transferidor de arquivos.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class FileExchangerConfiguration {

  /**
   * Modo corrente de transferncia.
   */
  private ExchangeMode transferMode = ExchangeMode.STREAM;

  /**
   * Tamanho do buffer no modo ExchangeMode.STREAM.
   */
  private BlockSize blockSize = BlockSize.NORMAL;

  /**
   * Valor default da varivel importMultipleFiles.
   */
  public final static boolean DEFAULT_IMPORT_MULTIPLE_FILES = false;

  /**
   * Se importa mltiplos arquivos.
   */
  private boolean importMultipleFiles = DEFAULT_IMPORT_MULTIPLE_FILES;

  /**
   * Valor default da varivel importDirectories.
   */
  public final static boolean DEFAULT_IMPORT_DIRECTORIES = false;

  /**
   * Se importa diretrios.
   */
  private boolean importDirectories = DEFAULT_IMPORT_DIRECTORIES;

  /**
   * Valor default da varivel exportMultipleFiles.
   */
  public final static boolean DEFAULT_EXPORT_MULTIPLE_FILES = false;

  /**
   * Se exporta mltiplos arquivos.
   */
  private boolean exportMultipleFiles = DEFAULT_EXPORT_MULTIPLE_FILES;

  /**
   * Valor default da varivel exportDirectories.
   */
  public final static boolean DEFAULT_EXPORT_DIRECTORIES = false;
  /**
   * Se exporta diretrios.
   */
  private boolean exportDirectories = DEFAULT_EXPORT_DIRECTORIES;

  /**
   * Valor default da varivel minTransfersStartInterval (em ms).
   */
  public final static long DEFAULT_MIN_INTERVAL_TO_CHECK_PENDING_TRANSFERS =
    100;

  /**
   * Intervalo mnimo para checar transferncias pendentes (em ms).
   */
  private long minIntervalToCheckPendingTransfers =
    DEFAULT_MIN_INTERVAL_TO_CHECK_PENDING_TRANSFERS;

  /**
   * Valor default da varivel maxTransfersStartInterval (em ms).
   */
  public final static long DEFAULT_MAX_INTERVAL_TO_CHECK_PENDING_TRANSFERS =
    3000;

  /**
   * Intervalo mximo para checar transferncias pendentes.
   */
  private long maxIntervalToCheckPendingTransfers =
    DEFAULT_MAX_INTERVAL_TO_CHECK_PENDING_TRANSFERS;

  /**
   * Valor default da varivel maxSimultaneousTransfers.
   */
  public final static long DEFAULT_MAX_SIMULTANEOUS_TRANSFERS = 5;

  /**
   * Nmero mximo de transferncias simultneas.
   */
  private long maxSimultaneousTransfers = DEFAULT_MAX_SIMULTANEOUS_TRANSFERS;

  /**
   * Valor default da varivel guiUpdateInterval (em ms).
   */
  public final static long DEFAULT_GUI_UPDATE_INTERVAL = 3000;

  /**
   * Intervalo de atualizao da GUI (em ms).
   */
  private long guiUpdateInterval = DEFAULT_GUI_UPDATE_INTERVAL;

  /**
   * Propriedades relacionadas a este transferidor.
   */
  private FileExchanger fileExchanger = null;

  /**
   * @return Modo de transferncia.
   */
  public ExchangeMode getTransferMode() {
    return transferMode;
  }

  /**
   * @return O tamanho do buffer no modo ExchangeMode.STREAM.
   */
  public BlockSize getBlockSize() {
    return blockSize;
  }

  /**
   * @return Booleano, se importa mltiplos arquivos.
   */
  public boolean isImportMultipleFiles() {
    return importMultipleFiles;
  }

  /**
   * @return Booleano, se importa diretrios.
   */
  public boolean isImportDirectories() {
    return importDirectories;
  }

  /**
   * @return Booleano, se exporta mltiplos arquivos.
   */
  public boolean isExportMultipleFiles() {
    return exportMultipleFiles;
  }

  /**
   * @return Booleano, se exporta diretrios.
   */
  public boolean isExportDirectories() {
    return exportDirectories;
  }

  /**
   * @return O intervalo mnimo para checar transferncias pendentes (em ms).
   */
  public long getMinIntervalToCheckPendingTransfers() {
    return minIntervalToCheckPendingTransfers;
  }

  /**
   * @return O intervalo mximo para checar transferncias pendentes (em ms).
   */
  public long getMaxIntervalToCheckPendingTransfers() {
    return maxIntervalToCheckPendingTransfers;
  }

  /**
   * @return O nmero mximo de transferncias simultneas.
   */
  public long getMaxSimultaneousTransfers() {
    return maxSimultaneousTransfers;
  }

  /**
   * @return O intervalo para atualizao da GUI (em ms).
   */
  public long getGuiUpdateInterval() {
    return guiUpdateInterval;
  }

  /**
   * @return O transferidor para o qual este objeto est configurado.
   */
  public FileExchanger getFileExchanger() {
    return fileExchanger;
  }

  /**
   * Define o valor do tamanho do buffer no modo ExchangeMode.STREAM.
   * 
   * @param blockSize Valor do tamanho do buffer no modo ExchangeMode.STREAM.
   */
  public void setBlockSize(BlockSize blockSize) {
    this.blockSize = blockSize;
  }

  /**
   * Define o valor do modo de transferncia.
   * 
   * @param transferMode Valor do modo de transferncia.
   */
  public void setTransferMode(final ExchangeMode transferMode) {
    this.transferMode = transferMode;
  }

  /**
   * Construtor. L as propriedades do transferidor de arquivos.
   * 
   * @param fileExchanger Transferidor de arquivos.
   */
  public FileExchangerConfiguration(final FileExchanger fileExchanger) {
    this.fileExchanger = fileExchanger;
    if (fileExchanger != null) {
      importMultipleFiles =
        fileExchanger.getBooleanSpecificProperty("importMultipleFiles",
          DEFAULT_IMPORT_MULTIPLE_FILES);

      importDirectories =
        fileExchanger.getBooleanSpecificProperty("importDirectories",
          DEFAULT_IMPORT_DIRECTORIES);

      exportMultipleFiles =
        fileExchanger.getBooleanSpecificProperty("exportMultipleFiles",
          DEFAULT_EXPORT_MULTIPLE_FILES);

      exportDirectories =
        fileExchanger.getBooleanSpecificProperty("exportDirectories",
          DEFAULT_EXPORT_DIRECTORIES);

      minIntervalToCheckPendingTransfers =
        fileExchanger.getLongSpecificProperty(
          "minIntervalToCheckPendingTransfers",
          DEFAULT_MIN_INTERVAL_TO_CHECK_PENDING_TRANSFERS);

      maxIntervalToCheckPendingTransfers =
        fileExchanger.getLongSpecificProperty(
          "maxIntervalToCheckPendingTransfers",
          DEFAULT_MAX_INTERVAL_TO_CHECK_PENDING_TRANSFERS);

      maxSimultaneousTransfers =
        fileExchanger.getLongSpecificProperty("maxSimultaneousTransfers",
          DEFAULT_MAX_SIMULTANEOUS_TRANSFERS);

      guiUpdateInterval =
        fileExchanger.getLongSpecificProperty("guiUpdateInterval",
          DEFAULT_GUI_UPDATE_INTERVAL);
    }
  }
}
