package csbase.client.applications.flowapplication.graph;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.Window;
import java.awt.datatransfer.Transferable;
import java.awt.geom.Dimension2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.geom.RoundRectangle2D;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.vix.TypeMessage;
import tecgraf.vix.VO;
import csbase.client.algorithms.AlgorithmConfiguratorView;
import csbase.client.algorithms.AlgorithmConfiguratorViewListener;
import csbase.client.algorithms.tasks.ViewValidationTask;
import csbase.client.algorithms.validation.ViewValidationResult;
import csbase.client.algorithms.validation.ViewValidator;
import csbase.client.applications.flowapplication.CompassDirection;
import csbase.client.applications.flowapplication.SVGVO;
import csbase.client.applications.flowapplication.SVGVO.HorizontalAlignment;
import csbase.client.applications.flowapplication.SVGVO.VerticalAlignment;
import csbase.client.applications.flowapplication.graph.GraphNodeImageDecoration.DecorationType;
import csbase.client.applications.flowapplication.graph.utils.CopyAndPasteOperation;
import csbase.client.applications.flowapplication.graph.utils.GraphNodeUpdateOperation;
import csbase.client.applications.flowapplication.graph.utils.OperationStatus;
import csbase.client.applications.flowapplication.messages.ErrorMessage;
import csbase.client.applications.flowapplication.messages.PasteMessage;
import csbase.client.applications.flowapplication.messages.PickNodeMessage;
import csbase.client.applications.flowapplication.messages.ShowParameterValuesMessage;
import csbase.client.applications.flowapplication.util.FlowApplicationRemoteTask;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.tasks.GetChildFromNameTask;
import csbase.exception.ParseException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfiguratorListener;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.parameters.AbstractFileParameter;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.logic.algorithms.parameters.InputFileParameter;
import csbase.logic.algorithms.parameters.InputURLParameter;
import csbase.logic.algorithms.parameters.OutputFileParameter;
import csbase.logic.algorithms.parameters.OutputURLParameter;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.ValidationMode;

/**
 * <p>
 * Viso do N de Algoritmo
 * </p>
 *
 * <p>
 * Um n pode estar no estado mal-formado. Isto ocorre quando o algoritmo ou a
 * verso do algoritmo que ele representa foi removida do sistema.
 * </p>
 *
 * @author lmoreira
 */
public final class GraphNode extends GraphElement implements ViewValidator {

  /**
   * Cor da sombra
   */
  private static final Color SHADOW_COLOR = new Color(100, 100, 100, 100);

  /**
   * Fator da desenho de sombra
   */
  private static final double SHADOW_FACTOR = 1.075;

  /**
   * Largura/altura extra a ser considerada no clculo da borda do n.
   */
  private static final double BORDER_MARGIN = 5.0;

  /**
   * Trao padro da borda.
   */
  private static final Stroke DEFAULT_STROKE = new BasicStroke(1F);

  /**
   * Fonte utilizada para o nome do algoritmo no n.
   */
  private static final Font ALGORITHM_NAME_FONT = new Font(Font.SANS_SERIF,
    Font.BOLD, 12);

  /**
   * Fonte utilizada para a verso do algoritmo no n.
   */
  private static final Font ALGORITHM_VERSION_FONT = new Font(Font.SANS_SERIF,
    Font.PLAIN, 10);

  /**
   * Trao da borda quando n est desviado
   */
  private static final Stroke BYPASSED_STROKE = new BasicStroke(0f,
    BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL, 0, new float[] { 9 }, 0);

  /**
   * Cor-padro da face (texto e borda).
   */
  private static final Color DEFAULT_FOREGROUNG_COLOR = Color.BLACK;
  /**
   * Cor para indicar que existe um erro no n.
   */
  private static final Color ERROR_COLOR = new Color(255, 182, 182);
  /**
   * Cor do texto quando o n foi desviado
   */
  private static final Color BYPASSED_TEXT_COLOR = new Color(150, 150, 150);
  /**
   * Maior dimenso do cone.
   */
  private static final double MAX_IMAGE_DIMENSION = 25.0;

  /**
   * Altura mnima.
   */
  private static final double MIN_HEIGHT = 36.0;
  /**
   * Largura mnima.
   */
  private static final double MIN_WIDTH = 72.0;
  /**
   * ndice do canto nordeste.
   */
  private static final int NORTH_EAST_INDEX = 1;
  /**
   * ndice do canto noroeste.
   */
  private static final int NORTH_WEST_INDEX = 0;
  /**
   * Cor para sinalizar que o n no est pronto para executar.
   */
  private static final Color NOT_READY_BACKGROUND_COLOR = new Color(255, 255,
    210);
  /**
   * Cor para sinalizar que o n est pronto para executar.
   */
  private static final Color READY_BACKGROUND_COLOR = new Color(210, 255, 210);
  /**
   * ndice para o canto sudeste.
   */
  private static final int SOUTH_EAST_INDEX = 2;
  /**
   * ndice para o canto sudoeste.
   */
  private static final int SOUTH_WEST_INDEX = 3;

  /**
   * O algoritmo deste n.
   */
  private AlgorithmInfo algorithm;
  /**
   * A viso do nome do algoritmo deste n.
   */
  private final GraphString algorithmNameView;
  /**
   * A viso do identificador da verso do algoritmo deste n.
   */
  private final HideableGraphString algorithmVersionIdView;
  /**
   * O identificador do algoritmo.
   */
  private final AlgorithmVersionId algorithmVersionId;
  /**
   * A cor atual da borda.
   */
  private Color borderColor;
  /**
   * A cor de fundo atual.
   */
  private Color backgroundColor;

  /**
   * A viso do configurador de algoritmos.
   */
  private AlgorithmConfiguratorView configuratorView;
  /**
   * As vises dos cantos do n.
   */
  private GraphPoint[] cornerViews;
  /**
   * As dimenses deste n.
   */
  private Dimension dimension;
  /**
   * O cone principal deste n.
   */
  private SVGVO icon;

  /**
   * As vises dos arquivos de entrada do algoritmo.
   */
  private List<GraphFileDescriptor> inputFileTypeViews;
  /**
   * As vises dos arquivos de sada do algoritmo.
   */
  private List<GraphFileDescriptor> outputFileTypeViews;
  /**
   * O arquivo de sada (stdout) de um algoritmo sendo executado.
   */
  private FileURLValue standardOutputFile;

  /**
   * Indica se o n prov um cdigo de sada durante sua execuo.
   */
  private boolean hasExitCode;
  /**
   * O arquivo que armazena o cdigo de sada de execuo do algoritmo.
   */
  private FileURLValue exitCodeLogFile;

  /**
   * Os rtulos dos parmetros do n indexados pelo nome dos mesmos.
   */
  private final Map<String, String> parameterLabelsByName;
  /**
   * O nome dos parmetros.
   */
  private final Set<String> parameterNames;
  /**
   * Os tipos dos parmetros do n indexados pelo nome dos mesmos.
   */
  private final Map<String, String> parameterTypesByName;
  /**
   * Os valores dos parmetros do n indexados pelo nome dos mesmos.
   */
  private final Map<String, String> parameterValuesByName;
  /**
   * O ponto superior-esquerdo do n.
   */
  private Point northWestPoint;

  /**
   * A forma do n.
   */
  private Shape shape;

  /**
   * A forma do n.
   */
  private Shape shadowShape;

  /**
   * Sinaliza se o usurio indicou que o n deve ser desviado durante a
   * execuo. O desvio propriamente dito s pode ser feito se um conjunto de
   * condies forem atendidas @see {@link #validateBypass()}.
   */
  private boolean bypassed;

  /**
   * Sinaliza se o n pode ser desviado no fluxo. Caso positivo, o algoritmo
   * continua a aparecer no editor de fluxos, mas no  executado. Todas as
   * entradas do n so desviadas para o n seguinte no fluxo durante a
   * execuo. Caso negativo, mesmo que o usurio indique que queira fazer o
   * desvio, este no  feito.
   */
  private boolean canBypass;

  /**
   * A decorao (cone de status) utilizada neste n.
   */
  private GraphNodeDecoration<?> decoration;

  /**
   * Indica se o layout interno do n foi modificado (e portanto precisa ser
   * refeito).
   */
  private boolean layoutChanged;

  /**
   * Cria uma viso de n para um algoritmo que est instalado no sistema
   * (bem-formado).
   *
   * @param graph A viso do fluxo (No aceita {@code null}).
   * @param id O identificador do n.
   * @param configuratorView A viso do configurador de algoritmos (No aceita
   *        {@code null}).
   * @param point O ponto superior-esquerdo (No aceita {@code null}).
   * @param bypassed Sinal que indica se o n deve ser desviado.
   * @param versionInfoVisible Sinal que indica se a informao de verso do
   *        algoritmo deve ser mostrada no n.
   */
  GraphNode(final Graph graph, final int id,
    final AlgorithmConfiguratorView configuratorView, final Point2D point,
    final boolean bypassed, final boolean versionInfoVisible) {
    this(graph, id, configuratorView, point, null, bypassed, versionInfoVisible);
  }

  /**
   * Obtm o sinal que indica se o cdigo de sada  retornado durante a
   * execuo do algoritmo.
   *
   * @return Verdadeiro se o n prov um cdigo de sada, ou falso, caso
   *         contrrio.
   */
  public boolean hasExitCode() {
    return hasExitCode;
  }

  /**
   * Atribui o sinal que indica que o n prov cdigo de sada durante sua
   * execuo.
   *
   * @param hasExitCode Sinal que indica se o n prov cdigo de sada.
   */
  public void setHasExitCode(final boolean hasExitCode) {
    this.hasExitCode = hasExitCode;
  }

  /**
   * Cria uma viso de n para um algoritmo que est instalado no sistema
   * (bem-formado).
   *
   * @param graph A viso do fluxo (No aceita {@code null}).
   * @param id O identificador do n.
   * @param configuratorView A viso do configurador de algoritmos (No aceita
   *        {@code null}).
   * @param point O ponto superior-esquerdo (No aceita {@code null}).
   * @param size A dimenso do n (No aceita {@code null}).<br>
   *        Se a altura ou a largura for menor do que a altura ou a largura
   *        mnima, ela ser desprezada e o n ter altura ou largura mnima.
   * @param bypassed Sinal que indica se o n deve ser desviado.
   * @param versionInfoVisible Sinal que indica se a informao de verso do
   *        algoritmo deve ser mostrada no n.
   */
  GraphNode(final Graph graph, final int id,
    final AlgorithmConfiguratorView configuratorView, final Point2D point,
    final Dimension2D size, final boolean bypassed,
    final boolean versionInfoVisible) {
    super(graph, id);
    this.bypassed = bypassed;
    this.canBypass = true;
    this.borderColor = DEFAULT_FOREGROUNG_COLOR;
    this.backgroundColor = NOT_READY_BACKGROUND_COLOR;
    this.configuratorView = configuratorView;
    final AlgorithmConfigurator configurator =
      this.configuratorView.getConfigurator();
    final AlgorithmVersionInfo algorithmVersion =
      configurator.getAlgorithmVersion();
    this.algorithm = algorithmVersion.getInfo();
    this.algorithmNameView =
      new GraphString(this, this.algorithm.toString(), ALGORITHM_NAME_FONT);
    this.algorithmVersionId = algorithmVersion.getId();
    this.algorithmVersionIdView =
      new HideableGraphString(this, algorithmVersionId.toString(),
        ALGORITHM_VERSION_FONT, versionInfoVisible);
    this.parameterLabelsByName = new HashMap<String, String>();
    this.parameterValuesByName = new HashMap<String, String>();
    this.parameterTypesByName = new HashMap<String, String>();
    this.parameterNames = new HashSet<String>();
    this.hasExitCode = configurator.hasExitCode();
    this.layoutChanged = false;

    final Map<String, String> parameterValues = new HashMap<String, String>();
    for (final String parameterName : configurator.getParameterNames()) {
      String parameterLabel;
      String parameterValue;
      String parameterType;
      try {
        parameterLabel = configurator.getParameterLabel(parameterName);
        parameterType = configurator.getParameterType(parameterName);
        addParameter(parameterName, parameterLabel, parameterType);
        parameterValue = configurator.getParameterValue(parameterName);
        parameterValues.put(parameterName, parameterValue);
      }
      catch (ParameterNotFoundException e) {
        throw new IllegalStateException(e);
      }
    }
    try {
      setParameterValuesByName(parameterValues);
    }
    catch (ParseException e) {
      throw new IllegalStateException(e);
    }
    catch (ParameterNotFoundException e) {
      throw new IllegalStateException(e);
    }
    if (GraphNodeUpdateOperation.hasNewerVersion(this)) {
      this.icon = new SVGVO(GraphImages.ICON_OUTDATED);
      this.icon.setVerticalAlignment(VerticalAlignment.CENTER);
      this.icon.setHorizontalAlignment(HorizontalAlignment.LEFT);
    }
    adjustTextPadding();

    this.decoration = null;
    if (size != null) {
      setBounds2D(point.getX(), point.getY(), size.getWidth(), size.getHeight());
    }
    else {
      computeMeasures(point);
    }
    setUpVix();
    adjustGrid();
    setListeners();
    fullValidation();
  }

  /**
   * Ajusta o espaamento em volta do texto do n.
   */
  private void adjustTextPadding() {
    int normalPadding = 3;
    int imagePadding = 0;
    int rightPadding = normalPadding;
    int leftPadding = normalPadding;
    if (decoration != null) {
      rightPadding = imagePadding;
    }
    if (icon != null) {
      leftPadding = imagePadding;
    }
    this.algorithmNameView.setRightPadding(rightPadding);
    this.algorithmVersionIdView.setRightPadding(rightPadding);
    this.algorithmNameView.setLeftPadding(leftPadding);
    this.algorithmVersionIdView.setLeftPadding(leftPadding);
    this.algorithmNameView.setTopPadding(0);
    this.algorithmVersionIdView.setTopPadding(0);
    this.algorithmVersionIdView.setBottomPadding(0);
  }

  /**
   * Faz a validao completa do n, indicando o resultado visualmente.
   */
  private void fullValidation() {
    validateBypass();
    ViewValidationResult result = validate(ValidationMode.FULL);
    highlightValidationResult(result);
  }

  /**
   * Cria uma viso de n para um algoritmo cuja verso no est instalada no
   * sistema (mal-formado).
   *
   * @param graph A viso do fluxo (No aceita {@code null}).
   * @param id O identificador do n.
   * @param algorithmInfo O algoritmo (No aceita {@code null}).
   * @param versionId O identificador da verso do algoritmo (No aceita
   *        {@code null}).
   * @param point O ponto superior-esquerdo (No aceita {@code null}).
   * @param size A dimenso do n (No aceita {@code null}).<br>
   *        Se a altura ou a largura for menor do que a altura ou a largura
   *        mnima, ela ser desprezada e o n ter altura ou largura mnima.
   * @param bypassed Sinal que indica se o n deve ser desviado.
   * @param versionInfoVisible Sinal que indica se a informao de verso do
   *        algoritmo deve ser mostrada no n.
   */
  GraphNode(final Graph graph, final int id, final AlgorithmInfo algorithmInfo,
    final AlgorithmVersionId versionId, final Point2D point,
    final Dimension2D size, final boolean bypassed,
    final boolean versionInfoVisible) {
    this(graph, id, algorithmInfo.getName(), versionId, point, size, bypassed,
      versionInfoVisible);
    this.algorithm = algorithmInfo;
  }

  /**
   * Cria uma viso de n para um algoritmo que no est instalado no sistema
   * (mal-formado).
   *
   * @param graph A viso do fluxo (No aceita {@code null}).
   * @param id O identificador do n.
   * @param algorithmName O nome do algoritmo (No aceita {@code null}).
   * @param versionId O identificador da verso do algoritmo (No aceita
   *        {@code null}).
   * @param point O ponto superior-esquerdo (No aceita {@code null}).
   * @param size A dimenso do n (No aceita {@code null}).<br>
   *        Se a altura ou a largura for menor do que a altura ou a largura
   *        mnima, ela ser desprezada e o n ter altura ou largura mnima.
   * @param bypassed Sinal que indica se o n deve ser desviado.
   * @param versionInfoVisible Sinal que indica se a informao de verso do
   *        algoritmo deve ser mostrada no n.
   */
  GraphNode(final Graph graph, final int id, final String algorithmName,
    final AlgorithmVersionId versionId, final Point2D point,
    final Dimension2D size, final boolean bypassed,
    final boolean versionInfoVisible) {
    super(graph, id);
    this.bypassed = bypassed;
    this.canBypass = true;

    this.inputFileTypeViews = new LinkedList<GraphFileDescriptor>();
    this.outputFileTypeViews = new LinkedList<GraphFileDescriptor>();

    this.borderColor = DEFAULT_FOREGROUNG_COLOR;
    this.backgroundColor = ERROR_COLOR;
    this.algorithmNameView =
      new GraphString(this, algorithmName, ALGORITHM_NAME_FONT);
    this.algorithmVersionId = versionId;
    this.algorithmVersionIdView =
      new HideableGraphString(this, versionId.toString(),
        ALGORITHM_VERSION_FONT, versionInfoVisible);
    this.parameterNames = new HashSet<String>();
    this.parameterLabelsByName = new HashMap<String, String>();
    this.parameterValuesByName = new HashMap<String, String>();
    this.parameterTypesByName = new HashMap<String, String>();
    this.decoration = null;
    this.hasExitCode = false;
    this.layoutChanged = false;
    setUpVix();
    adjustTextPadding();
    setBounds2D(point.getX(), point.getY(), size.getWidth(), size.getHeight());
  }

  /**
   * Atribui o valor do arquivo que deve receber a sada padro da execuo do
   * algoritmo representado por este n.
   *
   * @param standardOutputFile arquivo que recebe a sada padro da execuo do
   *        algoritmo
   */
  public void setStandardOutputFile(final FileURLValue standardOutputFile) {
    this.standardOutputFile = standardOutputFile;
    // Garante consistncia.
    if (null != configuratorView && null != configuratorView.getConfigurator()) {
      configuratorView.getConfigurator().setStandardOutputFile(
        standardOutputFile);
    }
  }

  /**
   * Atribui o valor do arquivo que deve armazenar o cdigo de sada da execuo
   * do algortimo representado por este n.
   *
   * @param exitCodeLogFile arquivo que armazena o cdido de sada de execuo
   *        do algoritmo
   */
  public void setExitCodeLogFile(final FileURLValue exitCodeLogFile) {
    this.exitCodeLogFile = exitCodeLogFile;
    // Garante consistncia.
    if (null != configuratorView && null != configuratorView.getConfigurator()) {
      configuratorView.getConfigurator().setExitCodeLogFile(exitCodeLogFile);
    }
  }

  /**
   * Obtm a sada padro que ser utilizada durante a execuo do algortmo
   * representado por este n.
   *
   * @return a sada padro que ser utilizada durante a execuo do algortmo.
   */
  public FileURLValue getStandardOutputFile() {
    return this.standardOutputFile;
  }

  /**
   * Obtm o arquivo que armazena o cdigo de sada da execuo do algoritmo
   * representado por este n.
   *
   * @return o arquivo que armazena o cdigo de sada da execuo do algoritmo.
   */
  public FileURLValue getExitCodeLogFile() {
    return this.exitCodeLogFile;
  }

  /**
   * Adiciona uma viso de arquivo de entrada.
   *
   * @param graphFileDescriptor A viso de arquivo de entrada (No aceita
   *        {@code null}).
   *
   * @return {@code true} se foi possvel adicionar a viso ou {@code false} se
   *         j existia uma viso igual a esta.
   */
  public boolean addInputFileDescriptor(
    final GraphFileDescriptor graphFileDescriptor) {
    if (graphFileDescriptor == null) {
      throw new IllegalArgumentException(
        "O parmetro graphFileDescriptor est nulo.");
    }
    if (this.inputFileTypeViews.contains(graphFileDescriptor)) {
      return false;
    }
    this.inputFileTypeViews.add(graphFileDescriptor);
    changeVO(null, graphFileDescriptor);
    graphFileDescriptor.changeVS(null, this);
    updateInputFileTypes();
    return true;
  }

  /**
   * Adiciona uma viso de arquivo de sada.
   *
   * @param graphFileDescriptor A viso de arquivo de sada (No aceita
   *        {@code null}).
   *
   * @return {@code true} se foi possvel adicionar a viso ou {@code false} se
   *         j existia uma viso igual a esta.
   */
  public boolean addOutputFileDescriptor(
    final GraphFileDescriptor graphFileDescriptor) {
    if (graphFileDescriptor == null) {
      throw new IllegalArgumentException(
        "O parmetro graphFileDescriptor est nulo.");
    }
    if (this.outputFileTypeViews.contains(graphFileDescriptor)) {
      return false;
    }
    this.outputFileTypeViews.add(graphFileDescriptor);
    changeVO(null, graphFileDescriptor);
    graphFileDescriptor.changeVS(null, this);
    updateOutputFileTypes();
    return true;
  }

  /**
   * <p>
   * Cria um parmetro na viso.
   * </p>
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   *
   * @param parameterLabel O rtulo do parmetro (No aceita {@code null}).
   * @param parameterType O tipo do parmetro (No aceita {@code null}).
   *
   * @return {@code true} em caso de sucesso ou {@code false} se j existe um
   *         parmetro com o mesmo nome cadastrado
   */
  public boolean addParameter(final String parameterName,
    final String parameterLabel, final String parameterType) {
    if (parameterName == null) {
      throw new IllegalArgumentException("O parmetro parameterName est nulo.");
    }
    if (parameterLabel == null) {
      throw new IllegalArgumentException(
        "O parmetro parameterLabel est nulo.");
    }
    if (parameterType == null) {
      throw new IllegalArgumentException("O parmetro parameterType est nulo.");
    }
    if (this.parameterNames.contains(parameterName)) {
      return false;
    }
    this.parameterNames.add(parameterName);
    this.parameterLabelsByName.put(parameterName, parameterLabel);
    this.parameterTypesByName.put(parameterName, parameterType);
    return true;
  }

  /**
   * Exibe a viso de configurador de algoritmos para que o usurio possa
   * parametrizar o n.
   */
  public void askForParameterValues() {
    if (isWellFormed()) {
      this.configuratorView.setEnabled(true);
      this.configuratorView.launch();
    }
    else {
      showError();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void callbackRepaint(final Graphics2D g) {
    if (this.layoutChanged) {
      if (computeMeasures(getCenterPoint())) {
        this.layoutChanged = false;
      }
      else {
        return;
      }
    }
    Stroke stroke;
    if (isBypassed() && canBeBypassed()) {
      stroke = BYPASSED_STROKE;
    }
    else {
      g.setPaint(SHADOW_COLOR);
      g.fill(shadowShape);
      g.setPaint(this.backgroundColor);
      g.fill(this.shape);
      stroke = DEFAULT_STROKE;
    }

    g.setPaint(this.borderColor);
    g.setStroke(stroke);
    g.draw(this.shape);
    if (isSelected()) {
      for (final GraphPoint cornerView : this.cornerViews) {
        cornerView.paint(g);
      }
    }
    super.callbackRepaint(g);
  }

  /**
   * Comear uma conexo para o arquivo de sada que possui o ponto informado.
   *
   * @param pt O ponto (No aceita {@code null}).
   *
   * @return A conexo ou {@code null} seno existir um arquivo de sada no
   *         ponto informado.
   */
  public GraphLink createLink(final Point2D pt) {
    for (final GraphFileDescriptor outputFileType : this.outputFileTypeViews) {
      if (!outputFileType.isWellFormed()) {
        return null;
      }
      if (outputFileType.contains(pt)) {
        final GraphLink link = getGraph().createLink(outputFileType);
        return link;
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void deattach() {
    for (final GraphLink link : getLinkCollection()) {
      link.deattach();
    }
    super.deattach();
  }

  /**
   * Termina de criar a conexo no arquivo de entrada especificado pelo ponto
   * informado.
   *
   * @param link A conexo (No aceita {@code null}).
   * @param pt O ponto informado (No aceita {@code null}).
   *
   * @return {@code true} em caso de sucesso ou {@code false} se no houver um
   *         arquivo de entrada no ponto informado.
   */
  public boolean finishLink(final GraphLink link, final Point2D pt) {
    final GraphFileDescriptor inputFileType = getInputFileDescriptor(pt);
    if (inputFileType == null) {
      return false;
    }
    return link.finish(inputFileType);
  }

  /**
   * Obtm o algoritmo do n.
   *
   * @return O algoritmo do n ou {@code null} se ele no foi informado.
   */
  public AlgorithmInfo getAlgorithm() {
    return this.algorithm;
  }

  /**
   * Obtm a viso de configurador algoritmo do n.
   *
   * @return A viso de configurador algoritmo do n ou {@code null} se ela no
   *         foi informada.
   */
  public AlgorithmConfiguratorView getAlgorithmConfiguratorView() {
    return this.configuratorView;
  }

  /**
   * Obtm o nome do algoritmo do n.
   *
   * @return O nome algoritmo do n.
   */
  public String getAlgorithmName() {
    return this.algorithmNameView.getText();
  }

  /**
   * Obtm o identificador da verso do algoritmo.
   *
   * @return O identificador da verso do algoritmo.
   */
  public AlgorithmVersionId getAlgorithmVersionId() {
    return this.algorithmVersionId;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Rectangle2D getBounds2D() {
    final Rectangle2D inputFileBounds =
      getBoundsFileTypes(this.inputFileTypeViews);
    final Rectangle2D outputFileBounds =
      getBoundsFileTypes(this.outputFileTypeViews);
    Rectangle2D bounds = this.shape.getBounds();
    final double w = bounds.getWidth();
    final double h = bounds.getHeight();
    final double y = bounds.getMinY();
    final double x = bounds.getMinX();
    bounds.setFrame(x, y, w, h);
    bounds = bounds.createUnion(inputFileBounds);
    bounds = bounds.createUnion(outputFileBounds);
    return bounds;
  }

  /**
   * <p>
   * Obtm o ponto cardeal do ponto em relao ao centro do n.
   * </p>
   *
   * <p>
   * Se o ponto estiver na regio considerada como borda do n (a borda
   * propriamente dita + uma margem interna {@value #BORDER_MARGIN}), este
   * mtodo informa qual  o ponto cardeal relacionado com o ponto.
   * </p>
   *
   * <p>
   * Exemplos, ou seja, se o ponto estiver prximo do canto superior-esquerdo
   * ser {@link CompassDirection#NORTH_WEST}, se ele estiver na borda inferior
   * e longe dos cantos ser {@link CompassDirection#SOUTH}.
   * </p>
   *
   * @param pt O ponto (No aceita {@code null}).
   *
   * @return O ponto cardeal ou {@code null} para um ponto que no esteja na
   *         borda do n.
   */
  public CompassDirection getCompassDirection(final Point2D pt) {
    final double westX =
      this.cornerViews[NORTH_WEST_INDEX].getLocation().getX();
    final double eastX =
      this.cornerViews[NORTH_EAST_INDEX].getLocation().getX();
    final double northY =
      this.cornerViews[NORTH_WEST_INDEX].getLocation().getY();
    final double southY =
      this.cornerViews[SOUTH_WEST_INDEX].getLocation().getY();
    if (isContainedInRectangle(westX, northY, westX + BORDER_MARGIN, northY
      + BORDER_MARGIN, pt)) {
      return CompassDirection.NORTH_WEST;
    }
    if (isContainedInRectangle(eastX - BORDER_MARGIN, northY, eastX, northY
      + BORDER_MARGIN, pt)) {
      return CompassDirection.NORTH_EAST;
    }
    if (isContainedInRectangle(westX, southY - BORDER_MARGIN, westX
      + BORDER_MARGIN, southY, pt)) {
      return CompassDirection.SOUTH_WEST;
    }
    if (isContainedInRectangle(eastX - BORDER_MARGIN, southY - BORDER_MARGIN,
      eastX, southY, pt)) {
      return CompassDirection.SOUTH_EAST;
    }
    if (this.cornerViews[SOUTH_EAST_INDEX].getBounds().contains(pt)) {
      return CompassDirection.SOUTH_EAST;
    }
    if (isContainedInRectangle(westX, northY, eastX, northY + BORDER_MARGIN, pt)) {
      return CompassDirection.NORTH;
    }
    if (isContainedInRectangle(westX, southY - BORDER_MARGIN, eastX, southY, pt)) {
      return CompassDirection.SOUTH;
    }
    if (isContainedInRectangle(eastX - BORDER_MARGIN, northY, eastX, southY, pt)) {
      return CompassDirection.EAST;
    }
    if (isContainedInRectangle(westX, northY, westX + BORDER_MARGIN, southY, pt)) {
      return CompassDirection.WEST;
    }

    return null;
  }

  /**
   * Obtm a altura do n.
   *
   * @return A altura do n.
   */
  public double getHeight() {
    return this.shape.getBounds().getHeight();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getHint(final Point2D pt) {
    /* Arquivo */
    final GraphFileDescriptor fileType = pickFileType(pt);
    if (fileType != null) {
      return fileType.getHint();
    }
    if (contains(pt)) {
      if (this.icon != null) {
        if (this.icon.getBounds2D().contains(pt)) {
          return LNG.get("algorithms.warning.new_version");
        }
      }
      if (this.decoration != null) {
        VO image = this.decoration.getVO();
        if (image != null && image.getBounds2D().contains(pt)) {
          return decoration.getDescription();
        }
      }

      return this.algorithmNameView + " " + this.algorithmVersionIdView + " ("
      + getId() + ")";
    }
    return null;
  }

  /**
   * Obtm a viso de arquivo de entrada que esteja localizada no ponto
   * informado.
   *
   * @param pt O ponto (No aceita {@code null}).
   *
   * @return A viso ou {@code null} se no houver uma viso de arquivo de
   *         entrada no ponto informado.
   */
  public GraphFileDescriptor getInputFileDescriptor(final Point2D pt) {
    for (final GraphFileDescriptor inputFileType : this.inputFileTypeViews) {
      if (inputFileType.contains(pt)) {
        return inputFileType;
      }
    }
    return null;
  }

  /**
   * Obtm a viso de arquivo de entrada que cujo nome do parmetro  igual ao
   * nome informado.
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   *
   * @return A viso ou {@code null} se no houver uma viso de arquivo de
   *         entrada com o nome informado.
   */
  public GraphFileDescriptor getInputFileDescriptor(final String parameterName) {
    for (final GraphFileDescriptor inputFileType : this.inputFileTypeViews) {
      if (inputFileType.getParameterName().equals(parameterName)) {
        return inputFileType;
      }
    }
    return null;
  }

  /**
   * <p>
   * Obtm as vises de arquivo de entrada deste n.
   * </p>
   *
   * <p>
   * A coleo retornada  imutvel (veja
   * {@link Collections#unmodifiableCollection(Collection)}).
   * </p>
   *
   * @return As vises de arquivo de entrada ou uma coleo vazia caso no haja
   *         vises.
   */
  public Collection<GraphFileDescriptor> getInputFileDescriptorCollection() {
    return Collections.unmodifiableCollection(this.inputFileTypeViews);
  }

  /**
   * Indica se este n  capaz de criar um arquivo de log.<br>
   * XXX Ver a questo do arquivo de log
   *
   * @return .
   */
  public boolean canCreateLogFile() {
    if (configuratorView == null) {
      return false;
    }
    final AlgorithmConfigurator configurator =
      configuratorView.getConfigurator();
    final FileURLValue logFile = configurator.getLogFile();
    if (logFile == null) {
      return false;
    }
    return true;
  }

  /**
   * Obtm o arquivo da sada padro do algoritmo.<br>
   * XXX Esse arquivo  URLValue (verificar isso)
   *
   * @return O arquivo da sada padro ou {@code null} se ele no existir.
   */
  public ClientProjectFile getStandardOutputClientProjectFile() {
    if (configuratorView == null) {
      return null;
    }

    final AlgorithmConfigurator configurator =
      configuratorView.getConfigurator();
    final FileURLValue stdout = configurator.getStandardOutputFile();
    if (stdout == null) {
      return null;
    }
    final CommonClientProject project = DesktopFrame.getInstance().getProject();
    final ClientProjectFile projectDirectory = project.getRoot();
    final String[] path = stdout.getPathAsArray();
    ClientProjectFile previousFile = projectDirectory;
    ClientProjectFile currentFile = null;
    for (final String fileName : path) {
      currentFile = GetChildFromNameTask.runTask(previousFile, fileName);
      if (currentFile == null) {
        // Fora um refresh na rvore, pois o arquivo pode existir no servidor,
        // mas ainda no est no cliente.
        if (refreshProjectTree()) {
          return null;
        }
        currentFile = GetChildFromNameTask.runTask(previousFile, fileName);
        if (currentFile == null) {
          return null;
        }
        return null;
      }
      previousFile = currentFile;
    }
    return currentFile;
  }

  /**
   * Atualiza a rvore de projetos.
   *
   * @return {@code true} em caso de sucesso ou {@code false} em caso de falha.
   */
  private boolean refreshProjectTree() {
    final FlowApplicationRemoteTask<Void> task =
      new FlowApplicationRemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        final CommonClientProject project =
          DesktopFrame.getInstance().getProject();
        project.refreshTree();
      }
    };
    final Graph graph = getGraph();
    final Window window = graph.getParentWindow();
    final String title =
      LNG.get(GraphNode.class.getSimpleName() + ".refresh_tree.title");
    final String message =
      LNG.get(GraphNode.class.getSimpleName() + ".refresh_tree.msg");
    return task.execute(window, title, message);
  }

  /**
   * Obtm a viso de arquivo de sada que esteja localizada no ponto informado.
   *
   * @param pt O ponto (No aceita {@code null}).
   *
   * @return A viso ou {@code null} se no houver uma viso de arquivo de sada
   *         no ponto informado.
   */
  public GraphFileDescriptor getOutputFileDescriptor(final Point2D pt) {
    for (final GraphFileDescriptor outputFileType : this.outputFileTypeViews) {
      if (outputFileType.contains(pt)) {
        return outputFileType;
      }
    }
    return null;
  }

  /**
   * Obtm a viso de arquivo de sada que cujo nome do parmetro  igual ao
   * nome informado.
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   *
   * @return A viso ou {@code null} se no houver uma viso de arquivo de sada
   *         com o nome informado.
   */
  public GraphFileDescriptor getOutputFileDescriptor(final String parameterName) {
    for (final GraphFileDescriptor outputFileType : this.outputFileTypeViews) {
      if (outputFileType.getParameterName().equals(parameterName)) {
        return outputFileType;
      }
    }
    return null;
  }

  /**
   * <p>
   * Obtm as vises de arquivo de sada deste n.
   * </p>
   *
   * <p>
   * A coleo retornada  imutvel (veja
   * {@link Collections#unmodifiableCollection(Collection)}).
   * </p>
   *
   * @return As vises de arquivo de sada ou uma coleo vazia caso no haja
   *         vises.
   */
  public Collection<GraphFileDescriptor> getOutputFileDescriptorCollection() {
    return Collections.unmodifiableCollection(this.outputFileTypeViews);
  }

  /**
   * Obtm o rtulo do parmetro cujo nome  igual ao nome indicado.
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   *
   * @return O rtulo.
   *
   * @throws ParameterNotFoundException Se no existir um parmetro com o nome
   *         indicado.
   */
  public String getParameterLabel(final String parameterName)
    throws ParameterNotFoundException {
    final String label = this.parameterLabelsByName.get(parameterName);
    if (label == null) {
      throw new ParameterNotFoundException(parameterName);
    }
    return label;
  }

  /**
   * <p>
   * Obtm o conjunto dos nomes dos parmetros deste n.
   * </p>
   *
   * <p>
   * O conjunto retornado  imutvel (veja
   * {@link Collections#unmodifiableSet(Set)}).
   * </p>
   *
   * @return O conjunto dos nomes ou um conjunto vazio caso no haja parmetros
   *         neste n.
   */
  public Set<String> getParameterNames() {
    return Collections.unmodifiableSet(this.parameterNames);
  }

  /**
   * Obtm o tipo do parmetro cujo nome  igual ao nome indicado.
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   *
   * @return O tipo.
   *
   * @throws ParameterNotFoundException Se no existir um parmetro com o nome
   *         indicado.
   */
  public String getParameterType(final String parameterName)
    throws ParameterNotFoundException {
    final String type = this.parameterTypesByName.get(parameterName);
    if (type == null) {
      throw new ParameterNotFoundException(parameterName);
    }
    return type;
  }

  /**
   * Obtm o valor do parmetro cujo nome  igual ao nome indicado.
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   *
   * @return O valor ({@code null}  um valor vlido).
   *
   * @throws ParameterNotFoundException Se no existir um parmetro com o nome
   *         indicado.
   */
  public String getParameterValue(final String parameterName)
    throws ParameterNotFoundException {
    if (isWellFormed()) {
      return this.configuratorView.getConfigurator().getParameterValue(
        parameterName);
    }
    if (!this.parameterNames.contains(parameterName)) {
      throw new ParameterNotFoundException(parameterName);
    }
    return this.parameterValuesByName.get(parameterName);
  }

  /**
   * Obtm a largura do n.
   *
   * @return A largura do n.
   */
  public double getWidth() {
    return this.shape.getBounds().getWidth();
  }

  /**
   * Obtm a abscissa esquerda do n.
   *
   * @return A abscissa esquerda do n.
   */
  public double getX() {
    return this.shape.getBounds().getX();
  }

  /**
   * Obtm a ordenada superior do n.
   *
   * @return A ordenada superior do n.
   */
  public double getY() {
    return this.shape.getBounds().getY();
  }

  /**
   * Indica se saem conexes deste n.
   *
   * @return {@code true} se saem ou {@code false} caso contrrio.
   */
  public boolean hasLinksFrom() {
    return !getLinkFromCollection().isEmpty();
  }

  /**
   * Indica se chegam conexes neste n.
   *
   * @return {@code true} se chegam ou {@code false} caso contrrio.
   */
  public boolean hasLinksTo() {
    return !getLinkToCollection().isEmpty();
  }

  /**
   * <p>
   * Indica se este n est bem-formado.
   * </p>
   *
   * <p>
   * Um n bem-formado possui um configurador de algoritmos associado a ele.
   * </p>
   *
   * @return {@code true} se est bem-formado ou {@code false} caso contrrio.
   */
  public boolean isWellFormed() {
    return (getAlgorithmConfiguratorView() != null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean msgHandlerVO(final TypeMessage message) {
    if (message instanceof PickNodeMessage) {
      return handlePickNodeMessage((PickNodeMessage) message);
    }
    if (message instanceof ShowParameterValuesMessage) {
      return handleShowParameterValuesMessage();
    }
    if (message instanceof PasteMessage) {
      return handlePasteMessage((PasteMessage) message);
    }

    return super.msgHandlerVO(message);
  }

  /**
   * Redimensiona o n.
   *
   * @param pt O ponto final (No aceita {@code null}).
   * @param compassDirection O sentido do redirecionamento (No aceita
   *        {@code null}).
   */
  public void resize(final Point2D pt, final CompassDirection compassDirection) {
    if (CompassDirection.NORTH.equals(compassDirection)) {
      resizeNorth(pt);
    }
    else if (CompassDirection.SOUTH.equals(compassDirection)) {
      resizeSouth(pt);
    }
    else if (CompassDirection.EAST.equals(compassDirection)) {
      resizeEast(pt);
    }
    else if (CompassDirection.WEST.equals(compassDirection)) {
      resizeWest(pt);
    }
    else if (CompassDirection.NORTH_EAST.equals(compassDirection)) {
      resizeNorthEast(pt);
    }
    else if (CompassDirection.NORTH_WEST.equals(compassDirection)) {
      resizeNorthWest(pt);
    }
    else if (CompassDirection.SOUTH_EAST.equals(compassDirection)) {
      resizeSouthEast(pt);
    }
    else if (CompassDirection.SOUTH_WEST.equals(compassDirection)) {
      resizeSouthWest(pt);
    }
    for (final GraphElementListener listener : getListenerList()) {
      listener.wasResized(this);
    }
  }

  /**
   * Atribui o valor a um dos parmetros deste n.
   *
   * @param parameterName Nome do parmetro.
   * @param parameterValue Valor do parmetro.
   *
   * @throws ParseException Se o valor passado no puder ser convertido para o
   *         valor armazenado no parmetro. Exemplo: quando tentamos atribuir o
   *         texto <i>abcd</i> em um parmetro que aceita apenas valores
   *         inteiros.
   * @throws ParameterNotFoundException Se o parmetro com o nome especificado
   *         no exisitr no n.
   */
  public void setParameterValue(String parameterName, String parameterValue)
    throws ParseException, ParameterNotFoundException {
    if (isWellFormed()) {
      final AlgorithmConfigurator configurator =
        this.configuratorView.getConfigurator();
      configurator.setParameterValue(parameterName, parameterValue);
    }
    else {
      if (this.parameterNames.contains(parameterName)) {
        this.parameterValuesByName.put(parameterName, parameterValue);
      }
    }

  }

  /**
   * Atribui os valores dos parmetros a este n.
   *
   * @param parameterValuesByName Uma tabela que associa os valores dos
   *        parmetros aos seus nomes (No aceita {@code null}).</br> Os
   *        parmetros que no esto representados nesta tabela tero valor
   *        {@code null} e valores {@code null} so vlidos.
   *
   * @throws ParseException Se o valor passado no puder ser convertido para o
   *         valor armazenado no parmetro. Exemplo: quando tentamos atribuir o
   *         texto <i>abcd</i> em um parmetro que aceita apenas valores
   *         inteiros.
   * @throws ParameterNotFoundException Caso no exista um parmetro com o nome
   *         fornecido.
   */
  public void setParameterValuesByName(
    final Map<String, String> parameterValuesByName) throws ParseException,
    ParameterNotFoundException {
    if (isWellFormed()) {
      final AlgorithmConfigurator configurator =
        this.configuratorView.getConfigurator();
      configurator.setParameterValuesByName(parameterValuesByName);
    }
    else {
      for (final String parameterName : parameterValuesByName.keySet()) {
        final String parameterValue = parameterValuesByName.get(parameterName);
        if (this.parameterNames.contains(parameterName)) {
          this.parameterValuesByName.put(parameterName, parameterValue);
        }
      }
    }
  }

  /**
   * Retorna uma mensagem de erro informando porque o nome est mal-formado (o
   * algoritmo no existe ou a verso no existe).
   *
   * @return A mensagem de erro.
   */
  protected LocalizedMessage getErrorMessage() {
    LocalizedMessage errorMessage;
    if (this.algorithm == null) {
      errorMessage =
        new LocalizedMessage(GraphNode.class, "error_algorithm_not_exists",
          new Object[] { this.algorithmNameView.getText() });
    }
    else {
      errorMessage =
        new LocalizedMessage(GraphNode.class, "error_version_not_exists",
          new Object[] { this.algorithmNameView.getText(),
          this.algorithmVersionIdView });
    }
    return errorMessage;
  }

  /**
   * Exibe uma mensagem de erro informando porque o nome est mal-formado (o
   * algoritmo no existe ou a verso no existe).
   */
  public void showError() {
    LocalizedMessage errorMessage = getErrorMessage();
    new ErrorMessage(errorMessage.getKey(), errorMessage.getArgs())
    .sendVS(this);
  }

  /**
   * Exibe os valores dos parmetros do n.
   */
  public void showParameterValues() {
    if (this.configuratorView != null) {
      this.configuratorView.setEnabled(false);
      this.configuratorView.launch();
    }
    else {
      new NodeParameterViewer(this).execute();
    }
  }

  /**
   * Preende o nicio da conexo a um arquivo de sada deste n.
   *
   * @param link A conexo (No aceita {@code null}).
   *
   * @param pt O ponto (No aceita {@code null}).
   *
   * @return {@code true} em caso de sucesso ou {@code false} se no h um
   *         arquivo de entrada no ponto informado.
   */
  public boolean startLink(final GraphLink link, final Point2D pt) {
    for (final GraphFileDescriptor outputFileType : this.outputFileTypeViews) {
      if (outputFileType.contains(pt)) {
        return link.start(outputFileType);
      }
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return this.algorithmNameView.getText();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void turnOffHighlight() {
    for (final GraphFileDescriptor inputFileType : this.inputFileTypeViews) {
      inputFileType.turnOffHightLight();
    }
    for (final GraphFileDescriptor outputFileType : this.outputFileTypeViews) {
      outputFileType.turnOffHightLight();
    }
    this.borderColor = DEFAULT_FOREGROUNG_COLOR;
    Color textColor;
    if (isBypassed() && canBeBypassed()) {
      textColor = BYPASSED_TEXT_COLOR;
    }
    else {
      textColor = DEFAULT_FOREGROUNG_COLOR;
    }
    this.algorithmNameView.setColor(textColor);
    this.algorithmVersionIdView.setColor(textColor);
  }

  /**
   * Destaca o n.
   */
  public void turnOnHighlight() {

    for (final GraphFileDescriptor inputFileType : this.inputFileTypeViews) {
      inputFileType.turnOnHightLight();
    }
    for (final GraphFileDescriptor outputFileType : this.outputFileTypeViews) {
      outputFileType.turnOnHightLight();
    }
    this.borderColor = HIGHT_LIGHT_COLOR;
    this.algorithmNameView.setColor(HIGHT_LIGHT_COLOR);
    this.algorithmVersionIdView.setColor(HIGHT_LIGHT_COLOR);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean turnOnHighlight(final Point2D pt) {
    if (!contains(pt)) {
      return false;
    }

    for (final GraphFileDescriptor inputFileType : this.inputFileTypeViews) {
      if (inputFileType.contains(pt)) {
        inputFileType.turnOnHightLight();
        return true;
      }
    }
    for (final GraphFileDescriptor outputFileType : this.outputFileTypeViews) {
      if (outputFileType.contains(pt)) {
        outputFileType.turnOnHightLight();
        return true;
      }
    }

    this.borderColor = HIGHT_LIGHT_COLOR;
    this.algorithmNameView.setColor(HIGHT_LIGHT_COLOR);
    this.algorithmVersionIdView.setColor(HIGHT_LIGHT_COLOR);
    return true;
  }

  /**
   * <p>
   * Valida os valores dos parmetros e conexes deste n.
   * </p>
   *
   * <p>
   * Ele pode exibir mensagens de com o motivo do erro de validao. Ele
   * atualiza a cor do n para estar de acordo com o estado (pronto para
   * executar ou no-pronto para executar).
   * </p>
   *
   * <p>
   * O n est invlido se  mal-formado ou se o configurador de algoritmos for
   * invlido ou se pelo menos um dos arquivos de entrada ou de sada forem
   * invlidos.
   * </p>
   *
   * <p>
   * O configurador  invlido se os valores dos parmetros no forem adequados.
   * </p>
   *
   * <p>
   * Os arquivos so invlidos se esto associados a conexes mal-formadas.
   * </p>
   *
   * @param mode Modo de validao ({@link ValidationMode#FULL} ou
   *        {@link ValidationMode#ALLOW_EMPY_VALUES}).
   *
   * @return O resultado da validao.
   */
  @Override
  public ViewValidationResult validate(final ValidationMode mode) {
    if (!isBypassed()) {
      if (!isWellFormed()) {
        final LocalizedMessage errorMessage = getErrorMessage();
        return new ViewValidationResult(errorMessage, this);
      }
      ViewValidationResult validation = validateInputFiles();
      if (!validation.isWellSucceded()) {
        return validation;
      }
      validation = validateOutputFiles();
      if (!validation.isWellSucceded()) {
        return validation;
      }
      validation = validateAlgorithmConfigurator(mode);
      if (!validation.isWellSucceded()) {
        return validation;
      }
    }
    return new ViewValidationResult(this);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean contains(final Point2D pt) {
    return getBounds2D().contains(pt);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void doDrag(final double tx, final double ty) {
    final double x = this.northWestPoint.getX() + tx;
    final double y = this.northWestPoint.getY() + ty;
    this.northWestPoint.setLocation(x, y);
    if (this.icon != null || this.decoration != null) {
      updateImages();
    }
    createShape();
    updateFileTypes();
    updateTextLocation();
    adjustGrid();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean doDrag(final Point2D startPoint, final Point2D endPoint) {
    final double tx = endPoint.getX() - startPoint.getX();
    final double ty = endPoint.getY() - startPoint.getY();
    final Point2D newPoint =
      new Point2D.Double(this.northWestPoint.getX() + tx, this.northWestPoint
        .getY()
        + ty);
    this.northWestPoint.setLocation(newPoint);
    if (this.icon != null || this.decoration != null) {
      updateImages();
    }
    createShape();
    updateFileTypes();
    updateTextLocation();
    adjustGrid();
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void doDrop(final Point2D pt) {
    doDrag(getFirstPoint(), Grid.adjustPoint(getFirstPoint()));
    adjustFileTypes();
  }

  /**
   * Atualiza a posio das vises de arquivo para que elas se alinhem a grade.
   */
  private void adjustFileTypes() {

    for (final GraphFileDescriptor descriptor : this.inputFileTypeViews) {
      descriptor.adjustGrid();
    }
    for (final GraphFileDescriptor descriptor : this.outputFileTypeViews) {
      descriptor.adjustGrid();
    }
  }

  /**
   * Indica se  possvel chegar ao n informado navegando pelas conexes que se
   * originam deste n.
   *
   * @param node O n (No aceita {@code null}).
   *
   * @return {@code true} se  possvel chegar at o n ou {@code false} caso
   *         contrrio.
   */
  boolean canReach(final GraphNode node) {
    if (equals(node)) {
      return true;
    }
    for (final GraphLink link : getLinkFromCollection()) {
      final GraphNode endNode = link.getEndNode();
      if (endNode != null) {
        if (endNode.canReach(node)) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * <p>
   * Obtm todas conexes que esto ligadas a este n.
   * </p>
   *
   * <p>
   * A coleo retorna  imutvel (veja
   * {@link Collections#unmodifiableCollection(Collection)}).
   * </p>
   *
   * @return A coleo das conexes ou uma coleo vazia caso no haja conexes
   *         ligadas a este n.
   */
  public Collection<GraphLink> getLinkCollection() {
    final Set<GraphLink> linkSet = new HashSet<GraphLink>();
    linkSet.addAll(getLinkFromCollection());
    linkSet.addAll(getLinkToCollection());
    return linkSet;
  }

  /**
   * <p>
   * Obtm todas conexes que se originam deste n.
   * </p>
   *
   * <p>
   * A coleo retorna  imutvel (veja
   * {@link Collections#unmodifiableCollection(Collection)}).
   * </p>
   *
   * @return A coleo das conexes ou uma coleo vazia caso no haja conexes
   *         ligadas a este n.
   */
  public Collection<GraphLink> getLinkFromCollection() {
    final Collection<GraphLink> linkFromCollection =
      new LinkedList<GraphLink>();
    for (final GraphFileDescriptor outputFileType : this.outputFileTypeViews) {
      linkFromCollection.addAll(outputFileType.getLinkFromCollection());
    }
    return linkFromCollection;
  }

  /**
   * <p>
   * Obtm todas conexes que se destinam a este n.
   * </p>
   *
   * <p>
   * A coleo retorna  imutvel (veja
   * {@link Collections#unmodifiableCollection(Collection)}).
   * </p>
   *
   * @return A coleo das conexes ou uma coleo vazia caso no haja conexes
   *         ligadas a este n.
   */
  public Collection<GraphLink> getLinkToCollection() {
    final Collection<GraphLink> linkToCollection = new LinkedList<GraphLink>();

    for (final GraphFileDescriptor inputFileType : this.inputFileTypeViews) {
      final GraphLink link = inputFileType.getLinkTo();
      if (link != null) {
        linkToCollection.add(link);
      }
    }
    return linkToCollection;
  }

  /**
   * Verifica se este n est pronto para executar.
   *
   * @return {@code true} se estiver pront ou {@code false} caso contrrio.
   */
  boolean isReadyToExecute() {
    return this.backgroundColor.equals(READY_BACKGROUND_COLOR);
  }

  /**
   * Ajusta a forma do n para que ele se alinhe a grade.
   */
  private void adjustGrid() {
    Double tX = null;
    Double tY = null;
    if (getBounds2D().getMinX() < 0) {
      tX = -1 * getBounds2D().getMinX();
    }
    if (getBounds2D().getMinY() < 0) {
      tY = -1 * getBounds2D().getMinY();
    }
    if (tX == null && tY == null) {
      return;
    }
    if (tX == null) {
      tX = 0.0;
    }
    if (tY == null) {
      tY = 0.0;
    }
    doDrag(tX, tY);
  }

  /**
   * Obtm a dimenso mnima do n. O tamanho varia de acordo com os itens que
   * esto sendo mostrados.
   *
   * @return a dimenso mnima.
   */
  public Dimension getMinimumSize() {
    if (!canComputeSize()) {
      return null;
    }
    double width = getMinimumWidth() + getImagesWidth();
    double height = getMinimumHeight();
    Dimension size = new Dimension();
    size.setSize(width, height);
    return size;
  }

  /**
   * Calcula as medidas de todos os componentes visuais que esto no n (cantos,
   * os textos e a forma dele). Pode no ser possvel computar as medidas da
   * caixa (p.ex. quando os componentes ainda no tiveram seus tamanhos
   * computados). Nesse caso o mtodo retorna o vaor falso.
   *
   * @param centerPoint O ponto central (No aceita {@code null}).
   * @return verdadeiro se foi possvel computar o tamanho da caixa ou falso,
   *         caso contrrio.
   */
  private boolean computeMeasures(final Point2D centerPoint) {
    if (!canComputeSize()) {
      return false;
    }
    Dimension size = getMinimumSize();
    if (shape != null) {
      Rectangle previousSize = shape.getBounds();
      double width = Math.max(size.getWidth(), previousSize.getWidth());
      double height = Math.max(size.getHeight(), previousSize.getHeight());
      size.setSize(width, height);
    }
    Point2D origin = getOriginFromCenter(centerPoint, size);
    origin = Grid.adjustPoint(origin);
    setBounds2D(origin.getX(), origin.getY(), size.getWidth(), size.getHeight());
    return true;
  }

  /**
   * Determina a posio do n a partir do seu ponto central.
   *
   * @param centerPoint o novo ponto central do n.
   */
  public void setLocation(final Point2D centerPoint) {
    if (shape != null) {
      Dimension previousSize = shape.getBounds().getSize();
      Point2D newOrigin = getOriginFromCenter(centerPoint, previousSize);
      newOrigin = Grid.adjustPoint(newOrigin);
      setBounds2D(newOrigin.getX(), newOrigin.getY(), previousSize.getWidth(),
        previousSize.getHeight());
    }
  }

  /**
   * Obtm o ponto de "origem" do n a partir do seu ponto central e seu
   * tamanho. O ponto de origem corresponde ao canto superior esquerdo da sua
   * borda.
   *
   * @param centerPoint o ponto centra do n.
   * @param size o tamanho do n.
   * @return o ponto de origem.
   */
  public Point2D getOriginFromCenter(final Point2D centerPoint, Dimension2D size) {
    double width = size.getWidth();
    double height = size.getHeight();
    double centerX = centerPoint.getX();
    double centerY = centerPoint.getY();
    double x0 = Math.max(0, centerX - width / 2);
    double y0 = Math.max(0, centerY - height / 2);
    return new Point2D.Double(x0, y0);
  }

  /**
   * Determina se j  possvel computar o tamanho da caixa que representa o n.
   *
   * @return verdadeiro se for possvel computar o tamanho da caixa ou falso,
   *         caso contrrio.
   */
  private boolean canComputeSize() {
    return algorithmNameView.getBounds2D().getWidth() != 0
      && algorithmNameView.getBounds2D().getHeight() != 0;
  }

  /**
   * Obtm a altura mnima do n. A altura varia de acordo com os itens que
   * esto sendo mostrados.
   *
   * @return a altura mnima.
   */
  private double getMinimumHeight() {
    double height = 0;
    double nameHeight = algorithmNameView.getBounds2D().getHeight();
    if (nameHeight > 0) {
      height += nameHeight;
    }
    if (isVersionInfoVisible()) {
      double versionHeight = algorithmVersionIdView.getBounds2D().getHeight();
      if (versionHeight > 0) {
        height += versionHeight;
      }
    }
    return height = Math.max(height, MIN_HEIGHT);
  }

  /**
   * Obtm a largura mnima do n. A largura varia de acordo com os itens que
   * esto sendo mostrados.
   *
   * @return a largura mnima.
   */
  private double getMinimumWidth() {
    double width = 0;
    double nameWidth = algorithmNameView.getBounds2D().getWidth();
    double versionWidth = 0;
    if (isVersionInfoVisible()) {
      versionWidth = this.algorithmVersionIdView.getBounds2D().getWidth();
    }
    width = Math.max(nameWidth, versionWidth);
    double minWidth = Math.max(MIN_WIDTH, width);
    return minWidth;
  }

  /**
   * Obtm a largura total ocupada pelas imagens mostradas no n.
   *
   * @return a largura das imagens combinadas.
   */
  private double getImagesWidth() {
    double iconWidth = getIconDimension().getWidth();
    double decorationWidth = getDecorationDimension().getWidth();
    double imagesWidth = iconWidth + decorationWidth;
    /*
     * Se pelo menos uma das imagens no estiver visvel (tamanho zero), dobra o
     * tamanho da imagem que estiver visvel para a caixa ficar simtrica. No
     * caso das duas estarem invisveis, o tamanho total vai continuar zero.
     */
    if (iconWidth == 0 || decorationWidth == 0) {
      imagesWidth *= 2;
    }
    return imagesWidth;
  }

  /**
   * Obtm as dimenses da decorao do n.
   *
   * @return as dimenses da decorao ou (0,0) caso no exista decorao.
   */
  private Dimension getDecorationDimension() {
    Dimension size = new Dimension(0, 0);
    if (this.decoration != null) {
      double h = Math.min(this.decoration.getHeight(), MAX_IMAGE_DIMENSION);
      double w = Math.min(this.decoration.getWidth(), MAX_IMAGE_DIMENSION);
      size.setSize(w, h);
    }
    return size;
  }

  /**
   * Obtm as dimenses do cone do n.
   *
   * @return as dimensesdo cone ou (0,0) caso no exista cone.
   */
  private Dimension getIconDimension() {
    Dimension size = new Dimension(0, 0);
    if (this.icon != null) {
      double h = Math.min(this.icon.getHeight(), MAX_IMAGE_DIMENSION);
      double w = Math.min(this.icon.getWidth(), MAX_IMAGE_DIMENSION);
      size.setSize(w, h);
    }
    return size;
  }

  /**
   * Cria as vises dos cantos.
   */
  private void createCornerPoints() {
    final Rectangle2D boundBox = this.shape.getBounds2D();
    this.cornerViews = new GraphPoint[4];
    this.cornerViews[NORTH_WEST_INDEX] =
      new GraphPoint(boundBox.getX(), boundBox.getY());
    this.cornerViews[NORTH_EAST_INDEX] =
      new GraphPoint(boundBox.getX() + boundBox.getWidth(), boundBox.getY());
    this.cornerViews[SOUTH_EAST_INDEX] =
      new GraphPoint(boundBox.getX() + boundBox.getWidth(), boundBox.getY()
        + boundBox.getHeight());
    this.cornerViews[SOUTH_WEST_INDEX] =
      new GraphPoint(boundBox.getX(), boundBox.getY() + boundBox.getHeight());
  }

  /**
   * <p>
   * Cria as vises dos arquivos de entrada.
   * </p>
   *
   * <p>
   * S deve ser invocado em ns bem-formados.
   * </p>
   */
  private void createInputFileTypes() {
    /* Arquivos */
    this.inputFileTypeViews = new ArrayList<GraphFileDescriptor>();
    for (final InputFileParameter inputFileParameter : this.configuratorView
      .getConfigurator().getInputFileParameters()) {
      if (inputFileParameter.usesPipe().equals(
          FileParameterPipeAcceptance.TRUE) || inputFileParameter.usesPipe()
          .equals(FileParameterPipeAcceptance.ALWAYS)) {
        GraphFileDescriptor graphFileDescriptor =
          new GraphFileDescriptor(inputFileParameter, this);
        this.inputFileTypeViews.add(graphFileDescriptor);
      }
    }
    /* URLs */
    for (final InputURLParameter inputURLParameter : this.configuratorView
      .getConfigurator().getInputURLParameters()) {
      if ((inputURLParameter.getMode().equals(FileParameterMode.REGULAR_FILE) || inputURLParameter
        .getMode().equals(FileParameterMode.DIRECTORY))
        && (inputURLParameter.usesPipe().equals(
          FileParameterPipeAcceptance.TRUE) || inputURLParameter.usesPipe()
          .equals(FileParameterPipeAcceptance.ALWAYS))) {
        GraphFileDescriptor graphFileDescriptor =
          new GraphFileDescriptor(inputURLParameter, this);
        this.inputFileTypeViews.add(graphFileDescriptor);
      }
    }
  }

  /**
   * <p>
   * Cria as vises dos arquivos de sada.
   * </p>
   *
   * <p>
   * S deve ser invocado em ns bem-formados.
   * </p>
   */
  private void createOutputFileTypes() {
    /* Arquivos */
    this.outputFileTypeViews = new ArrayList<GraphFileDescriptor>();
    for (final OutputFileParameter outputFileParameter : this.configuratorView
      .getConfigurator().getOutputFileParameters()) {
      if (!outputFileParameter.isLogFile()
        && (outputFileParameter.getMode()
          .equals(FileParameterMode.REGULAR_FILE) || (outputFileParameter
            .getMode().equals(FileParameterMode.DIRECTORY)))
            && (outputFileParameter.usesPipe().equals(
              FileParameterPipeAcceptance.TRUE) || outputFileParameter.usesPipe()
              .equals(FileParameterPipeAcceptance.ALWAYS))) {
        GraphFileDescriptor graphFileDescriptor =
          new GraphFileDescriptor(outputFileParameter, this);
        this.outputFileTypeViews.add(graphFileDescriptor);
      }
    }
    /* URLs */
    for (final OutputURLParameter outputURLParameter : this.configuratorView
      .getConfigurator().getOutputURLParameters()) {
      if ((outputURLParameter.getMode().equals(FileParameterMode.REGULAR_FILE) || (outputURLParameter
        .getMode().equals(FileParameterMode.DIRECTORY)))
        && (outputURLParameter.usesPipe().equals(
          FileParameterPipeAcceptance.TRUE) || outputURLParameter.usesPipe()
          .equals(FileParameterPipeAcceptance.ALWAYS))) {
        GraphFileDescriptor graphFileDescriptor =
          new GraphFileDescriptor(outputURLParameter, this);
        this.outputFileTypeViews.add(graphFileDescriptor);
      }
    }
  }

  /**
   * Cria a forma do n.
   */
  private void createShape() {
    final double w = this.dimension.getWidth();
    final double h = this.dimension.getHeight();
    final double x = this.northWestPoint.getX();
    final double y = this.northWestPoint.getY();
    final double arc = Math.min(w, h) * 0.3;
    final double dx = w * (SHADOW_FACTOR - 1.0);
    final double dy = h * (SHADOW_FACTOR - 1.0);
    final double d = Math.min(dx, dy);
    this.shape = new RoundRectangle2D.Double(x, y, w, h, arc, arc);
    this.shadowShape =
      new RoundRectangle2D.Double(x + d, y + d, w, h, arc, arc);

    createCornerPoints();
  }

  /**
   * Cria uma caixa envolvente que engloba todos as vises de arquivo.
   *
   * @param fileTypeCollection A coleo de vises de arquivo (No aceita
   *        {@code null}).
   *
   * @return A caixa envolvente.
   */
  private Rectangle2D getBoundsFileTypes(
    final Collection<GraphFileDescriptor> fileTypeCollection) {
    Rectangle2D bounds = this.shape.getBounds2D();
    for (final GraphFileDescriptor fileType : fileTypeCollection) {
      bounds = bounds.createUnion(fileType.getBounds2D());
    }
    return bounds;
  }

  /**
   * Obtm o ponto central do n.
   *
   * @return O ponto central.
   */
  private Point2D getCenterPoint() {
    final Point2D centerPoint = new Point();
    centerPoint.setLocation(getBounds2D().getCenterX(), getBounds2D()
      .getCenterY());
    return centerPoint;
  }

  /**
   * Obtm o ponto superior-esquerdo deste n.
   *
   * @return O ponto superior-esquerdo.
   */
  private Point2D getFirstPoint() {
    final Point2D firstPoint = new Point();
    firstPoint.setLocation(this.shape.getBounds2D().getMinX(), this.shape
      .getBounds2D().getMinY());
    return firstPoint;
  }

  /**
   * Trata mensagens do tipo Pegar N.
   *
   * @param message A mensagem (No aceita {@code null}).
   *
   * @return {@code true} se a mensagem era para pegar este n ou {@code false}
   *         se a mensagem no era para pegar este n.
   */
  private boolean handlePickNodeMessage(final PickNodeMessage message) {
    if (!contains(message.getPoint())) {
      return false;
    }
    message.setNode(this);
    return true;
  }

  /**
   * Trata a mensagem exibir valores dos parmetros se estiver selecionado.
   *
   * @return {@code true} se ele tratou a mensagem ou {@code false} se ele no
   *         tratou a mensagem.
   */
  private boolean handleShowParameterValuesMessage() {
    if (isSelected()) {
      showParameterValues();
      return true;
    }
    return false;
  }

  /**
   * Trata a mensagem do tipo "colar".
   *
   * @param pasteMessage a mensagem.
   * @return {@code true} se ele tratou a mensagem ou {@code false} se ele no
   *         tratou a mensagem.
   */
  private boolean handlePasteMessage(PasteMessage pasteMessage) {
    if (!isSelected()) {
      return false;
    }
    final Transferable transferable = pasteMessage.getTransferable();
    if (transferable instanceof ParametersTransferable) {
      ParametersTransferable parametersTransferable =
        (ParametersTransferable) transferable;
      final CopyAndPasteOperation operation =
        new CopyAndPasteOperation(parametersTransferable, this);
      final OperationStatus status = operation.execute();
      if (status.isConfirmed()) {
        if (status.hasWarnings() || status.hasErrors()) {
          askForParameterValues();
        }
        pasteMessage.setAffectedElement(this);
      }
    }
    return true;
  }

  /**
   * Pega uma viso de arquivo.
   *
   * @param pt O ponto (No aceita {@code null}).
   *
   * @return A viso para o arquivo ou {@code null} caso no haja uma viso no
   *         ponto informado.
   */
  private GraphFileDescriptor pickFileType(final Point2D pt) {
    final GraphFileDescriptor fileType =
      pickFileType(pt, this.outputFileTypeViews);
    if (fileType != null) {
      return fileType;
    }
    return pickFileType(pt, this.inputFileTypeViews);
  }

  /**
   * Pega uma viso de arquivo entre as vises informadas.
   *
   * @param pt O ponto (No aceita {@code null}).
   * @param fileTypeCollection A coleo de viso de arquivos (No aceita
   *        {@code null}).
   *
   * @return A viso para o arquivo ou {@code null} caso no haja uma viso no
   *         ponto informado.
   */
  private GraphFileDescriptor pickFileType(final Point2D pt,
    final Collection<GraphFileDescriptor> fileTypeCollection) {
    for (final GraphFileDescriptor fileType : fileTypeCollection) {
      if (fileType.pick(pt) != null) {
        return fileType;
      }
    }
    return null;
  }

  /**
   * Redimensiona o n alterando a posio da borda leste.
   *
   * @param pt O ponto (No aceita {@code null}).
   */
  private void resizeEast(final Point2D pt) {
    final Rectangle2D bounds = this.shape.getBounds2D();
    final double newX = pt.getX();
    final double eastX = bounds.getMaxX();
    final double x = bounds.getMinX();
    final double y = bounds.getMinY();
    final double height = bounds.getHeight();
    final double width = bounds.getWidth();
    double newWidth = width + newX - eastX;
    double minWidth = MIN_WIDTH + getImagesWidth();
    if (newWidth < minWidth) {
      newWidth = minWidth;
    }
    setBounds2D(x, y, newWidth, height);
  }

  /**
   * Redimensiona o n alterando a posio da borda norte.
   *
   * @param pt O ponto (No aceita {@code null}).
   */
  private void resizeNorth(final Point2D pt) {
    final Rectangle2D bounds = this.shape.getBounds2D();
    final double x = bounds.getMinX();
    final double northY = bounds.getMinY();
    double newY = pt.getY();
    final double width = bounds.getWidth();
    final double height = bounds.getHeight();
    double newHeight = height + northY - newY;
    double minHeight = getMinimumHeight();
    if (newHeight < minHeight) {
      newY = bounds.getMaxY() - minHeight;
      newHeight = minHeight;
    }
    setBounds2D(x, newY, width, newHeight);
  }

  /**
   * Redimensiona o n alterando a posio do canto nordeste.
   *
   * @param pt O ponto (No aceita {@code null}).
   */
  private void resizeNorthEast(final Point2D pt) {
    resizeNorth(pt);
    resizeEast(pt);
  }

  /**
   * Redimensiona o n alterando a posio do canto noroeste.
   *
   * @param pt O ponto (No aceita {@code null}).
   */
  private void resizeNorthWest(final Point2D pt) {
    resizeNorth(pt);
    resizeWest(pt);
  }

  /**
   * Redimensiona o n alterando a posio da borda sul.
   *
   * @param pt O ponto (No aceita {@code null}).
   */
  private void resizeSouth(final Point2D pt) {
    final Rectangle2D bounds = this.shape.getBounds2D();
    final double x = bounds.getMinX();
    final double y = bounds.getMinY();
    final double newY = pt.getY();
    final double southY = bounds.getMaxY();
    final double width = bounds.getWidth();
    final double height = bounds.getHeight();
    double newHeight = height + newY - southY;
    double minHeight = getMinimumHeight();
    if (newHeight < minHeight) {
      newHeight = minHeight;
    }
    setBounds2D(x, y, width, newHeight);
  }

  /**
   * Redimensiona o n alterando a posio do canto sudeste.
   *
   * @param pt O ponto (No aceita {@code null}).
   */
  private void resizeSouthEast(final Point2D pt) {
    resizeSouth(pt);
    resizeEast(pt);
  }

  /**
   * Redimensiona o n alterando a posio do canto sudoeste.
   *
   * @param pt O ponto (No aceita {@code null}).
   */
  private void resizeSouthWest(final Point2D pt) {
    resizeSouth(pt);
    resizeWest(pt);
  }

  /**
   * Redimensiona o n alterando a posio da borda oeste.
   *
   * @param pt O ponto (No aceita {@code null}).
   */
  private void resizeWest(final Point2D pt) {
    final Rectangle2D bounds = this.shape.getBounds2D();
    double newX = pt.getX();
    final double westX = bounds.getMinX();
    final double y = bounds.getMinY();
    final double height = bounds.getHeight();
    final double width = bounds.getWidth();
    double newWidth = width + westX - newX;
    double minWidth = MIN_WIDTH + getImagesWidth();
    if (newWidth < minWidth) {
      newWidth = minWidth;
      newX = bounds.getMaxX() - minWidth;
    }
    setBounds2D(newX, y, newWidth, height);
  }

  /**
   * Modifica a posio e as dimenses do n.
   *
   * @param xBnd A abscissa esquerda.
   * @param yBnd A ordenada direita.
   * @param wBnd A largura.
   * @param hBnd A altura.
   */
  protected void setBounds2D(final double xBnd, final double yBnd,
    final double wBnd, final double hBnd) {
    double imagesWidth = getImagesWidth();
    double width = Math.max(wBnd, MIN_WIDTH + getImagesWidth());
    double height = Math.max(hBnd, MIN_HEIGHT);
    double x = Math.max(xBnd, 0);
    double y = Math.max(yBnd, 0);

    final Rectangle2D.Double rect =
      new Rectangle2D.Double(xBnd, yBnd, wBnd, hBnd);
    final Rectangle2D newBounds = Grid.adjustBounds(rect);
    x = newBounds.getMinX();
    y = newBounds.getMinY();
    width = newBounds.getWidth();
    height = newBounds.getHeight();
    this.northWestPoint = new Point();
    this.northWestPoint.setLocation(x, y);
    this.dimension = new Dimension();
    this.dimension.setSize(width, height);
    if (this.icon != null || this.decoration != null) {
      updateImages();
    }

    createShape();
    updateFileTypes();
    adjustFileTypes();
    double maxTextWidth = width - imagesWidth;
    this.algorithmNameView.setMaxWidth(maxTextWidth);
    this.algorithmVersionIdView.setMaxWidth(maxTextWidth);
    updateTextLocation();
    repaint();
  }

  /**
   * Configura os observadores.
   */
  private void setListeners() {
    getGraph().addGraphListener(new GraphListener() {
      @Override
      public void wasChangedWorkspace(final Graph graph) {
        // Ignora este evento.
      }

      @Override
      public void wasElementCreated(final Graph graph,
        final GraphElement element) {
        // Ignora este evento.
      }

      @Override
      public void wasElementDragged(final Graph graph,
        final GraphElement element, final double tx, final double ty) {
        // Ignora este evento.
      }

      @Override
      public void wasElementDragged(final Graph graph,
        final GraphElement element, final Point2D startPoint,
        final Point2D endPoint) {
        // Ignora este evento.
      }

      @Override
      public void wasElementDropped(final Graph graph,
        final GraphElement element, final Point2D pt) {
        // Ignora este evento.
      }

      @Override
      public void wasElementParametrized(final Graph graph,
        final GraphElement element) {
        // Ignora este evento.
      }

      @Override
      public void wasElementRemoved(final Graph graph,
        final GraphElement element) {
        // Ignora este evento.
      }

      @Override
      public void wasElementSelected(final Graph graph,
        final GraphElement element) {
        // Ignora este evento.
      }

      @Override
      public void wasLinkIncreased(final Graph graph, final GraphLink link) {
        // Ignora este evento.
      }

      @Override
      public void wasNodeResized(final Graph graph, final GraphNode node) {
        // Ignora este evento.
      }

      @Override
      public void wasParameterSetEnabled(final Graph graph,
        final GraphNode node, final String parameterName,
        final boolean isEnabled) {
        // Ignora este evento.
      }

      @Override
      public void wasReseted(final Graph graph) {
        // Ignora este evento.
      }

      @Override
      public void wasParameterSetVisible(final Graph graph,
        final GraphNode node, final String parameterName,
        final boolean isVisible) {
        // Ignora este evento.
      }

      @Override
      public void wasLinkAnchored(final Graph graph, final GraphLink link,
        final GraphFileDescriptor fileDescriptor) {
        validateIfLinkedTo(link);
      }

      @Override
      public void wasLinkStatusChanged(final Graph graph, final GraphLink link) {
        validateIfLinkedTo(link);
      }

      @Override
      public void wasLinkUnanchored(final Graph graph, final GraphLink link,
        final GraphFileDescriptor fileDescriptor) {
        validateIfLinkedTo(link);
      }
    });
    this.configuratorView
    .addAlgorithmConfiguratorViewListener(new AlgorithmConfiguratorViewListener() {

      @Override
      public void wasConfirmed(final AlgorithmConfiguratorView view) {
        for (final GraphElementListener listener : getListenerList()) {
          listener.wasParametrized(GraphNode.this);
        }
      }

      @Override
      public void wasCancelled(final AlgorithmConfiguratorView view) {
        // Ignora este evento.
      }
    });
    final AlgorithmConfigurator algorithmConfigurator =
      this.configuratorView.getConfigurator();
    algorithmConfigurator
    .addAlgorithmConfiguratorListener(new AlgorithmConfiguratorListener() {
      @Override
      public void parameterLabelWasChanged(
        final AlgorithmConfigurator configurator, final String name,
        final String label) {
        // Ignora este evento.
      }

      @Override
      public void parameterWasSetEnabled(
        final AlgorithmConfigurator configurator, final String parameterName,
        final boolean parameterIsEnabled) {
        fullValidation();
        for (final GraphElementListener listener : getListenerList()) {
          listener.wasParameterSetEnabled(GraphNode.this, parameterName,
            parameterIsEnabled);
        }
      }

      @Override
      public void parameterWasSetVisible(
        final AlgorithmConfigurator configurator, final String parameterName,
        final boolean parameterIsVisible) {
        fullValidation();
        for (final GraphElementListener listener : getListenerList()) {
          listener.wasParameterSetVisible(GraphNode.this, parameterName,
            parameterIsVisible);
        }
      }

      @Override
      public <V> void parameterValueWasChanged(
        final AlgorithmConfigurator configurator, final String parameterName,
        final V parameterValue) {
        fullValidation();
      }

      @Override
      public void wasSetEnabled(final AlgorithmConfigurator configurator) {
        // Ignora este evento.
      }
    });
  }

  /**
   * Valida o n se este fizer parte de uma determinada ligao.
   *
   * @param link a ligao.
   */
  private void validateIfLinkedTo(final GraphLink link) {
    GraphNode endNode = link.getEndNode();
    if (endNode != null) {
      if (getId() == endNode.getId()) {
        fullValidation();
        return;
      }
    }
    GraphNode startNode = link.getStartNode();
    if (startNode != null) {
      if (getId() == startNode.getId()) {
        fullValidation();
      }
    }
  }

  /**
   * Configura a cadeida de objetos <i>vix</i>.
   */
  private void setUpVix() {
    changeVO(null, this.algorithmNameView);
    this.algorithmNameView.changeVS(null, this);
    changeVO(null, this.algorithmVersionIdView);
    this.algorithmVersionIdView.changeVS(null, this);

    for (final GraphFileDescriptor inputFileType : this.inputFileTypeViews) {
      changeVO(null, inputFileType);
      inputFileType.changeVS(null, this);
    }
    for (final GraphFileDescriptor outputFileType : this.outputFileTypeViews) {
      changeVO(null, outputFileType);
      outputFileType.changeVS(null, this);
    }

    if (this.icon != null) {
      changeVO(null, this.icon);
      this.icon.changeVS(null, this);
    }

    if (this.decoration != null) {
      VO image = this.decoration.getVO();
      if (image != null) {
        changeVO(null, image);
        image.changeVS(null, this);
      }
    }
  }

  /**
   * Verifica se o ponto informado est contido no retngulo formado pelos
   * pontos ({@code x1}, {@code y1}) e ({@code x2}, {@code y2}).
   *
   * @param x1 A abscissa esquerda.
   * @param y1 A ordenada superior.
   * @param x2 A abscissa direita.
   * @param y2 A ordenada inferior.
   *
   * @param pt O ponto procurado (No aceita {@code null}).
   *
   * @return {@code true} se estiver contido no retngulo ou {@code false} caso
   *         contrrio.
   */
  private boolean isContainedInRectangle(final double x1, final double y1,
    final double x2, final double y2, final Point2D pt) {
    final Rectangle2D bounds = new Rectangle2D.Double();
    bounds.setFrameFromDiagonal(x1, y1, x2, y2);
    return bounds.contains(pt);
  }

  /**
   * Atualiza a posio das vises de arquivo.
   */
  private void updateFileTypes() {
    updateInputFileTypes();
    updateOutputFileTypes();
  }

  /**
   * Atualiza a posio do cone principal.
   */
  private void updateIconLocation() {
    if (this.icon != null) {
      Dimension iconDimension = getIconDimension();
      double h = iconDimension.getHeight();
      double w = iconDimension.getWidth();
      final double origH = this.dimension.getHeight();
      final double deltaH = origH - h;

      final double x = this.northWestPoint.getX();
      final double y = this.northWestPoint.getY() + deltaH;
      final Rectangle2D.Double rect = new Rectangle2D.Double(x, y, w, h);
      this.icon.setRectangle(rect);
    }
  }

  /**
   * Atualiza o tamanho/posio das imagens no n.
   */
  private void updateImages() {
    updateIconLocation();
    updateDecorationLocation();
  }

  /**
   * Atualiza a posio do cone de decorao.
   */
  private void updateDecorationLocation() {
    if (this.decoration != null) {
      Dimension decorationDimension = getDecorationDimension();
      double h = decorationDimension.getHeight();
      double w = decorationDimension.getWidth();
      final double origH = this.dimension.getHeight();
      final double origW = this.dimension.getWidth();
      final double deltaH = origH - h;
      final double deltaW = origW - w;

      final double x = this.northWestPoint.getX() + deltaW;
      final double y = this.northWestPoint.getY() + deltaH;

      final Rectangle2D.Double rect = new Rectangle2D.Double(x, y, w, h);
      this.decoration.setRectangle(rect);
    }
  }

  /**
   * Atualiza a posio das vises de arquivo de entrada.
   */
  private void updateInputFileTypes() {
    if (this.inputFileTypeViews == null) {
      createInputFileTypes();
    }
    final Rectangle2D bounds = this.shape.getBounds2D();
    final double leftX = bounds.getMinX();
    final double rightX = bounds.getMaxX();
    final double upperY = bounds.getMinY();
    final double inputInterval =
      (rightX - leftX) / (this.inputFileTypeViews.size() + 1.0);
    double x = leftX;
    for (final GraphFileDescriptor inputFileType : this.inputFileTypeViews) {
      x += inputInterval;
      final Point2D pt = new Point();
      pt.setLocation(x, upperY);
      inputFileType.setLocation(pt);
    }
  }

  /**
   * Atualiza a posio das vises de arquivo de sada.
   */
  private void updateOutputFileTypes() {
    if (this.outputFileTypeViews == null) {
      createOutputFileTypes();
    }
    final Rectangle2D bounds = this.shape.getBounds2D();
    final double leftX = bounds.getMinX();
    final double rightX = bounds.getMaxX();
    final double lowerY = bounds.getMaxY();
    final double outputInterval =
      (rightX - leftX) / (this.outputFileTypeViews.size() + 1.0);
    double x = leftX;
    for (final GraphFileDescriptor outputFileType : this.outputFileTypeViews) {
      x += outputInterval;
      final Point2D pt = new Point();
      pt.setLocation(x, lowerY);
      outputFileType.setLocation(pt);
    }
  }

  /**
   * Atualiza a posio dos textos.
   */
  private void updateTextLocation() {
    final double centerX = getCenterPoint().getX();
    double y = this.northWestPoint.getY();
    if (isVersionInfoVisible()) {
      double nameHeight = this.algorithmNameView.getBounds2D().getHeight();
      double versionHeight =
        this.algorithmVersionIdView.getBounds2D().getHeight();
      // Espao entre o topo da caixa e o topo dos textos
      double inset =
        y + (this.dimension.getHeight() - nameHeight - versionHeight) / 2;
      double nameCenterY = inset + nameHeight / 2;
      this.algorithmNameView.setLocation(centerX, nameCenterY);
      double versionCenterY = inset + nameHeight + versionHeight / 2;
      this.algorithmVersionIdView.setLocation(centerX, versionCenterY);
    }
    else {
      this.algorithmNameView.setLocation(centerX, y
        + this.dimension.getHeight() / 2);
    }
  }

  /**
   * <p>
   * Valida os valores dos parmetros deste n.
   * </p>
   *
   * <p>
   * Ele pode exibir mensagens de com o motivo do erro de validao. Ele
   * atualiza a cor do n para estar de acordo com o estado (pronto para
   * executar ou no-pronto para executar).
   * </p>
   *
   * <p>
   * Este n precisa estar bem-formado.
   * </p>
   *
   * <p>
   * O configurador  invlido se os valores dos parmetros no forem adequados.
   * </p>
   *
   * @param mode Modo de validao ({@link ValidationMode#FULL} ou
   *        {@link ValidationMode#ALLOW_EMPY_VALUES}).
   *
   * @return O resultado da validao.
   */
  private ViewValidationResult validateAlgorithmConfigurator(ValidationMode mode) {
    final ViewValidationResult validation =
      ViewValidationTask.runTask(getGraph().getParentWindow(),
        configuratorView, mode, false);
    return validation;
  }

  /**
   * <p>
   * Valida os arquivos de entrada deste n.
   * </p>
   *
   * <p>
   * Um arquivo de entrada  considerado invlido se existir uma conexo
   * mal-formada partindo dele.
   * </p>
   *
   * @return O resultado da validao.
   */
  private ViewValidationResult validateInputFiles() {

    for (final GraphFileDescriptor descriptor : this.inputFileTypeViews) {
      if (!descriptor.validate()) {
        LocalizedMessage message =
          new LocalizedMessage(GraphNode.class,
            "error_invalid_input_file_descriptor", new Object[] { descriptor
              .toString() });
        return new ViewValidationResult(message, this);
      }
      if (descriptor.getFileParameter().isEnabled()) {
        final GraphLink link = descriptor.getLinkTo();
        if (link != null) {
          if (!link.isWellFormed()) {
            LocalizedMessage message =
              new LocalizedMessage(GraphNode.class, "error_invalid_link",
                new Object[] { link.toString() });
            return new ViewValidationResult(message, this);
          }
        }
      }
    }
    return new ViewValidationResult(this);
  }

  /**
   * <p>
   * Valida os arquivos de sada deste n.
   * </p>
   *
   * <p>
   * O arquivo de sada  considerado invlido se existir pelo menos uma conexo
   * mal-formada chegando nele.
   * </p>
   *
   * @return O resultado da validao.
   */
  private ViewValidationResult validateOutputFiles() {

    for (final GraphFileDescriptor descriptor : this.outputFileTypeViews) {
      if (!descriptor.validate()) {
        LocalizedMessage message =
          new LocalizedMessage(GraphNode.class,
            "error_invalid_output_file_descriptor", new Object[] { descriptor
              .toString() });
        return new ViewValidationResult(message, this);
      }
      if (descriptor.getFileParameter().isEnabled()) {
        for (final GraphLink link : descriptor.getLinkFromCollection()) {
          if (!link.isWellFormed()) {
            LocalizedMessage message =
              new LocalizedMessage(GraphNode.class, "error_invalid_link",
                new Object[] { link.toString() });
            return new ViewValidationResult(message, this);
          }
        }
      }
    }
    return new ViewValidationResult(this);
  }

  /**
   * Indica que esse n deve ser desviado ou no no fluxo. Caso positivo, no
   * entanto, ainda vai ser verificado se o desvio  possvel. Caso negativo, o
   * n deixa de ser desviado.
   *
   * @param bypassed indica se o n deve ser desviado ou no.
   * @return verdadeiro se foi possvel fazer/remover o desvio do algoritmo ou
   *         falso caso contrrio. No h porque impedir a remoo do desvio, no
   *         momento, portanto se o parmetro bypassed for falso, o mtodo deve
   *         sempre retornar verdadeiro.
   */
  public boolean setBypassed(final boolean bypassed) {
    validateBypass();
    if (!bypassed || canBypass) {
      this.bypassed = bypassed;
      ViewValidationResult result = validate(ValidationMode.FULL);
      highlightValidationResult(result);
      updateFileTypes();
      return true;
    }
    return false;
  }

  /**
   * Retorna verdadeiro se o n estiver marcado para ser desviado ou falso, caso
   * contrrio. O n pode estar marcado para ser desviado, mas no estar sendo
   * realmente desviado na execuo no momento, caso no esteja cumprindo as
   * condies necessrias.
   *
   * @return verdadeiro se o n est marcado para ser desviado ou falso, caso
   *         contrrio.
   */
  public boolean isBypassed() {
    return bypassed;
  }

  /**
   * Retorna verdadeiro se o n estiver cumprindo as condies necessrias para
   * que seja desviado ou falso, caso contrrio. O fato de retornar verdadeiro
   * no necessariamente indica que o n est sendo desviado no momento. O n s
   * ser desviado caso esteja marcado para o desvio (atributo bypassed
   * verdadeiro) e esteja cumprindo as condies (atributo canBypass
   * verdadeiro).
   *
   * @return verdadeiro se o n estiver em condio de ser desviado ou falso
   *         caso contrrio.
   */
  public boolean canBeBypassed() {
    return canBypass;
  }

  /**
   * Valida o n em relao s condies necessrias para que ele possa ser
   * desviado. Se o n cumprir todas as condies, o desvio pode ser feito (a
   * varivel canBypass recebe verdadeiro).
   *
   * Para que um n possa ser desviado as entradas do seu algoritmo tm que ser
   * compatveis com suas sadas. @see
   * {@link #isEquivalent(Collection, Collection)}
   *
   * O fato de cada n do fluxo poder ser desviado individualmente no indica
   * que o fluxo vai poder ser executado com os desvios. Existem algumas
   * condies de desvios que no permitem a execuo do fluxo. @see
   * FlowAlgorithmConfigurator
   */
  protected void validateBypass() {
    final Collection<GraphLink> linkFromCollection =
      this.getLinkFromCollection();
    final Collection<GraphLink> linkToCollection = this.getLinkToCollection();
    if (inputFileTypeViews.size() == outputFileTypeViews.size()) {
      if (linkFromCollection.size() > 0 && linkToCollection.size() > 0) {
        if (isEquivalent(linkFromCollection, linkToCollection)) {
          canBypass = true;
          return;
        }
      }
    }
    canBypass = false;
  }

  /**
   * Determina se as ligaes de entrada e sada de um n so equivalentes. O
   * nmero de entradas precisa ser igual ao de sadas e o nmero de entradas de
   * um determinado tipo tambm precisa ser o mesmo das sadas deste tipo.
   *
   * @param linkFromCollection ligaes de entrada do n
   * @param linkToCollection ligaes de sada do n
   * @return verdadeiro se as colees de ligaes de entrada e sada so
   *         verdadeiras ou falso caso contrrio.
   */
  protected boolean isEquivalent(
    final Collection<GraphLink> linkFromCollection,
    final Collection<GraphLink> linkToCollection) {
    final Map<String, Integer> linkToTypeCount =
      getLinkToTypeCount(linkToCollection);
    final Map<String, Integer> linkFromTypeCount =
      getLinkFromTypeCount(linkFromCollection);
    if (linkToTypeCount.keySet().size() == linkFromTypeCount.keySet().size()) {
      for (final String type : linkToTypeCount.keySet()) {
        if (!(linkToTypeCount.get(type).equals(linkFromTypeCount.get(type)))) {
          return false;
        }
      }
      return true;
    }
    return false;
  }

  /**
   * Agrupa as ligaes de sada de um n pelos seus tipos e retorna o nmero de
   * ligaes de cada tipo.
   *
   * Se mais de uma ligao estiver conectada a uma mesma sada ("split"), as
   * ligaes contaro como uma nica, ou seja, s ser contada uma ligao por
   * sada.
   *
   * As ligaes sem tipo especificado sero agrupadas sob a String
   * "NOT_SPECIFIED"
   *
   * @param linkFromCollection ligaes de sada do n
   * @return mapa com o nmero de ligaes de cada tipo
   */
  protected Map<String, Integer> getLinkFromTypeCount(
    final Collection<GraphLink> linkFromCollection) {
    final Map<String, Integer> types = new HashMap<String, Integer>();
    final List<GraphFileDescriptor> outputs =
      new ArrayList<GraphFileDescriptor>();
    for (final GraphLink linkFrom : linkFromCollection) {
      final GraphFileDescriptor inputFileDescriptor =
        linkFrom.getInputFileDescriptor();
      if (inputFileDescriptor != null) {
        String outputFileType = "NOT_SPECIFIED";
        final AbstractFileParameter fileParameter =
          inputFileDescriptor.getFileParameter();
        if (fileParameter != null && fileParameter.getFileType() != null) {
          outputFileType = fileParameter.getFileType();
        }

        final GraphFileDescriptor outputFileDescriptor =
          linkFrom.getOutputFileDescriptor();
        if (outputFileDescriptor != null) {
          //S conta um link por output
          if (!outputs.contains(outputFileDescriptor)) {
            outputs.add(outputFileDescriptor);
            if (types.get(outputFileType) == null) {
              types.put(outputFileType, 1);
            }
            else {
              Integer typeCount = types.get(outputFileType);
              types.put(outputFileType, typeCount++);
            }
          }
        }
      }
    }
    return types;
  }

  /**
   * Agrupa as ligaes de entrada de um n pelos seus tipos e retorna o nmero
   * de ligaes de cada tipo.
   *
   * As ligaes sem tipo especificado sero agrupadas sob a String
   * "NOT_SPECIFIED"
   *
   * @param linkToCollection ligaes de entrada do n
   * @return mapa com o nmero de ligaes de cada tipo
   */
  protected Map<String, Integer> getLinkToTypeCount(
    final Collection<GraphLink> linkToCollection) {
    final Map<String, Integer> types = new HashMap<String, Integer>();
    for (final GraphLink linkTo : linkToCollection) {
      final GraphFileDescriptor outputFileDescriptor =
        linkTo.getOutputFileDescriptor();
      if (outputFileDescriptor != null) {
        final AbstractFileParameter fileParameter =
          outputFileDescriptor.getFileParameter();
        String inpuFileType = "NOT_SPECIFIED";
        if (fileParameter != null && fileParameter.getFileType() != null) {
          inpuFileType = fileParameter.getFileType();
        }
        if (types.get(inpuFileType) == null) {
          types.put(inpuFileType, 1);
        }
        else {
          Integer typeCount = types.get(inpuFileType);
          types.put(inpuFileType, typeCount++);
        }
      }
    }
    return types;
  }

  /**
   * Adiciona a decorao do tipo imagem especificada a este n.
   *
   * @param type o tipo de decorao a ser adicionada.
   */
  public void addImageDecoration(final DecorationType type) {
    final GraphNodeDecoration<?> newDecoration =
      GraphNodeImageDecoration.createDecoration(type);
    replaceDecoration(newDecoration);
  }

  /**
   * Adiciona a decorao do tipo texto especificada a este n.
   *
   * @param text o texto.
   * @param description a descrio da decorao.
   */
  public void addTextDecoration(final String text, final String description) {
    replaceDecoration(new GraphNodeTextDecoration(getVS(), text, description));
  }

  /**
   * Remove a decorao utilizada atualmente por este n.
   */
  public void removeDecoration() {
    replaceDecoration(new GraphNodeImageDecoration());
  }

  /**
   * Substitui o cone da decorao atual por outro, da nova decorao escolhida
   * no <i>vix</i>.
   *
   * @param newDecoration cone da nova decorao.
   */
  private void replaceDecoration(final GraphNodeDecoration<?> newDecoration) {
    if (decoration != null) {
      final VO image = this.decoration.getVO();
      if (image != null) {
        image.changeVS(this, null);
        changeVO(image, null);
      }
    }
    this.decoration = newDecoration;
    if (decoration != null) {
      final VO image = decoration.getVO();
      if (image != null) {
        image.changeVS(null, this);
        changeVO(null, image);
      }
    }
    adjustTextPadding();
    updateImages();
    this.layoutChanged = true;
    repaint();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean highlightValidationResult(ViewValidationResult result) {
    if (isWellFormed()) {
      if (result.isWellSucceded()) {
        this.backgroundColor = READY_BACKGROUND_COLOR;
      }
      else {
        this.backgroundColor = NOT_READY_BACKGROUND_COLOR;
      }
    }
    else {
      this.backgroundColor = ERROR_COLOR;
    }
    if (isBypassed() && canBeBypassed()) {
      this.algorithmNameView.setColor(BYPASSED_TEXT_COLOR);
      this.algorithmVersionIdView.setColor(BYPASSED_TEXT_COLOR);
    }
    else {
      this.algorithmNameView.setColor(DEFAULT_FOREGROUNG_COLOR);
      this.algorithmVersionIdView.setColor(DEFAULT_FOREGROUNG_COLOR);
    }
    repaint();
    return true;
  }

  /**
   * Determina se a informao de verso do algoritmo deve ser mostrada no n.
   *
   * @param visible verdadeiro se a informao deve ser mostrada ou falso, caso
   *        contrrio.
   */
  public void setVersionInfoVisible(boolean visible) {
    this.algorithmVersionIdView.setVisible(visible);
    this.layoutChanged = true;
    repaint();
  }

  /**
   * Retorna
   *
   * @return isHidden
   */
  protected boolean isVersionInfoVisible() {
    return algorithmVersionIdView.isVisible();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void translate(Point2D pt) {
    Rectangle2D bounds2d = this.shape.getBounds2D();
    double newX = bounds2d.getX() + pt.getX();
    double newY = bounds2d.getY() + pt.getY();
    setBounds2D(newX, newY, bounds2d.getWidth(), bounds2d.getHeight());
  }
}
