package csbase.client.applications.projectsmanager.proxy;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.models.ProjectInfoData;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.models.ProjectsManagerScope;
import csbase.client.applications.projectsmanager.proxy.core.ProjectsManagerTask;
import csbase.exception.BugException;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectAdminInfo;
import csbase.logic.ProjectPermissions.SharingType;
import csbase.logic.User;
import csbase.logic.UserProjectInfo;
import csbase.remote.AdministrationServiceInterface;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.DiskUsageServiceInterface;
import csbase.remote.ProjectServiceInterface;

/**
 * Tarefa que preenche a tabela com os dados dos projetos existentes no
 * servidor.
 * 
 * @author jnlopes
 */
public class FillProjectsTableAdminTask extends
  ProjectsManagerTask<ProjectInfoData> {

  /**
   * Indica se espao ocupado pelos projetos deve ser carregado ou no.
   */
  private boolean loadOccupiedSpace;

  /**
   * Projetos que no puderam ser abertos pois o arquivo .csbase_project_info
   * no existe ou est corrompido.
   */
  private List<UserProjectInfo> failedProjects =
    new ArrayList<UserProjectInfo>();

  /**
   * Projetos que tem alocao de rea (ou aguardam desalocao).
   */
  private HashSet<Object> projectWithAllocationSet = new HashSet<Object>();

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   * @param loadOccupiedSpace boolean que indica se espao ocupado pelos
   *        projetos deve ser carregado ou no.
   */
  public FillProjectsTableAdminTask(ProjectsManager projectsManager,
    boolean loadOccupiedSpace) {
    super(projectsManager);
    this.loadOccupiedSpace = loadOccupiedSpace;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() throws Exception {
    List<ProjectsManagerData> projectsData = getProjectsData();
    ProjectInfoData pid = new ProjectInfoData(projectsData, failedProjects);
    setResult(pid);
  }

  /**
   * Mtodo que efetivamente obtm os dados dos projetos.
   * 
   * @return vetor de dados de projeto.
   * @throws Exception em caso de falha na execuo de mtodo remoto.
   */
  private List<ProjectsManagerData> getProjectsData() throws Exception {
    List<ProjectsManagerData> projectList =
      new ArrayList<ProjectsManagerData>();
    getProjectsWithAllocation(projectList);
    getProjectsWithoutAllocation(projectList);
    return projectList;
  }

  /**
   * Pega todos os projetos que no tem alocao.
   * 
   * @param projectList Lista de projetos a ser atualizada.
   * @throws RemoteException em caso de falha na execuo de mtodo remoto.
   * @throws Exception em caso de falha geral.
   */
  private void getProjectsWithoutAllocation(
    List<ProjectsManagerData> projectList) throws RemoteException, Exception {
    ProjectServiceInterface psi = ClientRemoteLocator.projectService;
    AdministrationServiceInterface asi =
      ClientRemoteLocator.administrationService;
    DiskUsageServiceInterface dus = ClientRemoteLocator.diskUsageService;
    /*
     * Agora pegamos os projetos que no tem nenhuma alocao.
     */
    List<User> users = asi.getAllUsers();
    for (User user : users) {
      List<UserProjectInfo> userProjectsInfo =
        psi.getProjectsFromUser(user.getId());
      for (UserProjectInfo upi : userProjectsInfo) {
        if (projectWithAllocationSet.contains(upi.getProjectId())) {
          continue;
        }
        /*
         * Se usuario corrente nao for dono do projeto corrente, ignora.
         */
        if (!upi.getOwnerId().equals(user.getId())) {
          continue;
        }
        CommonClientProject ccp;
        try {
          final ProjectAdminInfo projectAdminInfo =
            psi.getProjectAdminInfo(upi.getProjectId());
          if (projectAdminInfo != null
            && (projectAdminInfo.isLocked() || projectAdminInfo
              .isWaitingAreaFree())) {
            continue;
          }
          ccp = psi.openProject(upi.getProjectId(), false);
        }
        catch (BugException be) {
          /*
           * Esta exceo  lanada quando o arquivo .csbase_project_info do
           * projeto que estamos tentando abrir no existe ou est corrompido. A
           * execuo no deve parar por causa disto, apenas uma mensagem de
           * erro deve ser exibida.
           */
          failedProjects.add(upi);
          continue;
        }
        long allocatedSpace = ccp.getLockingAreaSize();
        ProjectSpaceAllocation psa =
          ProjectSpaceAllocation.getItemFromSpace(allocatedSpace);
        /*
         * Se o projeto nao tem alocao, ento ele no foi lido pelo loop
         * anterior.
         */
        if (psa == ProjectSpaceAllocation.NO_ALLOCATION) {
          ProjectsManagerData pmd =
            new ProjectsManagerData(ccp.getId(), ccp.getName(), ccp.getUserId());
          pmd.setOwnerName(User.getName(ccp.getUserId()));
          pmd.setProjectSpaceAllocation(ProjectSpaceAllocation.NO_ALLOCATION);
          pmd.setProjectDescription(ccp.getDescription());
          pmd.setCreationDate(ccp.getCreationDate());
          pmd.setServerOwnerName(ccp.getOwnerServerName());
          pmd.setLocationInServer(ccp.getLocationInServer());
          pmd.setModificationDate(ccp.getLastModificationDate());
          if (loadOccupiedSpace) {
            pmd.setOccupiedSpace(dus.getUsedSpaceForProjectMb(ccp.getUserId(),
              ccp.getName()));
          }
          else {
            pmd.setOccupiedSpace(-1);
          }
          pmd.setSharingType(ccp.getSharingType());
          pmd.setUsers(ccp.getUsersRO(), ccp.getUsersRW());
          ccp.close(false);
          projectList.add(pmd);
        }
        else if (psa != ProjectSpaceAllocation.ALLOCATED) {
          /*
           * Esta condio  atingida quando um projeto possui espao alocado no
           * entanto seu estado indica que ele aguarda alocao / desalocao de
           * rea. Esta situao pode indicar que o arquivo
           * locked_projects.csbase no est correto.
           */
          failedProjects.add(upi);
        }
      }
    }
  }

  /**
   * Pega todos os projetos que esto passando por algum estado de alocao
   * (esperando alocao, alocado, esperando desalocao).
   * 
   * @param projectList Lista de projetos a ser atualizada.
   * @throws RemoteException em caso de falha na execuo de mtodo remoto.
   * @throws Exception em caso de falha geral.
   */
  private void getProjectsWithAllocation(List<ProjectsManagerData> projectList)
    throws RemoteException, Exception {
    ProjectServiceInterface psi = ClientRemoteLocator.projectService;
    DiskUsageServiceInterface dus = ClientRemoteLocator.diskUsageService;
    ProjectAdminInfo[] allInfo = ProjectAdminInfo.getAll();
    if (allInfo == null) {
      return;
    }
    for (ProjectAdminInfo prjAdminInfo : allInfo) {
      ProjectsManagerData pmd =
        new ProjectsManagerData(prjAdminInfo.getProjectId(),
          prjAdminInfo.getProjectName(), prjAdminInfo.getOwnerId());
      /*
       * inicializando com valores vazios.
       */
      pmd.setProjectDescription(null);
      pmd.setCreationDate(0);
      pmd.setServerOwnerName("VAZIO1");
      pmd.setLocationInServer(prjAdminInfo.getAbsolutePath());
      pmd.setModificationDate(0);
      pmd.setOccupiedSpace(0);
      pmd.setOwnerName(User.getName(prjAdminInfo.getOwnerId()));
      if (prjAdminInfo.isUnlockedWithAreaAllocated()) {
        pmd.setProjectSpaceAllocation(ProjectSpaceAllocation.ALLOCATED);
        long allocatedSpace = prjAdminInfo.getAreaLockedSize();
        pmd.setAllocatedSpace(allocatedSpace);
        CommonClientProject ccp;
        try {
          ccp = psi.openProject(prjAdminInfo.getProjectId(), false);
          if (ccp != null) {
            pmd.setProjectDescription(ccp.getDescription());
            pmd.setCreationDate(ccp.getCreationDate());
            pmd.setModificationDate(ccp.getLastModificationDate());
            pmd.setLocationInServer(ccp.getLocationInServer());
            if (loadOccupiedSpace) {
              pmd.setOccupiedSpace(dus.getUsedSpaceForProjectMb(
                ccp.getUserId(), ccp.getName()));
            }
            else {
              pmd.setOccupiedSpace(-1);
            }
            /*
             * FIXME - try adicionado para que excees lanadas por tentativa
             * de abrir projetos no formato de permisses antigo no impeam a
             * abertura do gerenciador de projetos.
             */
            try {
              pmd.setSharingType(ccp.getSharingType());
            }
            catch (IllegalStateException ise) {
              ccp.close(false);
              continue;
            }
            pmd.setUsers(ccp.getUsersRO(), ccp.getUsersRW());
            ccp.close(false);
          }
        }
        catch (BugException be) {
          /*
           * Esta exceo  lanada quando o arquivo .csbase_project_info do
           * projeto que estamos tentando abrir no existe ou est corrompido. A
           * execuo no deve parar por causa disto, apenas uma mensagem de
           * erro deve ser exibida.
           */
          failedProjects.add(prjAdminInfo);
          continue;
        }
      }
      else if (prjAdminInfo.isLocked()) {
        pmd.setRequestedSpace(prjAdminInfo.getAreaRequestSize());
        pmd
          .setProjectSpaceAllocation(ProjectSpaceAllocation.WAITING_ALLOCATION);
        pmd.setScope(ProjectsManagerScope.NOT_APPLICABLE);
        /*
         * assumimos o escopo default "privado"
         */
        pmd.setSharingType(SharingType.PRIVATE);
        pmd.setUsers(new HashSet<Object>(), new HashSet<Object>());
        pmd.setCreationDate(prjAdminInfo.getAreaRequestDate().getTime());
      }
      else if (prjAdminInfo.isWaitingAreaFree()) {
        pmd
          .setProjectSpaceAllocation(ProjectSpaceAllocation.WAITING_DEALLOCATION);
        pmd.setScope(ProjectsManagerScope.NOT_APPLICABLE);
        /*
         * assumimos o escopo default "privado"
         */
        pmd.setSharingType(SharingType.PRIVATE);
      }
      /*
       * Projetos sem alocao (ou solicitao de alocao) no so mantidos na
       * lista de ProjectAdminInfos.
       */
      projectWithAllocationSet.add(pmd.getProjectId());
      projectList.add(pmd);
    }
  }
}
