/*
 * Copyright (c) 2006-2007 Timothy Wall, All Rights Reserved
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. <p/> This library is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 */
package csbase.client.util;

import java.awt.Component;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.geom.AffineTransform;
import java.awt.image.ImageObserver;
import java.lang.ref.WeakReference;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.swing.CellRendererPane;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.SwingUtilities;

/**
 * cone que  animado mesmo quando  usado dentro de um
 * {@link CellRendererPane}. Tambm pode ser usado como base para animaes
 * feitas em java.
 * 
 * Retirada do projeto: http://furbelow.sourceforge.net/
 * 
 * @author twall
 */
public class AnimatedIcon implements Icon {

  /**
   * O cone que deve ser animado.
   */
  private ImageIcon original;

  /**
   * Conjunto de reas que devem ser redesenhadas.
   */
  private final Set<RepaintArea> repaints = new HashSet<RepaintArea>();

  /**
   * Para uso de classes derivadas que no possuem uma imagem original
   * (animaes feitas em java).
   */
  protected AnimatedIcon() {
  }

  /**
   * Cria um cone que  animado mesmo em componentes que usam um
   * CellRendererPane.
   * 
   * @param original Imagem a ser animada.
   */
  public AnimatedIcon(final ImageIcon original) {
    if (original == null) {
      throw new IllegalArgumentException("O parmetro original est nulo");
    }
    this.original = original;
    new AnimationObserver(this, original);
  }

  /**
   * Redesenha todas as reas requisitadas.
   */
  protected synchronized void repaint() {
    for (final Iterator<RepaintArea> i = repaints.iterator(); i.hasNext();) {
      i.next().repaint();
    }
    repaints.clear();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getIconHeight() {
    return original.getIconHeight();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getIconWidth() {
    return original.getIconWidth();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void paintIcon(final Component c, final Graphics g,
    final int x, final int y) {
    original.paintIcon(c, g, x, y);
    if (c != null) {
      int w = getIconWidth();
      int h = getIconHeight();
      final AffineTransform tx = ((Graphics2D) g).getTransform();
      w = (int) (w * tx.getScaleX());
      h = (int) (h * tx.getScaleY());
      repaints.add(new RepaintArea(c, x, y, w, h));
    }
  }

  /**
   * Representa uma rea de um componente que deve ser redesenhada.
   */
  private class RepaintArea {

    /**
     * Abscissa da rea a ser redesenhada.
     */
    public int x;

    /**
     * Ordenada da rea a ser redesenhada.
     */
    public int y;

    /**
     * Largura da rea a ser redesenhada.
     */
    public int w;

    /**
     * Altura da rea a ser redesenhada.
     */
    public int h;

    /**
     * Componente a ser redesenhado.
     */
    public Component component;

    /**
     * Cdigo de hash.
     */
    private final int hashCode;

    /**
     * Construtor.
     * 
     * @param c Componente a ser redesenhado.
     * @param x Abscissa da rea a ser redesenhada.
     * @param y Ordenada da rea a ser redesenhada.
     * @param w Largura da rea a ser redesenhada.
     * @param h Altura da rea a ser redesenhada.
     */
    public RepaintArea(final Component c, final int x, final int y,
      final int w, final int h) {
      Component nc = c;
      int nx = x;
      int ny = y;
      final Component ancestor = findNonRendererAncestor(nc);
      if (ancestor != nc) {
        final Point pt = SwingUtilities.convertPoint(nc, nx, ny, ancestor);
        nc = ancestor;
        nx = pt.x;
        ny = pt.y;
      }
      this.component = nc;
      this.x = nx;
      this.y = ny;
      this.w = w;
      this.h = h;
      final String hash = String.valueOf(nx) + "," + ny + ":" + nc.hashCode();
      this.hashCode = hash.hashCode();
    }

    /**
     * Encontra o primeiro ancestral do componente que <em>no</em> 
     * descendente de {@link CellRendererPane}.
     * 
     * @param c O componente.
     * @return O primeiro ancetral que no  descendente de
     *         {@link CellRendererPane}.
     */
    private Component findNonRendererAncestor(final Component c) {
      Component nc = c;
      final Component ancestor =
        SwingUtilities.getAncestorOfClass(CellRendererPane.class, nc);
      if (ancestor != null && ancestor != nc && ancestor.getParent() != null) {
        nc = findNonRendererAncestor(ancestor.getParent());
      }
      return nc;
    }

    /** Registra um pedido para redesenhar o componente. */
    public void repaint() {
      component.repaint(x, y, w, h);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean equals(final Object o) {
      if (o instanceof RepaintArea) {
        final RepaintArea area = (RepaintArea) o;
        return area.component == component && area.x == x && area.y == y
          && area.w == w && area.h == h;
      }
      return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int hashCode() {
      return hashCode;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
      return "Repaint(" + component.getClass().getName() + "@" + x + "," + y
        + " " + w + "x" + h + ")";
    }
  }

  /**
   * Detecta mudanas no estado da animao da imagem original e se remove, caso
   * essa imagem seja coletada pelo Garbage Collector.
   * 
   * @author twall
   */
  private static class AnimationObserver implements ImageObserver {

    /**
     * Referncia  animao.
     */
    private final WeakReference<AnimatedIcon> ref;

    /**
     * Imagem original.
     */
    private final ImageIcon original;

    /**
     * Construtor.
     * 
     * @param animIcon A animao.
     * @param original A imagem original.
     */
    public AnimationObserver(final AnimatedIcon animIcon,
      final ImageIcon original) {
      this.original = original;
      this.original.setImageObserver(this);
      ref = new WeakReference<AnimatedIcon>(animIcon);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean imageUpdate(final Image img, final int flags, final int x,
      final int y, final int width, final int height) {
      if ((flags & (FRAMEBITS | ALLBITS)) != 0) {
        final AnimatedIcon animIcon = ref.get();
        if (animIcon != null) {
          animIcon.repaint();
        }
        else {
          original.setImageObserver(null);
        }
      }
      return (flags & (ALLBITS | ABORT)) == 0;
    }
  }
}
