package csbase.client.util.xmlpanel;

import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;

import csbase.logic.ClientFile;

/**
 * Utilitrios
 * 
 * @author Tecgraf/PUC-Rio
 */
public class XMLPanelUtils {

  /**
   * Busca lista de ns filhos de um tipo do tipo {Node#ELEMENT_NODE}.
   * 
   * @param parentNode o n de busca (pai)
   * @return a lista
   */
  public static List<Node> getAttributeChildren(Node parentNode) {
    final List<Node> children = new ArrayList<Node>();
    if (parentNode == null) {
      return children;
    }

    final NamedNodeMap attributes = parentNode.getAttributes();
    if (attributes == null) {
      return children;
    }
    final int size = attributes.getLength();
    for (int i = 0; i < size; i++) {
      final Node item = attributes.item(i);
      children.add(item);
    }
    return children;
  }

  /**
   * Consulta valor de atributo de primeiro filho de n.
   * 
   * @param parentNode n
   * @param childName nome do tag do filho
   * @param attrName nome do atributo
   * @return texto
   */
  public static String getFirstChildAttributeValue(Node parentNode,
    String childName, String attrName) {
    Node firstChild = getFirstChild(parentNode, childName);
    if (firstChild == null) {
      return null;
    }
    return getAttributeValue(firstChild, attrName);
  }

  /**
   * Busca lista de ns filhos de um tipo do tipo {Node#ELEMENT_NODE}.
   * 
   * @param node o n de busca
   * @param attrName nome do atributo
   * @return a lista
   */
  public static String getAttributeValue(Node node, String attrName) {
    if (node == null) {
      return null;
    }
    final NamedNodeMap attributes = node.getAttributes();
    if (attributes == null) {
      return null;
    }
    final int size = attributes.getLength();
    for (int i = 0; i < size; i++) {
      final Node item = attributes.item(i);
      if (item.getNodeName().equals(attrName)) {
        return item.getNodeValue();
      }
    }
    return null;
  }

  /**
   * Indica de n tem um atributo com determinado valor textual.
   * 
   * @param node n
   * @param attrName nome do atributo
   * @param valueText texto buscado.
   * @return indicativo
   */
  public static boolean hasAttributeValue(Node node, String attrName,
    String valueText) {
    final String attributeValue = getAttributeValue(node, attrName);
    if (attributeValue == null) {
      return false;
    }
    return attributeValue.equals(valueText.trim());
  }

  /**
   * Busca primeiro filho com base em um nome
   * 
   * @param parentNode n
   * @param childName nome a ser busacado.
   * @return n filho (ou {@code null}).
   */
  public static Node getFirstChild(Node parentNode, String childName) {
    List<Node> children = getElementChildren(parentNode);
    for (Node child : children) {
      if (child.getNodeName().equals(childName)) {
        return child;
      }
    }
    return null;
  }

  /**
   * Busca lista de ns filhos de um tipo do tipo {Node#ELEMENT_NODE}.
   * 
   * @param parentNode o n de busca (pai)
   * @return a lista
   */
  public static List<Node> getElementChildren(Node parentNode) {
    List<Node> children = new ArrayList<Node>();
    if (parentNode == null) {
      return children;
    }
    final NodeList childNodes = parentNode.getChildNodes();
    for (int i = 0; i < childNodes.getLength(); i++) {
      final Node item = childNodes.item(i);
      if (item.getNodeType() == Node.ELEMENT_NODE) {
        children.add(item);
      }
    }
    return children;
  }

  /**
   * Criao de documento DOM com base em entrada.
   * 
   * @param clientFile arquivo
   * @param resolver resolver.
   * @param charset charset
   * @return documento DOM
   * @throws Exception em caso de erro.
   */
  public static Node getRootNode(ClientFile clientFile,
    EntityResolver resolver, Charset charset) throws Exception {
    if (clientFile == null) {
      return null;
    }
    final InputStream inputStream = clientFile.getInputStream();
    final Node document = getRootNode(inputStream, resolver, charset);
    return document;
  }

  /**
   * Criao de documento DOM com base em entrada.
   * 
   * @param stream entrada
   * @param er resolver.
   * @param charset charset
   * @return documento DOM
   * @throws Exception em caso de erro.
   */
  public static Node getRootNode(InputStream stream, EntityResolver er,
    Charset charset) throws Exception {
    final InputSource is = new InputSource(stream);
    if (charset != null) {
      is.setEncoding(charset.name());
    }
    DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
    if (er == null) {
      final String addr = "http://apache.org/";
      String dtd = addr + "xml/features/nonvalidating/load-dtd-grammar";
      factory.setAttribute(dtd, false);
      dtd = addr + "xml/features/nonvalidating/load-external-dtd";
      factory.setAttribute(dtd, false);
    }

    DocumentBuilder builder = factory.newDocumentBuilder();
    builder.setEntityResolver(er);
    final Document document = builder.parse(is);
    final Element rootNode = document.getDocumentElement();
    return rootNode;
  }
}
