package csbase.client.applications.projectsmanager.panels;

import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JPanel;

import tecgraf.javautils.gui.BorderUtil;
import tecgraf.javautils.gui.GBC;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.panels.filters.AbstractProjectFilter;
import csbase.client.applications.projectsmanager.panels.filters.ProjectBasicInfoFilterPanel;
import csbase.client.applications.projectsmanager.panels.filters.ProjectCreationDateFilterPanel;
import csbase.client.applications.projectsmanager.panels.filters.ProjectFileSystemLocationPanel;
import csbase.client.applications.projectsmanager.panels.filters.ProjectModificationDateFilterPanel;
import csbase.client.applications.projectsmanager.panels.filters.ProjectSpaceFilterPanel;

/**
 * Painel que contem os componentes de filtro da tabela de projetos.
 * 
 * @author jnlopes
 * 
 */
public class FilterPanel extends ProjectsManagerPanel {

  /**
   * Painel de filtragem por informaes elementares (nome do projeto / dono do
   * projeto).
   */
  final private ProjectBasicInfoFilterPanel projectBasicInfoFilterPanel;

  /**
   * Painel de filtragem por intervalo de data de criao de projeto.
   */
  final private ProjectCreationDateFilterPanel projectCreationDateFilterPanel;

  /**
   * Painel de filtragem por intervalo de data de modificao de projeto.
   */
  final private ProjectModificationDateFilterPanel projectModificationDateFilterPanel;

  /**
   * Painel de filtragem por intervalo de espao alocado e/ou ocupado.
   */
  final private ProjectSpaceFilterPanel projectSpaceFilterPanel;

  /**
   * Painel de filtragem por filesystem do projeto.
   */
  final private ProjectFileSystemLocationPanel projectFileSystemLocationPanel;

  /**
   * KeyListener genrico, que ser aplicado a todos os filtros.
   */
  final private KeyListener generalKeyListener;

  /**
   * ItemListener genrico, que ser aplicado a todos os filtros.
   */
  final private ItemListener generalItemListener;

  /**
   * Inserts tradicionais.
   */
  final private Insets insets = new Insets(3, 8, 3, 8);

  /**
   * Insets para separao vertical.
   */
  final private Insets verticalInsets = new Insets(3, 0, 3, 0);

  /**
   * Insets para maior espaamento a esquerda.
   */
  final private Insets largeLeftInsets = new Insets(3, 25, 3, 8);

  /**
   * Lista de todos os painis que implementam filtros.
   */
  private final List<AbstractProjectFilter> filters;

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public FilterPanel(final ProjectsManager projectsManager) {
    super(projectsManager);

    filters = new ArrayList<AbstractProjectFilter>();
    projectBasicInfoFilterPanel =
      new ProjectBasicInfoFilterPanel(projectsManager);
    projectCreationDateFilterPanel =
      new ProjectCreationDateFilterPanel(projectsManager);
    projectModificationDateFilterPanel =
      new ProjectModificationDateFilterPanel(projectsManager);
    projectSpaceFilterPanel = new ProjectSpaceFilterPanel(projectsManager);
    projectFileSystemLocationPanel =
      new ProjectFileSystemLocationPanel(projectsManager);

    filters.add(projectBasicInfoFilterPanel);
    filters.add(projectCreationDateFilterPanel);
    filters.add(projectModificationDateFilterPanel);
    filters.add(projectSpaceFilterPanel);
    filters.add(projectFileSystemLocationPanel);

    generalKeyListener = createGeneralKeyListener();
    generalItemListener = createGeneralItemListener();
    applyGeneralListener();

    buildPanel();
  }

  /**
   * Monta o painel principal.
   */
  private void buildPanel() {

    final ProjectsManager projectsManager = getProjectsManager();
    final JPanel panel;
    if (projectsManager.hasAreaReserved()) {
      panel = buildPanelWithAllocation();
    }
    else {
      panel = buildPanelWithoutAllocation();
    }

    setLayout(new GridBagLayout());
    add(panel, new GBC(0, 0).both());
    BorderUtil.setTitledBorder(this,
      getString("FilterPanel.project.filter.title"));
    validate();
    repaint();
  }

  /**
   * Monta o painel com campos referentes a alocao de espao.
   * 
   * @return Painel com alocao de espao.
   */
  private JPanel buildPanelWithAllocation() {

    final JPanel panel = new JPanel(new GridBagLayout());

    panel.add(projectBasicInfoFilterPanel.getProjectNameLabel(), new GBC(0, 0)
      .west().insets(verticalInsets));
    panel.add(projectBasicInfoFilterPanel.getProjectNameField(), new GBC(1, 0)
      .width(2).horizontal().insets(insets));
    panel.add(projectBasicInfoFilterPanel.getProjectOwnerLabel(), new GBC(3, 0)
      .west().insets(largeLeftInsets));
    panel.add(projectBasicInfoFilterPanel.getProjectOwnerPanel(), new GBC(4, 0)
      .width(2).horizontal().insets(verticalInsets));

    panel.add(projectFileSystemLocationPanel.getLabelPanel(), new GBC(0, 1)
      .width(1).west().insets(verticalInsets));
    panel.add(projectFileSystemLocationPanel.getTextPanel(), new GBC(1, 1)
      .width(2).insets(insets).horizontal());

    panel.add(projectCreationDateFilterPanel.getMainLabel(), new GBC(0, 2)
      .west().insets(verticalInsets));
    panel.add(projectCreationDateFilterPanel.getFromPanel(), new GBC(1, 2)
      .horizontal().insets(insets));
    panel.add(projectCreationDateFilterPanel.getUntilPanel(), new GBC(2, 2)
      .horizontal().insets(insets));

    panel.add(projectModificationDateFilterPanel.getMainLabel(), new GBC(3, 2)
      .west().insets(largeLeftInsets));
    panel.add(projectModificationDateFilterPanel.getFromPanel(), new GBC(4, 2)
      .horizontal().insets(insets));
    panel.add(projectModificationDateFilterPanel.getUntilPanel(), new GBC(5, 2)
      .horizontal().insets(insets));

    panel.add(projectSpaceFilterPanel.getAllocatedSpaceLabel(), new GBC(0, 3)
      .west().insets(verticalInsets));
    panel.add(projectSpaceFilterPanel.getAllocatedFromSpacePanel(), new GBC(1,
      3).horizontal().insets(insets));
    panel.add(projectSpaceFilterPanel.getAllocatedUntilSpacePanel(), new GBC(2,
      3).horizontal().insets(insets));

    panel.add(projectSpaceFilterPanel.getOccupiedSpaceLabel(), new GBC(3, 3)
      .west().insets(largeLeftInsets));
    panel.add(projectSpaceFilterPanel.getOccupiedFromSpacePanel(),
      new GBC(4, 3).horizontal().insets(insets));
    panel.add(projectSpaceFilterPanel.getOccupiedUntilSpacePanel(), new GBC(5,
      3).horizontal().insets(insets));

    panel.add(projectBasicInfoFilterPanel.getProjectStatusLabel(),
      new GBC(0, 4).west().insets(verticalInsets));
    panel.add(projectBasicInfoFilterPanel.getStatusComboPanel(), new GBC(1, 4)
      .width(2).horizontal().insets(verticalInsets));

    panel.add(projectBasicInfoFilterPanel.getProjectSharingLabel(), new GBC(3,
      4).west().insets(largeLeftInsets));
    panel.add(projectBasicInfoFilterPanel.getSharingComboPanel(), new GBC(4, 4)
      .width(2).horizontal().insets(verticalInsets));

    return panel;
  }

  /**
   * Monta o painel sem campos referentes a alocao de espao.
   * 
   * @return Painel sem alocao de espao.
   */
  private JPanel buildPanelWithoutAllocation() {

    final JPanel panel = new JPanel(new GridBagLayout());

    panel.add(projectBasicInfoFilterPanel.getProjectNameLabel(), new GBC(0, 0)
      .west().insets(verticalInsets));
    panel.add(projectBasicInfoFilterPanel.getProjectNameField(), new GBC(1, 0)
      .width(2).horizontal().insets(insets));
    panel.add(projectBasicInfoFilterPanel.getProjectOwnerLabel(), new GBC(3, 0)
      .west().insets(largeLeftInsets));
    panel.add(projectBasicInfoFilterPanel.getProjectOwnerPanel(), new GBC(4, 0)
      .width(2).horizontal().insets(verticalInsets));

    panel.add(projectCreationDateFilterPanel.getMainLabel(), new GBC(0, 1)
      .west().insets(verticalInsets));
    panel.add(projectCreationDateFilterPanel.getFromPanel(), new GBC(1, 1)
      .horizontal().insets(insets));
    panel.add(projectCreationDateFilterPanel.getUntilPanel(), new GBC(2, 1)
      .horizontal().insets(insets));

    panel.add(projectModificationDateFilterPanel.getMainLabel(), new GBC(3, 1)
      .west().insets(largeLeftInsets));
    panel.add(projectModificationDateFilterPanel.getFromPanel(), new GBC(4, 1)
      .horizontal().insets(insets));
    panel.add(projectModificationDateFilterPanel.getUntilPanel(), new GBC(5, 1)
      .horizontal().insets(insets));

    panel.add(projectSpaceFilterPanel.getOccupiedSpaceLabel(), new GBC(0, 2)
      .west().insets(verticalInsets));
    panel.add(projectSpaceFilterPanel.getOccupiedFromSpacePanel(),
      new GBC(1, 2).horizontal().insets(insets));
    panel.add(projectSpaceFilterPanel.getOccupiedUntilSpacePanel(), new GBC(2,
      2).horizontal().insets(insets));

    panel.add(projectBasicInfoFilterPanel.getProjectSharingLabel(), new GBC(3,
      2).west().insets(largeLeftInsets));
    panel.add(projectBasicInfoFilterPanel.getSharingComboPanel(), new GBC(4, 2)
      .width(2).horizontal().insets(verticalInsets));

    return panel;
  }

  /**
   * Executa filtros dos diversos paineis.
   * 
   * @param clearSelection boolean que indica se a seleo de linhas corrente
   *        deve ser desfeita.
   * @param refresh flag que indica se a tabela deve ser atualizada
   */
  private void runFilters(final boolean clearSelection, final boolean refresh) {
    final ProjectsManager projectsManager = getProjectsManager();
    final List<ProjectsManagerData> allProjects =
      projectsManager.getAllProjects();
    List<ProjectsManagerData> projects =
      new ArrayList<ProjectsManagerData>(allProjects);

    for (AbstractProjectFilter filter : filters) {
      projects = filter.runFilter(projects, false);
    }

    if (refresh) {
      projectsManager.refreshProjectsTable();
    }

    if (clearSelection) {
      projectsManager.clearProjectSelection();
    }

    projectsManager.clearInfoFields();
  }

  /**
   * Aplica todos os filtros e atualiza a tabela.
   * 
   * @param clearSelection boolean que indica se a seleo de linhas corrente
   *        deve ser desfeita.
   */
  public void runFilters(final boolean clearSelection) {
    runFilters(clearSelection, true);
  }

  /**
   * Verifica se um processo atende aos critrios de todos os filtros
   * configurados pelo usurio.
   * 
   * @param prj projeto
   * @return <code>true</code> se o projeto atende aos critrios de <b>todos</b>
   *         os filtros
   */
  public boolean projectMatchesFilters(final ProjectsManagerData prj) {
    for (AbstractProjectFilter filter : filters) {
      if (!filter.projectMatchesFilter(prj)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Cria o KeyListener geral para os diversos paineis para filtragem do
   * conteudo da tabela.
   * 
   * @return O KeyListener criado.
   */
  private KeyListener createGeneralKeyListener() {
    final FilterPanel fp = this;
    return new KeyListener() {
      @Override
      public void keyPressed(KeyEvent e) {
      }

      @Override
      public void keyReleased(KeyEvent e) {
        fp.runFilters(true);
      }

      @Override
      public void keyTyped(KeyEvent e) {
      }
    };
  }

  /**
   * Cria ItemListener para os diversos paineis para filtragem do conteudo da
   * tabela.
   * 
   * @return O ItemListener criado.
   */
  private ItemListener createGeneralItemListener() {

    final FilterPanel fp = this;

    return new ItemListener() {

      @Override
      public void itemStateChanged(ItemEvent e) {
        fp.runFilters(true);
      }

    };
  }

  /**
   * Associa o listener padrao aos paineis de filtragem.
   */
  private void applyGeneralListener() {
    projectBasicInfoFilterPanel.addListener(generalKeyListener);
    projectCreationDateFilterPanel.addListener(generalKeyListener);
    projectModificationDateFilterPanel.addListener(generalKeyListener);
    projectSpaceFilterPanel.addListener(generalKeyListener);
    projectFileSystemLocationPanel.addListener(generalKeyListener);
    projectBasicInfoFilterPanel.addItemListener(generalItemListener);
  }

  /**
   * Remove qualquer informao presente nos campos dos filtros.
   */
  public void emptyFiltersData() {
    projectBasicInfoFilterPanel.emptyFilterData();
    projectCreationDateFilterPanel.emptyFilterData();
    projectModificationDateFilterPanel.emptyFilterData();
    projectFileSystemLocationPanel.emptyFilterData();
    projectSpaceFilterPanel.emptyFilterData();
  }

}
