package csbase.client.util.table;

/**
 * Validador de Nmeros Reais.
 */
public final class DoubleValidator implements Validator {

  /**
   * Inclui o valor mnimo.
   */
  private boolean includeMinimum;

  /**
   * Inclui o valor mximo.
   */
  private boolean includeMaximum;

  /**
   * O mximo.
   */
  private Double maximum;

  /**
   * O mnimo.
   */
  private Double minimum;

  /**
   * Cria um validador de nmeros reais.
   * 
   * @param minimum O valor mnimo (Aceita {@code null}).
   * @param includeMinimum {@code true} Indica se o valor mnimo  considerado
   *        vlido.
   * @param maximum O valor mximo (Aceita {@code null}).
   * @param includeMaximum {@code true} Indica se o valor mximo  considerado
   *        vlido.
   */
  public DoubleValidator(Double minimum, boolean includeMinimum,
    Double maximum, boolean includeMaximum) {
    if (maximum == null) {
      maximum = Double.MAX_VALUE;
    }
    if (minimum == null) {
      minimum = Double.MIN_VALUE;
    }
    if (maximum.compareTo(minimum) < 0) {
      throw new IllegalArgumentException(String.format(
        "O mximo %f  menor do que o mnimo %f.", maximum, minimum));
    }
    if (!includeMaximum && !includeMinimum && maximum.equals(minimum)) {
      throw new IllegalArgumentException(String.format(
        "O mximo no pode ser igual ao mnimo (%f), "
          + "pois  o valor mximo e/ou mnimo no fazem parte"
          + " do intervalo vlido.", minimum));
    }
    this.minimum = minimum;
    this.includeMinimum = includeMinimum;
    this.maximum = maximum;
    this.includeMaximum = includeMaximum;
  }

  /**
   * Cria um validador de nmeros reais.
   * 
   * @param minimum O valor mnimo (Aceita {@code null}).
   * @param maximum O valor mximo (Aceita {@code null}).
   */
  public DoubleValidator(Double minimum, Double maximum) {
    this(minimum, true, maximum, true);
  }

  public String getValidationErrorMessage(Object value) {
    Double doubleValue = (Double) value;

    if (doubleValue == null) {
      return null;
    }

    if (includeMinimum) {
      if (doubleValue.compareTo(minimum) < 0) {
        return String.format("Valor abaixo do mnimo (%f).", minimum);
      }
    }
    else {
      if (doubleValue.compareTo(minimum) <= 0) {
        return String.format(
          "O valor informado deveria ser menor do que %f.", minimum);
      }
    }

    if (includeMaximum) {
      if (doubleValue.compareTo(maximum) > 0) {
        return String.format("Valor acima do mximo (%f).", maximum);
      }
    }
    else {
      if (doubleValue.compareTo(maximum) >= 0) {
        return String.format(
          "O valor informado deveria ser menor do que %f.", maximum);
      }
    }

    return null;
  }
}
