/*
 * $Id$
 */
package csbase.client.applicationmanager;

import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Hashtable;

import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.SwingConstants;
import javax.swing.border.Border;
import javax.swing.border.CompoundBorder;

import csbase.client.util.StandardErrorDialogs;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * A classe <code>ApplicationLink</code> implementa o componente da interface do
 * cliente utilizado para chamar uma aplicao.  uma classe abstrata. Suas
 * especializaes representam os links paras as diferentes aplicaes.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ApplicationLink extends JLabel {

  /**
   * Cria uma borda (com base em cor) ajustada para o link.
   * 
   * @param color cor
   * @return borda
   */
  static private Border createBorder(final Color color) {
    final int i = 2;
    final int e = 4;

    final Border external = BorderFactory.createEmptyBorder(e, e, e, e);
    final Border internal = BorderFactory.createEmptyBorder(i, i, i, i);
    final Border line = BorderFactory.createLineBorder(color);

    final CompoundBorder brd =
      BorderFactory.createCompoundBorder(external, BorderFactory
        .createCompoundBorder(line, internal));
    return brd;
  }

  /**
   * Borda para o link
   */
  private static final Border FOCUS_BORDER = createBorder(Color.GRAY);

  /**
   * Borda para o link
   */
  private static final Border NO_BORDER = createBorder(new JLabel()
    .getBackground());

  /**
   * Listener do mouse para o objeto.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private static class ApplicationLinkMouseListener extends MouseAdapter {

    /**
     * Componente que est sendo arrastado (cone)
     */
    final JComponent draggedComponent;

    /**
     * Referncia inicial para translao do cone
     */
    Point origin = null;

    /**
     * Evento de click do boto que ajusta a origem.
     * 
     * @param ev o evento.
     * 
     * @see #origin
     */
    @Override
    public void mousePressed(MouseEvent ev) {
      this.origin = ev.getPoint();
      this.draggedComponent.setBorder(FOCUS_BORDER);
    }

    /**
     * Evento de unclick que anula a origem.
     * 
     * @param ev o evento.
     * 
     * @see #origin
     */
    @Override
    public void mouseReleased(MouseEvent ev) {
      this.origin = null;
      this.draggedComponent.setBorder(NO_BORDER);
    }

    /**
     * Evento de drag com que produz a translao.
     * 
     * @param ev o evento.
     * 
     * @see #origin
     * @see #draggedComponent
     */
    @Override
    public void mouseDragged(MouseEvent ev) {
      if (this.origin == null) {
        return;
      }

      // Localizao do link e sua tranalao.
      final Point p = this.draggedComponent.getLocation();
      final double dx = ev.getX() - this.origin.getX();
      final double dy = ev.getY() - this.origin.getY();
      p.translate((int) dx, (int) dy);

      // Impedindo translao para fora dos limites inferiores do 
      // painel (pai) que possui os links
      if (p.x < 0) {
        p.x = 0;
      }
      if (p.y < 0) {
        p.y = 0;
      }

      // Impedindo translao para fora dos limites superiores do 
      // painel (pai) que possui os links
      final Container parent = draggedComponent.getParent();
      final int myWidth = draggedComponent.getWidth();
      final int parentWidth = parent.getWidth();
      if (p.x + myWidth > parentWidth) {
        p.x = parentWidth - myWidth;
      }
      final int myHeight = draggedComponent.getHeight();
      final int parentHeight = parent.getHeight();
      if (p.y + myHeight > parentHeight) {
        p.y = parentHeight - myHeight;
      }

      // Reajuste do layout para garantir o posicionamento livre.
      this.draggedComponent.setLocation(p);
      this.draggedComponent.getParent().setLayout(null);
    }

    /**
     * Evento de movimento de mouse (no tratado).
     * 
     * @param ev evento.
     */
    @Override
    public void mouseMoved(MouseEvent ev) {
    }

    /**
     * Construtor padro.
     * 
     * @param c o componente Java a ser mexido (cone).
     */
    ApplicationLinkMouseListener(JComponent c) {
      this.draggedComponent = c;
    }
  }

  /**
   * Nome do atributo de persistncia: posio X
   */
  private static final String POS_X_INFO = "POS_X_INFO";

  /**
   * Valor default para: posio X
   */
  private static final int DEFAULT_POS_X = 0;

  /**
   * Nome do atributo de persistncia: posio Y
   */
  private static final String POS_Y_INFO = "POS_Y_INFO";

  /**
   * Valor default para: posio Y
   */
  private static final int DEFAULT_POS_Y = 0;

  /**
   * Ao associada.
   */
  final private ApplicationStartAction action;

  /**
   * Consulta a imagem relativa ao link
   * 
   * @return a imagem
   */
  public ImageIcon getImageIcon() {
    return (ImageIcon) getIcon();
  }

  /**
   * Configura a posicao do link
   * 
   * @param appInfo o informativo de aplicao.
   */
  public void setInfo(Hashtable<String, Integer> appInfo) {
    int x = Utilities.getIntegerFromHash(appInfo, POS_X_INFO, DEFAULT_POS_X);
    int y = Utilities.getIntegerFromHash(appInfo, POS_Y_INFO, DEFAULT_POS_Y);
    this.setLocation(x, y);
  }

  /**
   * Obtem as configuracoes do link
   * 
   * @return uma hashtable com os os atributos das aplicaes.
   */
  public Hashtable<String, Integer> getInfo() {
    Hashtable<String, Integer> appInfo = new Hashtable<String, Integer>();
    appInfo.put(POS_X_INFO, getX());
    appInfo.put(POS_Y_INFO, getY());
    return appInfo;
  }

  /**
   * Execuo.
   */
  public void executeAction() {
    action.actionPerformed(new ActionEvent(this, 0, ""));
  }

  /**
   * Cria um componente que representa um link para uma aplicao.
   * 
   * @param action ao do aplicativo
   */
  public ApplicationLink(ApplicationStartAction action) {
    this.action = action;

    final Object acName = action.getValue(Action.NAME);
    final String actionName = (acName == null ? "???" : (String) acName);
    setText(actionName);

    final ApplicationRegistry registry = action.getApplicationRegistry();
    final byte[] acImg = registry.getImageDefinition();
    if (acImg != null) {
      setIcon(new ImageIcon(acImg));
    }
    setOpaque(false);
    setFont(new Font(Font.SANS_SERIF, Font.PLAIN, 12));
    setVerticalTextPosition(SwingConstants.BOTTOM);
    setHorizontalTextPosition(SwingConstants.CENTER);
    setBorder(NO_BORDER);

    addMouseListener(new MouseAdapter() {
      @Override
      public void mouseClicked(MouseEvent ev) {
        if (ev.getClickCount() == 2) {
          try {
            if (isEnabled()) {
              executeAction();
            }
          }
          catch (Exception ce) {
            StandardErrorDialogs.showErrorDialog(null, ce);
          }
        }
      }
    });
    setLocation(50, 50);
    setOpaque(false);
    setVisible(true);
    /*
     * Calcula o espao ocupado pela borda e adiciona ao tamanho do componente,
     * para quando ela estiver visvel no reduzir o tamanho do texto do
     * componente.
     */
    Dimension preferredSize = getPreferredSize();
    setPreferredSize(preferredSize);
    ApplicationLinkMouseListener listener =
      new ApplicationLinkMouseListener(this);
    this.addMouseListener(listener);
    this.addMouseMotionListener(listener);
  }
}
