/*
 * $Id$
 */

package csbase.client.applications.commandsmonitor.models;

import csbase.logic.CommandStatus;

/**
 * Informaes de um filtro de comandos.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class CommandsFilterDTO {

  /** Identificador nico do filtro. */
  private final String id;
  /**
   * Indica que o filtro deve considerar apenas ou desconsiderar todos os
   * comandos que estejam em um dos estados do atributo {@link #statuses},
   * dependendo se o valor deste atributo  <tt>true</tt> ou <tt>false</tt>
   * respectivamente.
   */
  private final boolean statuses_included;
  /**
   * Indica o estado dos comandos que sero (des)considerados, de acordo com o
   * valor de {@link #statuses_included}.
   */
  private final CommandStatus[] statuses;
  /**
   * Expresso que ir filtrar os comandos que passaram pela validao de
   * estado.
   */
  private final ExpressionDTO expressionInfo;

  /**
   * Construtor.
   * 
   * @param id Identificador nico do filtro.
   * @param statuses_included Indica se os comandos de interesse devem ou no
   *        devem estar em um dos estados do parmetro <i>statuses</i>.
   *        <tt>True</tt> ou <tt>false</tt>, respectivamente.
   * @param statuses Estado no qual os comandos de interesse devem ou na estar,
   *        de acordo com o valor do parmetro <i>statuses_included</i>.
   * @param expressionInfo Expresso que filtra os comandos que passaram pela
   *        validao de estado.
   */
  public CommandsFilterDTO(String id, boolean statuses_included,
    CommandStatus[] statuses, ExpressionDTO expressionInfo) {
    this.id = id;
    this.statuses_included = statuses_included;
    this.statuses = statuses;
    this.expressionInfo = expressionInfo;
  }

  /**
   * Obtm o identificador nico do filtro.
   * 
   * @return O identificador nico do filtro.
   */
  public String getId() {
    return id;
  }

  /**
   * Indica se os comandos de interesse devem ou no devem estar em um dos
   * estados de {@link #getStatuses()}. <tt>True</tt> ou <tt>false</tt>,
   * respectivamente.
   * 
   * @return <tt>true</tt> se os comandos de interesse devem estar em um dos
   *         estados de {@link #getStatuses()}, ou <tt>false</tt> caso eles
   *         no possam estar em nenhum destes estados.
   */
  public boolean areStatusesIncluded() {
    return statuses_included;
  }

  /**
   * Obtm a lista de estados no qual os comandos de interesse do filtro devem
   * ou no estar, de acordo com o valor de {@link #areStatusesIncluded()}.
   * 
   * @return A lista de estados no qual os comandos de interesse do filtro devem
   *         ou no estar, de acordo com o valor de
   *         {@link #areStatusesIncluded()}.
   */
  public CommandStatus[] getStatuses() {
    return statuses;
  }

  /**
   * Obtm a expresso que filtra os comandos que passaram pela validao de
   * estado.
   * 
   * @return A expresso que filtra os comandos que passaram pela validao de
   *         estado.
   */
  public ExpressionDTO getExpressionInfo() {
    return expressionInfo;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((id == null) ? 0 : id.hashCode());
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    final CommandsFilterDTO other = (CommandsFilterDTO) obj;
    if (id == null) {
      if (other.id != null)
        return false;
    }
    else if (!id.equals(other.id))
      return false;
    return true;
  }
}
