/*
 * $Id$
 */

package csbase.client.applications.filetransferclient.panels.remotepanel;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridBagLayout;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.JToolBar;
import javax.swing.ListSelectionModel;
import javax.swing.SortOrder;
import javax.swing.SwingConstants;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.filetransferclient.FileTransferClient;
import csbase.client.applications.filetransferclient.FileTransferClientUI;
import csbase.client.applications.filetransferclient.FileTransferProxy;
import csbase.client.applications.filetransferclient.actions.ChangeRemoteDirectoryAction;
import csbase.client.applications.filetransferclient.actions.HomeRemoteAction;
import csbase.client.applications.filetransferclient.actions.RefreshRemoteAction;
import csbase.client.applications.filetransferclient.actions.RootRemoteAction;
import csbase.client.applications.filetransferclient.actions.UpRemoteDirectoryAction;
import csbase.client.applications.filetransferclient.panels.core.FileTransferClientPanel;
import csbase.client.applications.filetransferclient.panels.localpanel.FileTransferClientLocalPanel;
import csbase.client.applications.filetransferclient.panels.queuepanel.FileTransferClientQueuePanel;
import csbase.logic.ClientProjectFile;
import csbase.logic.filetransferservice.FileTransferConnection;
import csbase.logic.filetransferservice.FileTransferElement;
import csbase.logic.filetransferservice.FileTransferProtocol;

/**
 * Painel de arquivos diretrios remotos.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class FileTransferClientRemotePanel extends FileTransferClientPanel {

  /**
   * Path do diretrio root remoto
   */
  final public static String ROOT_REMOTE_PATH = "/";

  /**
   * Path do diretrio remoto.
   */
  private String remoteDirectory;

  /**
   * Tabela de arquivos remotos.
   */
  private final SortableTable remoteTable = new SortableTable();

  /**
   * Texto com diretrio remoto.
   */
  private final JTextField remoteDirText = new JTextField();

  /**
   * Texto com nome do servidor remoto.
   */
  final private JTextField serverText = new JTextField(30);

  /**
   * Texto com usurio remoto.
   */
  final private JTextField userText = new JTextField(30);

  /**
   * Filtragem de arquivos comeados por ".".
   */
  private boolean dotFiltered = true;

  /**
   * Ao de home
   */
  final private HomeRemoteAction homeAction;
  
  /**
   * Toolbar.
   */
  final private JToolBar toolBar = new JToolBar(SwingConstants.HORIZONTAL);

  /**
   * Consulta se painel est filtrando elementos comeados por "."
   * 
   * @return indicativo
   */
  public boolean isDotFiltered() {
    return dotFiltered;
  }

  /**
   * Indica se painel deve filtrar elementos comeados por "."
   * 
   * @param dotFiltered indicativo
   */
  public void setDotFiltered(boolean dotFiltered) {
    this.dotFiltered = dotFiltered;
    refresh();
  }

  /**
   * @return o painel
   */
  private JPanel buildInfoPanel() {

    remoteDirText.setEditable(false);
    userText.setEditable(false);
    serverText.setEditable(false);

    final JLabel remoteDirLabel = new JLabel();
    remoteDirLabel.setText(getString("remote.directory.label"));

    final JLabel userLabel = new JLabel();
    userLabel.setText(getString("remote.user.label"));

    final JLabel serverLabel = new JLabel();
    serverLabel.setText(getString("remote.server.label"));

    final JPanel infoPanel = new JPanel();
    final JComponent[][] cmps =
      new JComponent[][] { { userLabel, userText },
          { serverLabel, serverText }, { remoteDirLabel, remoteDirText }, };
    GUIUtils.mountBasicGridPanel(infoPanel, cmps);

    final String title = getString("frame");
    final JLabel remoteLabel = new JLabel();
    remoteLabel.setIcon(FileTransferClientUI.REMOTE_ICON);
    remoteLabel.setText(title);
    
    final JPanel panel = new JPanel();
    panel.setLayout(new GridBagLayout());
    panel.add(remoteLabel, new GBC(0,0).none().west().insets(0, 0, 20, 0));
    panel.add(infoPanel, new GBC(0,1).both());
    return panel;
  }

  /**
   * Montagem da toolbar.
   */
  private void mountToolBar() {
    final FileTransferClient application = getApplication();
    toolBar.setFloatable(false);

    toolBar.add(homeAction);
    toolBar.addSeparator();
    toolBar.add(new RootRemoteAction(application));
    toolBar.add(new ChangeRemoteDirectoryAction(application));
    toolBar.add(new UpRemoteDirectoryAction(application));
    toolBar.add(new JPanel());
    toolBar.add(new RefreshRemoteAction(application));
  }

  /**
   * 
   */
  private void doubleClickedOnTable() {
    final List<FileTransferElement> sels = getSelectedElements();
    if (sels.size() != 1) {
      return;
    }
    final FileTransferElement selected = sels.get(0);
    if (selected == null || selected.isLink()) {
      return;
    }

    final String elementName = selected.getName();
    if (selected.isDirectory()) {
      final String remoteDir = getRemoteDirectory();
      final String newDir = remoteDir + "/" + elementName;
      setRemoteDirectory(newDir);
    }
    else {
      final FileTransferClient application = getApplication();

      final String remoteDir = getRemoteDirectory();
      final String remotePath = remoteDir + "/" + elementName;

      final ApplicationFrame frame = application.getApplicationFrame();
      final String title = getString("download.confirm.title");
      final String fmt = getString("download.confirm.message");
      final String question = String.format(fmt, remotePath);
      if (StandardDialogs.showYesNoDialog(frame, title, question) != 0) {
        return;
      }

      final FileTransferClientLocalPanel localPanel =
        application.getLocalPanel();
      final ClientProjectFile localDirectory = localPanel.getLocalDirectory();
      final String localDir = FileTransferClientUI.pathToString(localDirectory);
      final String localPath;
      if (FileTransferClientUI.isEmptyString(localDir)) {
        localPath = elementName;
      }
      else {
        localPath = localDir + "/" + elementName;
      }

      String lPath = localPath;
      while (lPath.startsWith("/")) {
        lPath = lPath.substring(1);
      }
      final String[] localFilePath = lPath.split("/");

      final FileTransferProxy serviceProxy = application.getServiceProxy();
      final FileTransferConnection connection = application.getConnection();
      final long size = selected.getSizeBytes();
      serviceProxy.download(connection, remotePath, localFilePath, size);
      localPanel.refresh();

      final FileTransferClientQueuePanel queuePanel =
        application.getQueuePanel();
      queuePanel.refresh();
    }

  }

  /**
   * Consulta ao path do diretrio remoto.
   * 
   * @return o diretrio
   */
  final public String getRemoteDirectory() {
    return remoteDirectory;
  }

  /**
   * Consulta o item (arquivo ou diretrio) selecionado.
   * 
   * @return o item selecionado
   */
  final public List<FileTransferElement> getSelectedElements() {
    final int[] sels = remoteTable.getSelectedRows();
    if (sels == null || sels.length == 0) {
      return null;
    }
    final RemoteTableModel model = (RemoteTableModel) remoteTable.getModel();
    final List<FileTransferElement> modelList = model.getFileList();

    final List<FileTransferElement> elems =
      new ArrayList<FileTransferElement>();
    for (final int sel : sels) {
      final int modelSel = remoteTable.convertRowIndexToModel(sel);
      final FileTransferElement file = modelList.get(modelSel);
      elems.add(file);
    }
    Collections.unmodifiableList(elems);
    return elems;
  }

  /**
   * Verifica se o diretrio  raiz remota.
   * 
   * @param remoteDir o diretrio
   * @return indicativo
   */
  final public boolean isRootDirectory(final String remoteDir) {
    if (remoteDir == null) {
      final String err = "remoteDir cannot be null";
      throw new IllegalArgumentException(err);
    }
    return remoteDir.trim().equals(ROOT_REMOTE_PATH);
  }

  /**
   * Montagem do painel
   */
  private void mountPanel() {
    setLayout(new BorderLayout());

    final JPanel infoPanel = buildInfoPanel();
    mountToolBar();

    final JScrollPane scrollTable = new JScrollPane(remoteTable);
    final String[] cNames =
      new String[] { getString("fd.column"), getString("name.column"),
          getString("size.column") };
    final RemoteTableModel model = new RemoteTableModel(cNames);
    remoteTable.setModel(model);

    remoteTable.setComparator(0, new RemoteTypeComparator());
    remoteTable.setNoSortStateEnabled(true);

    remoteTable
      .setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    remoteTable.sort(0, SortOrder.ASCENDING);

    final RemoteTableRenderer renderer = new RemoteTableRenderer();
    remoteTable.setDefaultRenderer(FileTransferElement.class, renderer);
    remoteTable.setDefaultRenderer(Long.class, renderer);
    remoteTable.setDefaultRenderer(String.class, renderer);

    remoteTable.addMouseListener(new MouseAdapter() {
      @Override
      public void mouseClicked(final MouseEvent e) {
        if (e.getClickCount() == 2) {
          doubleClickedOnTable();
        }
      }
    });

    final TableColumnModel colModel = remoteTable.getColumnModel();
    remoteTable.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
    final TableColumn c0 = colModel.getColumn(0);
    final TableColumn c1 = colModel.getColumn(1);
    final TableColumn c2 = colModel.getColumn(2);
    c0.setPreferredWidth(10);
    c1.setPreferredWidth(150);
    c2.setPreferredWidth(40);

    add(infoPanel, BorderLayout.NORTH);
    add(scrollTable, BorderLayout.CENTER);
    add(toolBar, BorderLayout.SOUTH);

    FileTransferClientUI.setEtchedBorder(this);
  }

  /**
   * Atualizao do painel
   */
  final public void refresh() {
    final FileTransferClient aaplication = getApplication();
    final FileTransferConnection connection = aaplication.getConnection();
    final RemoteTableModel model = (RemoteTableModel) remoteTable.getModel();

    if (connection == null) {
      model.setFileList(null);
      return;
    }

    final FileTransferProxy serviceProxy = aaplication.getServiceProxy();
    final List<FileTransferElement> files =
      serviceProxy.listContent(connection, remoteDirectory);

    if (files == null) {
      model.setFileList(null);
      return;
    }

    final List<FileTransferElement> filteredFiles =
      new ArrayList<FileTransferElement>();
    for (FileTransferElement file : files) {
      if (filterAllows(file)) {
        filteredFiles.add(file);
      }
    }
    model.setFileList(filteredFiles);
  }

  /**
   * Indica se arquivo deve aparecer na lista com base nos filtros existentes.
   * 
   * @param element arquivo
   * @return indicativo
   */
  private boolean filterAllows(FileTransferElement element) {
    final String name = element.getName();
    if (dotFiltered) {
      if (name.startsWith(".")) {
        return false;
      }
    }
    return true;
  }

  /**
   * Ajuste de conexo
   * 
   * @param connection conexo
   */
  final public void setConnection(final FileTransferConnection connection) {
    if (connection == null) {
      serverText.setText(null);
      userText.setText(null);
      setToolBarEnabled(false);
      return;
    }

    final FileTransferProtocol protocol = connection.getProtocol();
    final String serverName = connection.getServerName();
    final String userName = connection.getUserName();


    serverText.setText(serverName + " (" + protocol + ")");
    userText.setText(userName);
    setToolBarEnabled(true);

    // Ao final a ao de home fica condicionada a existir
    // um homePath para a conexo.
    final String homePath = connection.getHomePath();
    homeAction.setEnabled(homePath != null);
  }

  /**
   * (Des)Habilita a toolbar
   * 
   * @param enabled indicativo
   */
  private void setToolBarEnabled(final boolean enabled) {
    final Component[] cmps = toolBar.getComponents();
    for (Component cmp : cmps) {
      cmp.setEnabled(enabled);
    }
  }

  /**
   * Ajuste do diretrio remoto.
   * 
   * @param remoteDirectory diretrio remoto
   */
  final public void setRemoteDirectory(final String remoteDirectory) {
    if (remoteDirectory == null) {
      this.remoteDirectory = ROOT_REMOTE_PATH;
    }
    else {
      String nd = remoteDirectory;
      while (nd.startsWith("//")) {
        nd = nd.substring(1);
      }
      this.remoteDirectory = nd;
    }

    remoteDirText.setText(this.remoteDirectory);
    refresh();
  }

  /**
   * Construtor
   * 
   * @param application aplicao.
   */
  public FileTransferClientRemotePanel(final FileTransferClient application) {
    super(application);
    homeAction = new HomeRemoteAction(application);
    mountPanel();
    setRemoteDirectory(null);
  }

}
