/*
 * $Id$
 */

package csbase.client.facilities.configurabletable.column;

import java.util.Comparator;

import javax.swing.SwingConstants;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

import csbase.client.facilities.configurabletable.stringprovider.IStringProvider;
import csbase.client.facilities.configurabletable.table.ConfigurableTable;

/**
 * Coluna abstrata que deve ser estendida por cada coluna da tabela configurvel
 * {@link ConfigurableTable}.
 * 
 * @param <T> - tipo do objeto que prov os valores da coluna.
 * 
 * @author Tecgraf
 */
public abstract class AbstractConfigurableColumn<T> implements
  IConfigurableColumn<T> {

  /**
   * Identificador nico da coluna.
   */
  private String id;

  /**
   * Propriedade que define se a coluna  visvel ou no.
   */
  private boolean visible;

  /**
   * Usado para internacionalizar os nomes das colunas.
   */
  private IStringProvider stringProvider;

  /**
   * Alinhamento dos valores da tabela.
   */
  private int align;

  /**
   * String que sinaliza a ausncia de um valor.
   */
  public static String NO_VALUE = "---";

  /**
   * Construtor default.
   * 
   * @param id - identificador nico da coluna.
   * @param visible - visibilidade da coluna.
   * @param stringProvider - necessrio para internacionalizar o nome da coluna.
   * @param align - alinhamento da coluna.
   */
  protected AbstractConfigurableColumn(String id, Boolean visible,
    IStringProvider stringProvider, Integer align) {
    if (id == null) {
      throw new IllegalArgumentException("id no pode ser nulo.");
    }

    this.id = id;
    this.visible = (visible == null) ? false : visible;
    this.stringProvider = stringProvider;

    this.align = (align == null) ? SwingConstants.CENTER : align;

    if (stringProvider == null) {
      this.stringProvider = new IStringProvider() {
        @Override
        public String getString(String key) {
          return key;
        }
      };
    }
  }

  /**
   * Obtm uma string de idioma.
   * 
   * @param key - chave da string a ser internacionalizada.
   * @return string internacionalizada.
   */
  protected String getString(String key) {
    return stringProvider.getString(key);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getId() {
    return id;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isVisible() {
    return visible;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getAlign() {
    return align;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isEditable(T obj) {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValue(T obj, Object value) {
    // o default  no deixar modificar a celula.
  };

  /**
   * {@inheritDoc}
   */
  @Override
  public void setVisible(boolean visible) {
    this.visible = visible;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Comparator<?> getComparator() {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TableCellRenderer createTableCellRenderer() {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TableCellEditor createTableCellEditor() {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder();
    builder.append("id:");
    builder.append(getId());
    builder.append(";visible:");
    builder.append(isVisible());
    builder.append(";align:");
    builder.append(getAlign());
    builder.append(";string-provider:");
    builder.append(stringProvider.toString());

    return builder.toString();
  }
}
