package csbase.client.algorithms.parameters;

import java.util.LinkedList;
import java.util.List;

import csbase.client.algorithms.validation.ViewValidator;
import csbase.logic.algorithms.parameters.Parameter;

/**
 * Viso de Parmetro.
 * 
 * @param <T> Tipo de parmetro que esta viso representa.
 * 
 * @author lmoreira
 */
public abstract class ParameterView<T extends Parameter<?>> implements
  ViewValidator {

  /**
   * Modo de visualizao da viso.
   */
  public enum Mode {
    /**
     * Modo de configurao. Neste modo os valores dos parmetros podem ser
     * alterados.
     */
    CONFIGURATION,
    /**
     * Modo de relatrio. Utilizado para visualizar os valors dos parmetros sem
     * que estes possam ser alterados.
     */
    REPORT
  }

  /**
   * Os observadores.
   */
  private List<ParameterViewListener> listeners;

  /**
   * O parmetro.
   */
  private T parameter;

  /**
   * Modo de visualizao da viso do parmetro.
   */
  private final Mode mode;

  /**
   * Cria a viso.
   * 
   * @param parameter O parmetro (No aceita {@code null}).
   * @param mode Modo de visualizao. No aceita {@code null}, os possveis
   *        valores so: {@link ParameterView.Mode#CONFIGURATION} ou
   *        {@link ParameterView.Mode#REPORT}
   */
  public ParameterView(T parameter, Mode mode) {
    this.mode = mode;
    this.listeners = new LinkedList<ParameterViewListener>();
    setParameter(parameter);
  }

  /**
   * Adiciona um observador a esta viso.
   * 
   * @param listener O observador (No aceita {@code null}).
   */
  public final void addParameterViewListener(ParameterViewListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.listeners.add(listener);
  }

  /**
   * Indica se a viso est visvel.
   * 
   * @return .
   */
  protected abstract boolean isVisible();

  /**
   * Exibe/Oculta a viso.
   * 
   * @param isVisible .
   */
  protected abstract void setVisible(boolean isVisible);

  /**
   * Atualiza a viso em relao ao estado Em Exibio/Oculto.
   */
  protected void updateVisibilyView() {
    if (isVisible() != getParameter().isVisible()) {
      setVisible(getParameter().isVisible());
      fireVisibilityWasChanged();
    }
  }

  /**
   * Dispara o evento de mudana da visibilidade da viso do parmetro.
   */
  protected final void fireVisibilityWasChanged() {
    for (ParameterViewListener listener : this.listeners) {
      listener.visibilityWasChanged(this);
    }
  }

  /**
   * Indica se a viso do parmetro deve ser amplivel verticalmente. As vises
   * que implementam controles grandes e/ou com scroll podem indicar com esse
   * mtodo que querem ser redimensionadas vericalemente em conjunto com a
   * janela do configurador, aproveitando melhor o espao.
   * 
   * @return Verdadeiro se essa viso pode ser ampliada verticalmente ou falso,
   *         caso contrrio.
   */
  public abstract boolean fillVerticalSpace();

  /**
   * {@inheritDoc}
   */
  @Override
  public final String toString() {
    return getParameter().toString();
  }

  /**
   * Obtm o parmetro.
   * 
   * @return .
   */
  public T getParameter() {
    return this.parameter;
  }

  /**
   * Atribui o parmetro a esta viso.
   * 
   * @param parameter O parmetro (No aceita {@code null}).
   */
  private void setParameter(T parameter) {
    if (parameter == null) {
      throw new IllegalArgumentException("O parmetro parameter est nulo.");
    }
    this.parameter = parameter;
  }

  /**
   * Consulta o modo
   * 
   * @return o modo
   */
  public Mode getMode() {
    return mode;
  }
}
