/*
 * $Id$
 */

package csbase.client.applications.algorithmsmanager.versiontree.actions;

import java.io.File;
import java.net.URL;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.List;

import javax.swing.JFileChooser;

import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.algorithmsmanager.versiontree.VersionTree;
import csbase.client.desktop.Task;
import csbase.client.externalresources.ExternalResources;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.FileInfo;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.remote.ClientRemoteLocator;

/**
 * @author Tecgraf / PUC-Rio
 * 
 *         Ao de importao de executveis.
 */
public class ImportExecutableFile extends AbstractImportFileAction {

  /** Verso que ir receber os executveis. */
  private AlgorithmVersionInfo version;

  /** Plataforma que ir receber os executveis. */
  private String platformName;

  /** Caminho de destino na Plataforma que conter os executveis. */
  private String destPlataformPath;

  /**
   * Construtor.
   * 
   * @param tree Fonte da ao.
   * @param version verso de destino dos arquivos importados.
   * @param platformName plataforma de destino dos arquivos importados.
   */
  public ImportExecutableFile(VersionTree tree, AlgorithmVersionInfo version,
    String platformName) {

    super(tree, LNG.get("algomanager.menu.bin_file.add"),
      JFileChooser.FILES_AND_DIRECTORIES, true);

    this.version = version;
    this.platformName = platformName;
  }

  /**
   * Construtor.
   *
   * @param tree Fonte da ao.
   * @param version verso de destino dos arquivos importados.
   * @param platformName plataforma de destino dos arquivos importados.
   * @param destPlataformPath caminho de destino na Plataforma que conter os
   *        executveis.
   */
  public ImportExecutableFile(VersionTree tree, AlgorithmVersionInfo version,
    String platformName, String destPlataformPath) {
    this(tree, version, platformName);

    this.destPlataformPath = destPlataformPath;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Task<Boolean> createCheckFileExistenceTask(final String fileName) {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected RemoteFileChannelInfo prepareUpload(String fileName,
    boolean isZipLocalFile) throws RemoteException {
    AlgorithmInfo algorithm = version.getInfo();
    /*
     * Se o arquivo for do tipo ZipLocalFile, devemos informar que ele deve ser
     * descompactado assim que for carregado no servidor.
     */
    return ClientRemoteLocator.algorithmService.prepareUploadExecFile(algorithm
      .getId(), version.getId(), platformName, getFileNameDestPath(fileName),
      isZipLocalFile);
  }

  /**
   * Retorna o caminho relativo do arquivo a ser importado
   *
   * @param fileName nome do arquivo a ser importado
   * @return caminho relativo do arquivo a ser importado
   */
  private String getFileNameDestPath(String fileName) {
    if (destPlataformPath == null) {
      return fileName;
    }
    return destPlataformPath + File.separator + fileName;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void appletUpload() {
    new UploadExecutableDialog().show();
  }

  /**
   * Classe que representa um dilogo de carregamento de executveis no servidor
   * a partir do cliente.
   * 
   * @author Leonardo Barros
   */
  private class UploadExecutableDialog {
    /**
     * Ttulo do dilogo.
     */
    private String dialogTitle;

    /**
     * Cria o dilogo.
     */
    public UploadExecutableDialog() {
      List<FileInfo> execs = version.getExecutables(platformName);
      if (execs == null) {
        return;
      }
      Collections.sort(execs);
      dialogTitle =
        MessageFormat.format(LNG.get("algomanager.title.executable"),
          new Object[] { version.getInfo().getName(), version.getId(),
              platformName });
    }

    /**
     * Chama o browser para upload dos executveis. O upload s est disponivel
     * quando os recursos externos esto habilitados.
     */
    public void show() {
      if (ExternalResources.getInstance().isEnabled()) {
        String addr =
          AlgorithmManagementProxy.getExecutableURL(version.getInfo().getId(),
            version.getId(), platformName, getWindow());
        if (addr == null) {
          return;
        }
        try {
          final URL url = new URL(addr);
          ExternalResources.getInstance().showDocument(url);
        }
        catch (Exception ex) {
          StandardErrorDialogs.showErrorDialog(getWindow(), dialogTitle,
            LNG.get("PRJ_PROJECT_FILE_UPLOAD_IO_ERROR"), ex);
        }
      }
      else {
        StandardDialogs.showInfoDialog(getWindow(), dialogTitle,
          LNG.get("algomanager.error.upload_not_available"));
      }
    }
  }
}
