package csbase.client.preferences;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.util.LinkedList;
import java.util.List;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.preferences.annotations.Editor;
import csbase.client.preferences.annotations.Hidden;
import csbase.client.preferences.annotations.ReadWrite;
import csbase.client.preferences.annotations.Value;
import csbase.client.preferences.definition.PreferenceDefinition;
import csbase.client.preferences.definition.PreferencePolicy;
import csbase.client.preferences.util.PreferenceBundle;
import csbase.client.preferences.util.PreferencesUtil;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Classe encarregada de fazer a carga de todas as enumeraes que definem as
 * preferncias do sistema. So elas: <br/>
 * - Preferncias de sistema (por exemplo, desktop); <br/>
 * - Preferncias de aplicaes.
 * 
 * @see PreferenceCategory PreferenceValue PreferenceManager
 * 
 * @author Tecgraf
 */
class DefinitionLoader {

  /** Referncia para a instncia nica desta classe */
  private static DefinitionLoader instance;

  /** Objeto usado na internacionalizao a partir de bundles globais. */
  private static PreferenceBundle generalBundle;

  /**
   * Retorna a instncia nica desta classe.
   * 
   * @return instncia nica.
   */
  static DefinitionLoader getInstance() {
    if (instance == null) {
      instance = new DefinitionLoader();
    }
    return instance;
  }

  /**
   * Retorna o objeto usado na internacionalizao a partir de bundles globais.
   * 
   * @return objeto usado na internacionalizao.
   */
  PreferenceBundle getGeneralBundle() {
    if (generalBundle == null) {
      generalBundle = new PreferenceBundle() {
        @Override
        public String get(String key) {
          return LNG.get(key);
        }

        @Override
        public boolean has(String key) {
          return LNG.hasKey(key);
        }
      };
    }
    return generalBundle;
  }

  /**
   * Carrega as definies de preferncias de todas as aplicaes e adiciona na
   * raz de preferncias.
   * 
   * @param root raz das preferncias.
   */
  void loadAppDefinitions(PreferenceCategory root) {
    ApplicationManager am = ApplicationManager.getInstance();

    for (ApplicationRegistry registry : am.getAllApplicationRegistries()) {

      Class<? extends PreferenceDefinition> enumClass = loadAppEnum(registry);
      if (enumClass != null) {
        PreferenceBundle appBundle = createPreferenceBundle(registry);
        buildPreferenceNode(enumClass, root, appBundle);
      }
    }
  }

  /**
   * Carrega uma definio de preferncia e adiciona na raz de preferncias.
   * 
   * @param root raz da rvore de preferncias.
   * @param enumClass enumerao que define preferncias.
   */
  void loadDefinition(PreferenceCategory root,
    Class<? extends PreferenceDefinition> enumClass) {
    buildPreferenceNode(enumClass, root, getGeneralBundle());
  }

  /**
   * Faz a carga de uma enumerao que define as preferncias de uma aplicao.
   * 
   * @param registry registro da aplicao.
   * @return classe da enumerao que define as preferncias da aplicao, null
   *         caso a aplicao no tenha definido suas preferncias.
   */
  @SuppressWarnings("unchecked")
  Class<? extends PreferenceDefinition> loadAppEnum(ApplicationRegistry registry) {
    String enumName = PreferencesUtil.getAppEnumName(registry);

    try {
      Class<? extends PreferenceDefinition> clazz =
        (Class<? extends PreferenceDefinition>) Class.forName(enumName);
      validateEnum(clazz);
      return clazz;
    }
    catch (ClassNotFoundException e) {
      return null;
    }
  }

  /**
   * Preenche o objeto {@link PreferenceCategory} com as preferncias definidas
   * na enumerao.
   * 
   * @param enumClass enumerao que define as preferncias.
   * @param parent objeto que representa as preferncias do usurio.
   * @param bundle objeto usado para internacionalizao.
   */
  private void buildPreferenceNode(
    Class<? extends PreferenceDefinition> enumClass, PreferenceCategory parent,
    PreferenceBundle bundle) {

    PreferenceCategory node =
      parent.createCategory(enumClass.getName(), bundle);

    for (Object enumConstant : enumClass.getEnumConstants()) {
      Field field = loadField(enumConstant.toString(), enumClass);

      Value valueAnnot = field.getAnnotation(Value.class);
      validateAnnotation(valueAnnot, field, enumClass);

      // criando valor de preferncia.
      PreferencePolicy policy = getPolicy(field);

      PreferenceValue<?> value =
        createPreferenceValue(valueAnnot, (PreferenceDefinition) enumConstant,
          policy, bundle);

      // criando editor de preferncia.
      if (field.isAnnotationPresent(Editor.class)) {
        definePreferenceEditor(field.getAnnotation(Editor.class), value);
      }

      node.addPreference(enumConstant.toString(), value);
    }

    // criando ns internos recursivamente.
    for (Class<? extends PreferenceDefinition> internalEnum : getInternalEnums(enumClass)) {
      validateEnum(internalEnum);
      buildPreferenceNode(internalEnum, node, bundle);
    }
  }

  /**
   * Faz a carga de uma constante da enumerao.
   * 
   * @param fieldName nome do constante.
   * @param enumClass classe que encapsula a enumerao.
   * @return objeto que encapsula a constante da enumerao.
   */
  private Field loadField(String fieldName, Class<?> enumClass) {
    try {
      Field field = enumClass.getField(fieldName);
      return field;
    }
    catch (Exception e) {
      throw new PreferenceException("No foi possvel carregar a constante "
        + fieldName + " da enumerao " + enumClass.getName(), e);
    }
  }

  /**
   * Retorna uma lista contendo as enumeraes declaradas internamente e
   * externamente.
   * 
   * @param enumClass enumerao base.
   * @return lista com as enumeraes internas.
   */
  @SuppressWarnings("unchecked")
  private List<Class<? extends PreferenceDefinition>> getInternalEnums(
    Class<? extends PreferenceDefinition> enumClass) {

    List<Class<? extends PreferenceDefinition>> result =
      new LinkedList<Class<? extends PreferenceDefinition>>();

    for (Class<?> internalClass : enumClass.getDeclaredClasses()) {
      validateEnum(internalClass);
      result.add((Class<? extends PreferenceDefinition>) internalClass);
    }

    for (Field field : enumClass.getDeclaredFields()) {
      if (field.getAnnotations().length == 0 && !field.isSynthetic()
        && !field.getClass().equals(enumClass)) {

        Class<?> internalClass = field.getType();
        validateEnum(internalClass);
        result.add((Class<? extends PreferenceDefinition>) internalClass);

      }
    }

    return result;
  }

  /**
   * Retorna a poltica de visibilidade de uma constante que define uma
   * preferncia. Se no for definida nenhuma poltica de visibilidade assumimos
   * 'apenas leitura'.
   * 
   * @param field constante que representa a preferncia.
   * @return poltica de visibilidade.
   */
  private PreferencePolicy getPolicy(Field field) {
    PreferencePolicy policy = PreferencePolicy.READ_ONLY;
    if (field.isAnnotationPresent(ReadWrite.class)) {
      policy = PreferencePolicy.READ_WRITE;
    }
    if (field.isAnnotationPresent(Hidden.class)) {
      policy = PreferencePolicy.HIDDEN;
    }
    return policy;
  }

  /**
   * Cria o objeto que encapsula o valor de uma preferncia.
   * 
   * @param valueAnnot anotao que define o valor default de uma preferncia e
   *        a classe que a encapsula.
   * @param name nome da preferncia.
   * @param policy poltica de visibilidade da preferncia.
   * @param preferenceBundle objeto responsvel pela internacionalizao.
   * @return objeto que encapsula o valor de uma preferncia.
   */
  private PreferenceValue<?> createPreferenceValue(Value valueAnnot,
    PreferenceDefinition name, PreferencePolicy policy,
    PreferenceBundle preferenceBundle) {
    try {
      Constructor<?> construtor =
        valueAnnot.type().getConstructor(PreferenceDefinition.class,
          String.class, String.class, PreferencePolicy.class,
          PreferenceBundle.class);

      return (PreferenceValue<?>) construtor.newInstance(name,
        valueAnnot.defaultValue(), valueAnnot.defaultValue(), policy,
        preferenceBundle);
    }
    catch (Exception e) {
      throw new PreferenceException(
        "Erro ao criar o objeto que encapsula o valor da preferncia " + name,
        e);
    }
  }

  /**
   * Cria o objeto que encapsula o editor de uma preferncia.
   * 
   * @param <T> tipo do valor da preferncia.
   * 
   * @param editorAnnot anotao que define o editor de uma preferncia.
   * @param value objeto que encapsula o valor de uma preferncia.
   */
  @SuppressWarnings("unchecked")
  private <T> void definePreferenceEditor(Editor editorAnnot,
    PreferenceValue<T> value) {
    Class<? extends PreferenceEditor<T>> editorClass =
      (Class<? extends PreferenceEditor<T>>) editorAnnot.value();
    value.setPreferenceEditorClass(editorClass);
  }

  /**
   * Mtodo que valida uma enumerao que encapsula preferncias. Esta
   * enumerao deve definir os seguintes quesitos: <br/>
   * - Ser uma enumerao <br/>
   * - Implementar a interface {@link PreferenceDefinition} .
   * 
   * @param enumClass enumerao a ser validada.
   */
  private void validateEnum(Class<?> enumClass) {
    if (enumClass == null) {
      throw new IllegalArgumentException("Enumerao no pode ser nula.");
    }
    if (!enumClass.isEnum()) {
      throw new PreferenceException(enumClass.getName()
        + " deveria ser uma enumerao.");
    }

    boolean implement = false;
    for (Class<?> impl : enumClass.getInterfaces()) {
      if (impl.equals(PreferenceDefinition.class)) {
        implement = true;
        break;
      }
    }

    if (!implement) {
      throw new PreferenceException("Enumerao " + enumClass.getName()
        + " deve implementar a interface "
        + PreferenceDefinition.class.getName());
    }
  }

  /**
   * Mtodo que valida a anotao {@link Value}. Est anotao deve definir o
   * valor default e o tipo do valor.
   * 
   * @param valueAnnot anotao que define o valor da preferncia.
   * @param field campo anotado.
   * @param enumClass enumerao que contm o campo anotado.
   */
  private void validateAnnotation(Value valueAnnot, Field field,
    Class<? extends PreferenceDefinition> enumClass) {

    StringBuilder builder = new StringBuilder();
    builder.append("Anotao: ");
    builder.append(Value.class.getName());
    builder.append("; Constante: ");
    builder.append(field.getName());
    builder.append("; Enumerao: ");
    builder.append(enumClass.getName());

    String context = builder.toString();

    if (valueAnnot == null) {
      throw new PreferenceException("Anotao no definida - " + context);
    }
    if (valueAnnot.type() == null) {
      throw new PreferenceException("Tipo no definido - " + context);
    }
    if (valueAnnot.defaultValue() == null) {
      throw new PreferenceException("Valor default no definido - " + context);
    }
  }

  /**
   * Cria objeto usado na internacionalizao de preferncias de aplicao.
   * 
   * @param registry registro da aplicao.
   * @return objeto usado na internacionalizao.
   */
  private PreferenceBundle createPreferenceBundle(
    final ApplicationRegistry registry) {
    return new PreferenceBundle() {
      @Override
      public String get(String key) {

        String appPrefName =
          PreferencesUtil.getAppEnumName(registry) + ".label";

        if (appPrefName.endsWith(key)) {
          return ApplicationManager.getInstance().getApplicationName(registry);
        }
        return registry.getString(key);
      }

      @Override
      public boolean has(String key) {
        return registry.hasString(key);
      }
    };
  }

  /** Construtor padro. */
  private DefinitionLoader() {
  }
}
