/**
 * $Id$
 */
package csbase.client.project;

import java.rmi.RemoteException;
import java.util.Collection;
import java.util.LinkedList;

import csbase.logic.ProjectBasicInfo;
import csbase.logic.SharedObject;
import csbase.logic.User;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.SharedObjectServiceInterface;

/**
 * Gerencia a persistncia da lista de projetos abertos recentemente pelo
 * usurio. Os ltimos projetos so armazenados em uma fila, que  persistida
 * como um SharedObject.
 *
 * @author Tecgraf
 */
public class RecentProjectsManager {

  /**
   * Nmero mximo de projetos armazendos na fila
   */
  private static final int MAX_PROJECTS = 11;

  /**
   * Nome do objeto que armazena os projetos recentes como shared Object
   */
  private static final String RECENT_PROJECTS = "recent_projects";

  /**
   * Nome da categoria do shared object
   */
  private static final String RECENT_PROJECTS_CATEGORY = "RecentProjects";

  /**
   * Armazena a lista de projetos abertos recentemente.
   *
   * @return o {@link SharedObject} que armazena a lista de dados de projetos
   * abertos recentemente pelo usurio
   * @throws RemoteException em caso de falha na comunicao com o servidor.
   */
  private SharedObject readSharedObject() throws RemoteException {
    SharedObjectServiceInterface sharedObjectService =
      ClientRemoteLocator.sharedObjectService;
    try {
      SharedObject sharedObject = sharedObjectService
        .getSharedObject(RECENT_PROJECTS_CATEGORY, User.getLoggedUser().getId(),
          RECENT_PROJECTS);
      return sharedObject;
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Obtm as informaes de projetos recentes abertos a partir do
   * {@link SharedObject}.
   *
   * @param so Shared Object usado para persistir os dados de projetos
   * @return Coleo com Dados de projetos abertos recentemente. Os ltimos
   * utilizados esto nas primeiras posies.
   */
  @SuppressWarnings("unchecked")
  private Collection<ProjectBasicInfo> getProjectsInfosFromSharedObject(
    SharedObject so) {
    LinkedList<ProjectBasicInfo> projectsHistoryInfos = new LinkedList<>();
    if (so == null || so.getContents() == null) {
      return projectsHistoryInfos;
    }
    projectsHistoryInfos.addAll((Collection<ProjectBasicInfo>) so.getContents());
    return projectsHistoryInfos;
  }

  /**
   * Obtm a coleo de projetos abertos recentemente. Os ltimos utilizados
   * esto nas primeiras posies.
   *
   * @return A coleo de dados de projeto armazenados.
   * @throws RemoteException em caso de erro na leitra dos dados.
   */
  public Collection<ProjectBasicInfo> getProjectsInfosFromHistory() throws
    RemoteException {
    SharedObject so = readSharedObject();
    return getProjectsInfosFromSharedObject(so);
  }

  /**
   * Salva dados de projeto para que possam ser recuperados futuramente.
   *
   * @param projectBasicInfo Dados bsicos do projeto
   * @throws RemoteException em caso de falha na comunicao com o servidor.
   */
  public void saveProjectInfo(
    final ProjectBasicInfo projectBasicInfo) throws RemoteException {

    if (projectBasicInfo == null) {
      return;
    }

    SharedObject sharedObj = readSharedObject();
    LinkedList<ProjectBasicInfo> projectsHistoryInfos =
      (LinkedList<ProjectBasicInfo>) getProjectsInfosFromSharedObject(sharedObj);

    if (projectsHistoryInfos.contains(projectBasicInfo)) {
      projectsHistoryInfos.remove(projectBasicInfo);
    }
    /* Se j estiver com o nmero limite, exclui o ltimo */
    if (projectsHistoryInfos.size() == MAX_PROJECTS) {
      projectsHistoryInfos.removeLast();
    }
    /* Inclui o item atual no incio da fila */
    projectsHistoryInfos.addFirst(projectBasicInfo);

    if (sharedObj == null) {
      sharedObj = createSharedObject(projectsHistoryInfos);
    }
    else {
      sharedObj.setContents(projectsHistoryInfos);
    }

    SharedObjectServiceInterface sharedObjectService =
      ClientRemoteLocator.sharedObjectService;
    sharedObjectService.saveSharedObject(sharedObj);
  }

  /**
   * Cria um shared object com o contedo indicado
   *
   * @param contents o contedo
   * @return um novo shared object com o contedo indicado
   */
  private SharedObject createSharedObject(Object contents) {
    SharedObject obj =
      new SharedObject(RECENT_PROJECTS_CATEGORY, User.getLoggedUser().getId(),
        RECENT_PROJECTS, true, contents);
    return obj;
  }

}
