package csbase.client.applications.algorithmsmanager.versiontree.actions;

import java.net.URL;
import java.rmi.RemoteException;

import javax.swing.JFileChooser;

import csbase.client.applications.algorithmsmanager.dialogs.VersionDialog;
import csbase.client.applications.algorithmsmanager.versiontree.VersionTree;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.externalresources.ExternalResources;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.remote.ClientRemoteLocator;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;

/**
 * Ao de importao de arquivo de release notes.
 */
public class ImportReleaseNotesFile extends AbstractImportFileAction {

  /**
   * Verso que ir receber o release notes.
   */
  private final AlgorithmVersionInfo version;

  /**
   * Construtor.
   * 
   * @param tree Fonte da ao.
   * @param version verso de destino dos arquivos importados.
   */
  public ImportReleaseNotesFile(VersionTree tree,
    AlgorithmVersionInfo version) {
    super(tree, LNG.get("algomanager.menu.release.notes.add"),
      JFileChooser.FILES_ONLY, false);
    this.version = version;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void appletUpload() {
    new UploadReleaseNotesDialog().uploadFile();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Task<Boolean> createCheckFileExistenceTask(String fileName) {
    return new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        AlgorithmInfo algorithm = version.getInfo();
        setResult(ClientRemoteLocator.algorithmService.releaseNotesFileExists(
          algorithm.getId(), version.getId(), fileName));
      }
    };
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected RemoteFileChannelInfo prepareUpload(String fileName,
    boolean isZipLocalFile) throws RemoteException {
    AlgorithmInfo algorithm = version.getInfo();
    /*
     * Se o arquivo for do tipo ZipLocalFile, devemos informar que ele deve ser
     * descompactado assim que for carregado no servidor.
     */
    return ClientRemoteLocator.algorithmService.prepareUploadReleaseNotesFile(
      algorithm.getId(), version.getId(), fileName, isZipLocalFile);
  }

  /**
   * Classe que representa um dilogo para carregamento de release notes no
   * servidor a partir do cliente.
   */
  private class UploadReleaseNotesDialog extends VersionDialog {
    /**
     * Identificador da verso que ir receber o arquivo texto de release notes
     */
    private Object versionId;

    /**
     * Cria o dilogo.
     */
    public UploadReleaseNotesDialog() {
      super(version.getInfo());
    }

    /**
     * Chama o browser para upload do arquivo texto de release notes. O upload
     * s est disponvel quando os recursos externos esto habilitados.
     */
    public void uploadFile() {
      Object algoId = algoInfo.getId();
      if (ExternalResources.getInstance().isEnabled()) {
        String addr = AlgorithmManagementProxy.retrieveReleaseNotesUploadURL(
          algoId, versionId, getWindow());
        if (addr == null) {
          return;
        }
        try {
          final URL url = new URL(addr);
          ExternalResources.getInstance().showDocument(url);
        }
        catch (Exception ex) {
          StandardErrorDialogs.showErrorDialog(getWindow(), dialogTitle, LNG
            .get("PRJ_PROJECT_FILE_UPLOAD_IO_ERROR"), ex);
        }
      }
      else {
        StandardDialogs.showInfoDialog(getWindow(), dialogTitle, LNG.get(
          "algomanager.error.upload_not_available"));
      }
    }
  }

}
