/*
 * $Id$
 */

package csbase.client.desktop;

import java.awt.BorderLayout;
import java.awt.Font;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Observable;
import java.util.Observer;
import java.util.concurrent.atomic.AtomicLong;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JToolBar;

import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.applications.ApplicationImages;
import csbase.client.remote.srvproxies.NotificationProxy;
import csbase.client.remote.srvproxies.messageservice.MessageProxy;
import csbase.client.util.DateTranslator;
import csbase.client.util.NotificationTranslator;
import csbase.exception.ConfigurationException;
import csbase.logic.CommandNotification;
import csbase.logic.Notification;
import csbase.logic.NotificationHandler;
import csbase.logic.User;
import csbase.remote.RemoteEvent;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;

/**
 * Classe base para um painel simplificado de cliente de notificaes vindas de
 * um servidor de sistema integrador. Este painel implementa um polling no
 * servio definido pela interface <code>NotificationServiceInterface</code>.
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class NotificationPanel implements Observer {

  /**
   * Dilogo de composio de notificaes
   */
  protected NotificationCompositionFrame composeDialog = null;

  /**
   * rea de notificaes
   */
  protected JTextArea notificationArea = null;

  /**
   * Referncia ao painel principal
   */
  protected JPanel mainPanel = null;

  /**
   * Id do usurio.
   */
  protected Object userId = null;

  /**
   * Lista de handlers que sero chamados quando ocorre uma notificao.
   */
  protected LinkedList<NotificationHandler> notificationHandlers = null;

  /**
   * Configurao da classe.
   */
  final private Configuration configuration;

  /**
   * Mapa contendo o total de comandos terminados de acordo tendo como chave o
   * tipo de finalizao do comando.
   */
  private Map<Class<?>, AtomicLong> finishedCmdsCount =
    new HashMap<Class<?>, AtomicLong>();

  /**
   * Adio de linha de notificao.
   * 
   * @param line a linha a ser colocada.
   */
  public void addNotificationLine(final String line) {
    notificationArea.append(line);
    notificationArea.setCaretPosition(notificationArea.getText().length());
  }

  /**
   * Adio de vrias linhas (lista) de notificao.
   * 
   * @param lineList a linha a ser colocada.
   */
  final protected void writeNotificationsLines(final List<String> lineList) {
    for (Iterator<String> lineIterator = lineList.iterator(); lineIterator
      .hasNext();) {
      this.addNotificationLine(lineIterator.next());
    }
  }

  /**
   * Abertura do dilogo de composio de notificaes.
   */
  final protected void openComposeDialog() {
    if (this.composeDialog == null) {
      this.composeDialog = new NotificationCompositionFrame();
    }
    this.composeDialog.start();
  }

  /**
   * Construo do painel de notificao.
   * 
   * @param composeIcon cone a ser atribudo ao boto de compor mensagem
   * @param clearIcon cone a ser atribudo ao boto de limpar mensagem
   * @return o <code>JPanel</code>
   */
  final protected JToolBar createButtonsPanel(final ImageIcon composeIcon,
    final ImageIcon clearIcon) {

    // Boto de envio de mensagem
    final JButton notButton = new JButton(composeIcon);
    notButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        openComposeDialog();
      }
    });
    notButton.setToolTipText(LNG.get("notification.send.button.tooltip"));

    // Boto para apagar a rea de mensagens
    final JButton clrButton = new JButton(clearIcon);
    clrButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        clearPanel();
      }
    });
    clrButton.setToolTipText(LNG.get("notification.clear.button.tooltip"));

    final JToolBar toolbar = new JToolBar(JToolBar.VERTICAL);
    toolbar.setFloatable(false);
    toolbar.add(notButton);
    toolbar.add(clrButton);
    return toolbar;
  }

  /**
   * Limpeza da rea de notificaes
   */
  final protected void clearPanel() {
    notificationArea.setText("");
  }

  /**
   * Construo do painel de notificao.
   * 
   * @param composeIcon cone do boto para criar uma mensagem
   * @param clearIcon cone do boto para apagar a rea de notificaes
   * @return o <code>JPanel</code>
   */
  final protected JPanel createMainPanel(final ImageIcon composeIcon,
    final ImageIcon clearIcon) {
    notificationArea = new JTextArea();
    notificationArea.setEditable(false);
    notificationArea.setFont(new Font(Font.MONOSPACED, Font.PLAIN, 12));
    clearPanel();
    final JScrollPane scrolledPane = new JScrollPane(notificationArea);
    JPanel xPanel = new JPanel();
    xPanel.setLayout(new BorderLayout());
    xPanel.add(scrolledPane, BorderLayout.CENTER);
    xPanel.add(createButtonsPanel(composeIcon, clearIcon), BorderLayout.WEST);
    return xPanel;
  }

  /**
   * Consulta ao painel.
   * 
   * @return o <code>JPanel</code>
   */
  final public JPanel getPanel() {
    return mainPanel;
  }

  /**
   * Recebimento de mensagens de observao.
   * 
   * @param observable a fonte.
   * @param obj a informao.
   */
  @Override
  final public void update(Observable observable, Object obj) {
    if (obj == null) {
      return;
    }
    if (obj instanceof Notification) {
      final ArrayList<Notification> notifs = new ArrayList<Notification>();
      notifs.add((Notification) obj);
      sendMessages(notifs);
    }
    else if (obj instanceof RemoteEvent[]) {
      RemoteEvent[] events = (RemoteEvent[]) obj;
      final ArrayList<Notification> notifs = new ArrayList<Notification>();
      for (int i = 0; i < events.length; i++) {
        if (events[i] instanceof Notification) {
          notifs.add((Notification) events[i]);
        }
      }
      if (notifs.size() > 0) {
        sendMessages(notifs);
      }
    }
  }

  /**
   * Adiciona um handler para ser chamado quando a notificao chegar.
   * 
   * @param handler o mtodo que ser chamado.
   */
  public void addNotificationHandler(NotificationHandler handler) {
    synchronized (notificationHandlers) {
      this.notificationHandlers.add(handler);
    }
  }

  /**
   * Chama os handlers que tiverem interesse nessa notificao.
   * 
   * @param data os dados da notificao.
   */
  private void callNotificationHandlers(final Notification data) {
    synchronized (notificationHandlers) {
      for (NotificationHandler handler : notificationHandlers) {
        handler.gotNotification(data);
      }
    }
  }

  /**
   * Envio de mensagens.
   * 
   * @param notifications o vetor de notificaes.
   */
  private void sendMessages(final ArrayList<Notification> notifications) {
    if (notifications == null || notifications.size() == 0) {
      return;
    }

    /*
     * Indica que pelo menos uma mensagem foi inserida no painel de notificao.
     * Neste caso o sistema deve emitir um som chamando a ateno do usurio.
     */
    boolean msgDisplayed = false;

    final NotificationTranslator translator = getTranslator();

    if (translator == null) {
      SwingThreadDispatcher.invokeLater(new Runnable() {
        @Override
        public void run() {
          final Date now = new Date();
          final String msg = LNG.get("notification.translator.null.error");
          addNotificationLine(makeNotificationLine(now, "!!!", msg, false));
        }
      });
      msgDisplayed = true;
    }
    else {
      for (Notification notification : notifications) {
        if (User.getLoggedUser().isAdmin()
          && CommandNotification.class
            .isAssignableFrom(notification.getClass())) {

          synchronized (finishedCmdsCount) {
            AtomicLong count = finishedCmdsCount.get(notification.getClass());
            if (count == null) {
              count = new AtomicLong(1);
              finishedCmdsCount.put(notification.getClass(), count);
            }
            else {
              count.incrementAndGet();
            }
          }
          final StringBuilder infoMsg =
            new StringBuilder("[FIM DE COMANDO] CmdId[")
              .append(((CommandNotification) notification).getCommandId())
              .append("] Evento[")
              .append(notification.getClass().getSimpleName())
              .append("]. Total: ");
          for (Entry<Class<?>, AtomicLong> entry : finishedCmdsCount.entrySet()) {
            infoMsg.append(" [").append(entry.getKey().getSimpleName())
              .append(":").append(entry.getValue().get()).append("]");
          }
          final Date date = new Date(notification.getCreationDate());
          final String sender = notification.getSender();
          SwingThreadDispatcher.invokeLater(new Runnable() {
            @Override
            public void run() {
              String line =
                makeNotificationLine(date, sender, infoMsg.toString(), true);
              addNotificationLine(line);
            }
          });
        }

        Object result = null;
        try {
          // Usando reflexividade para chamar o mtodo certo
          final Method method =
            translator.getClass().getMethod("translate",
              new Class[] { notification.getClass() });
          result = method.invoke(translator, new Object[] { notification });
        }
        catch (Exception e) {
          result =
            LNG.get("notification.translation.error") + " - "
              + notification.toString();
        }
        final Date date = new Date(notification.getCreationDate());
        final String sender = notification.getSender();
        final String msg = result.toString();
        final boolean mustPopUp = notification.getMustPopUp();
        final boolean vol = notification.isVolatile();
        if (notification.getDisplay()) {
          SwingThreadDispatcher.invokeLater(new Runnable() {
            @Override
            public void run() {
              String line = makeNotificationLine(date, sender, msg, vol);
              addNotificationLine(line);
              if (mustPopUp) {
                showNotificationPopUp(sender, date, msg);
              }
            }
          });
          msgDisplayed = true;
        }
        // Essa chamada tem que ficar fora da thread do Swing
        //TODO O NotificationPanel repassa as notificaes para os handlers. Modificar para usar notificao.
        callNotificationHandlers(notification);
      }
    }

    if (msgDisplayed) {
      final String tag = "notification.beep";
      final boolean beepActivated =
        configuration.getOptionalBooleanProperty(tag, true);
      if (beepActivated) {
        final Toolkit toolkit = Toolkit.getDefaultToolkit();
        toolkit.beep();
      }
    }
  }

  /**
   * Exibe dilogo modal com a notificao recebida.
   * 
   * @param senderLogin o remetente.
   * @param date a data da mensagem.
   * @param text o texo.
   */
  private void showNotificationPopUp(final String senderLogin, final Date date,
    final String text) {
    NotificationReplyFrame replyFrame =
      new NotificationReplyFrame(senderLogin, date, text);
    replyFrame.start();
  }

  /**
   * Monta uma linha de notificao.
   * 
   * @param date a data.
   * @param senderName um texto do remetente.
   * @param line texto da mensagem.
   * @param volatileFlag flag indicativo de mensagem voltil.
   * @return retorna a linha de notificao montada
   */
  public String makeNotificationLine(Date date, String senderName,
    String line, boolean volatileFlag) {
    String prefix = volatileFlag ? "[-] " : "[+] ";
    String formDate = (DateTranslator.getInstance()).translate(date);
    return (prefix + formDate + " - " + "[" + senderName + "]: "
      + line.replace('\n', ' ') + "\n");
  }

  /**
   * Altera o usurio corrente
   * 
   * @param user o objeto que representa o usurio.
   */
  final public void setUser(final User user) {
    Object uid = null;
    if (user != null) {
      uid = user.getId();
    }
    else {
      if (composeDialog != null) {
        composeDialog.close();
      }
    }
    if (userId != null) {
      NotificationProxy.deleteUserObserver(this, userId);
    }
    if (uid != null) {
      NotificationProxy.addUserObserver(this, uid);
    }
    userId = uid;
  }

  /**
   * Mtodo de busca de um tradutor de notificaes (objetos)
   * 
   * @return um tradutor.
   */
  public abstract NotificationTranslator getTranslator();

  /**
   * Construtor de escolha da cones de mensagens.
   * 
   * @param composeIcon cone a ser atribudo ao boto de compor mensagem
   * @param clearIcon cone a ser atribudo ao boto de limpar mensagem
   */
  public NotificationPanel(final ImageIcon composeIcon,
    final ImageIcon clearIcon) {
    this.notificationHandlers = new LinkedList<NotificationHandler>();
    this.mainPanel = createMainPanel(composeIcon, clearIcon);
    try {
      final ConfigurationManager manager = ConfigurationManager.getInstance();
      configuration = manager.getConfiguration(this.getClass());
    }
    catch (ConfigurationManagerException e) {
      throw new ConfigurationException(e);
    }

    MessageProxy.addListener(new IMessageListener() {
      @Override
      public void onMessagesReceived(Message... messages) throws Exception {
        for (Message aMessage : messages) {
          ArrayList<Notification> notifs = new ArrayList<Notification>();
          notifs.add((Notification) aMessage.getBody());
          sendMessages(notifs);
        }
      }
    }, Notification.class);
  }

  /**
   * Construtor padro (sem escolha da cones).
   */
  public NotificationPanel() {
    this(ApplicationImages.ICON_COMPOSEMSG_24,
      ApplicationImages.ICON_CLEARMSG_24);
  }
}
