package csbase.client.applications.commandsmonitor;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.List;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applications.commandsmonitor.events.DoubleClickEvent;
import csbase.client.applications.commandsmonitor.table.CommandsTableFactory;
import csbase.client.desktop.DesktopFrame;
import csbase.client.facilities.commandtable.AbstractCommandMonitoringTable;
import csbase.client.project.ProjectTreeAdapter;
import csbase.client.util.event.EventManager;
import csbase.client.util.gui.JMultiSplitPane;
import csbase.logic.CommonClientProject;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Painel contendo tabelas de monitorao de comandos.
 * 
 * @author Tecgraf
 */
class TablesPanel extends JScrollPane {

  /**
   * Indica se a monitorao dos comandos est ativa.
   */
  private boolean running = false;

  /**
   * Gerenciador de seleo das tabelas deste painel.
   */
  protected UniqueTableSelectionManager selectionManager;
  /**
   * Lista das tabelas contidas nesta aba.
   */
  private AbstractCommandMonitoringTable[] tables;

  /**
   * Construtor.
   * 
   * @param tablesId Identificador das tabelas na ordem em que elas devem
   *        aparecer neste painel.
   * @param tablesFactory Fbrica de tabelas.
   * @param appRegistry Registro da aplicao utilizado para obter a fraseologia
   *        a ser utilizada.
   * @param eventManager gerencia os eventos gerados.
   * 
   * @throws ApplicationException Erro na criao da tabela.
   */
  public TablesPanel(List<String> tablesId, CommandsTableFactory tablesFactory,
    ApplicationRegistry appRegistry, final EventManager eventManager)
    throws ApplicationException {

    if (tablesId.size() == 0) {
      throw new IllegalArgumentException(
        "tablesId deve conter pelo menos um elemento.");
    }

    this.selectionManager = new UniqueTableSelectionManager(eventManager);
    this.setViewportView(createView(tablesId, tablesFactory, appRegistry,
      eventManager));
  }

  /**
   * @param tablesId
   * @param tablesFactory
   * @param appRegistry
   * @param eventManager
   * @return
   * @throws ApplicationException
   */
  private JComponent createView(List<String> tablesId,
    CommandsTableFactory tablesFactory, ApplicationRegistry appRegistry,
    final EventManager eventManager) throws ApplicationException {

    this.setPreferredSize(new Dimension(850, 500));

    tables = new AbstractCommandMonitoringTable[tablesId.size()];
    JPanel[] innerPanels = new JPanel[tablesId.size()];
    for (int inx = 0; inx < tablesId.size(); inx++) {
      String aTableId = tablesId.get(inx);

      Title title = new Title(aTableId, appRegistry);
      tables[inx] = tablesFactory.createTable(aTableId);
      tables[inx].setPreferredScrollableViewportSize(new Dimension(800, 100));
      /*
       * Adiciona um listener que notifica o eventManager quando ocorrer um
       * duplo clique na tabela.
       */
      final AbstractCommandMonitoringTable currTable = tables[inx];
      currTable.addMouseListener(new MouseAdapter() {
        @Override
        public void mouseClicked(MouseEvent e) {
          if (e.getClickCount() == 2) {
            eventManager.fireEvent(new DoubleClickEvent(currTable
              .getSelectedCommands()[0]));
          }
        }
      });
      /*
       * Permite que o selectionManager gerencie o sistema de seleo de linhas
       * desta tabela.
       */
      selectionManager.startManaging(tables[inx]);

      innerPanels[inx] = new JPanel(new GridBagLayout());

      GridBagConstraints constraints = new GridBagConstraints();
      constraints.insets = new Insets(5, 5, 0, 5);
      constraints.gridx = 0;
      constraints.gridy = 0;
      constraints.gridwidth = 2;
      constraints.fill = GridBagConstraints.BOTH;
      constraints.anchor = GridBagConstraints.WEST;
      innerPanels[inx].add(title, constraints);

      constraints = new GridBagConstraints();
      constraints.insets = new Insets(5, 5, 5, 5);
      constraints.gridx = 0;
      constraints.gridy = 1;
      constraints.gridwidth = 2;
      constraints.weightx = 1.0;
      constraints.weighty = 1.0;
      constraints.fill = GridBagConstraints.BOTH;
      constraints.anchor = GridBagConstraints.WEST;

      innerPanels[inx].add(new JScrollPane(tables[inx]), constraints);
    }

    if (innerPanels.length == 1) {
      return innerPanels[0];
    }

    final int split = JMultiSplitPane.VERTICAL_SPLIT;
    JMultiSplitPane pane = new JMultiSplitPane(split, false, innerPanels);
    pane.setOneTouchExpandable(true);
    return pane;
  }

  /**
   * Inicia a monitorao das tabelas de comandos nesta aba.
   */
  public void start() {
    if (!running) {
      for (AbstractCommandMonitoringTable aTable : tables) {
        aTable.start();
      }
    }
    running = true;
  }

  /**
   * Para a monitorao das tabelas de comandos nesta aba.
   */
  public void stop() {
    for (AbstractCommandMonitoringTable aTable : tables) {
      aTable.stop();
    }
    running = false;
  }

  class Title extends JLabel {
    /**
     * Marcador que se includo no ttulo de uma tabela, ser substitudo pelo
     * nome do projeto corrente e atualizado sempre que esse mudar.
     */
    private static final String TITLE_FLAG_PROJECT_NAME = "${project.name}";
    /**
     * Expresso regular utilizada para substituir o
     * {@link #TITLE_FLAG_PROJECT_NAME} no ttulo de uma tabela, pelo nome do
     * projeto corrente.
     */
    private static final String TITLE_FLAG_PROJECT_NAME_REGEX =
      "\\$\\{project\\.name\\}";

    /**
     * Chave utilizada para obter o ttulo da tabela atravs do mtodo
     * {@link ApplicationRegistry#getString(String)}.
     */
    private static final String TITLE_KEY = "table.%s.title";
    /**
     * Chave utilizada para obter o ttulo da tabela o ttulo original faz
     * referncia ao projeto corrente, e no h projeto aberto.
     */
    private static final String NULL_PROJECT_TITLE_KEY =
      "table.%s.title.project.none";

    public Title(final String tableId, final ApplicationRegistry appRegistry) {
      super();

      final String title =
        appRegistry.getString(String.format(TITLE_KEY, tableId));

      // Verifica se o ttulo tem um marcador aonde deve ir o nome do projeto.
      if (title.contains(TITLE_FLAG_PROJECT_NAME)) {
        /*
         * Se tiver, cria o ttulo substituindo o marcador pelo nome do projeto
         * ou utiliza um ttulo alternativo caso no haja projeto aberto.
         */

        // Obtm um ttulo alternativo para quando no houver projeto aberto.
        final String nullProjectTitle;
        String nullProjectKey = String.format(NULL_PROJECT_TITLE_KEY, tableId);
        if (appRegistry.getResourceBundle().containsKey(nullProjectKey)) {
          nullProjectTitle = appRegistry.getString(nullProjectKey);
        }
        else {
          nullProjectTitle =
            title.replaceAll(TITLE_FLAG_PROJECT_NAME_REGEX, "");
        }

        // Atribui o texto do ttulo de acordo com a existncia ou no de projeto. 
        CommonClientProject project = DesktopFrame.getInstance().getProject();
        setText(project == null ? nullProjectTitle : title.replaceAll(
          TITLE_FLAG_PROJECT_NAME_REGEX, project.getName()));

        /*
         * Cria um ouvinte da rvore de projeto para atualizar o ttulo sempre
         * que o projeto mudar ou for fechado.
         */
        DesktopFrame.getInstance().getTree()
          .addProjectTreeListener(new ProjectTreeAdapter() {
            @Override
            public void projectChanged(CommonClientProject project) {
              setText(project == null ? nullProjectTitle : title.replaceAll(
                TITLE_FLAG_PROJECT_NAME_REGEX, project.getName()));
            }
          });
      }
      else {
        setText(title);
      }
    }
  }
}
